package one.veriph.sdk.ui.verification.screen

import androidx.compose.foundation.clickable
import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.ExperimentalLayoutApi
import androidx.compose.foundation.layout.IntrinsicSize
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.Spacer
import androidx.compose.foundation.layout.WindowInsets
import androidx.compose.foundation.layout.fillMaxSize
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.height
import androidx.compose.foundation.layout.navigationBarsIgnoringVisibility
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.layout.windowInsetsBottomHeight
import androidx.compose.foundation.rememberScrollState
import androidx.compose.foundation.verticalScroll
import androidx.compose.material3.ExperimentalMaterial3Api
import androidx.compose.material3.Icon
import androidx.compose.material3.MaterialTheme
import androidx.compose.material3.ModalBottomSheet
import androidx.compose.material3.Text
import androidx.compose.material3.TextButton
import androidx.compose.material3.rememberModalBottomSheetState
import androidx.compose.runtime.Composable
import androidx.compose.runtime.collectAsState
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.rememberCoroutineScope
import androidx.compose.runtime.saveable.rememberSaveable
import androidx.compose.runtime.setValue
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.res.painterResource
import androidx.compose.ui.res.stringResource
import androidx.compose.ui.text.style.TextAlign
import androidx.compose.ui.unit.dp
import kotlinx.coroutines.launch
import one.veriph.sdk.R
import one.veriph.sdk.data.VerificationMethod
import one.veriph.sdk.ui.verification.VerificationViewModel
import one.veriph.sdk.ui.verification.composable.InverseOTPAttempt
import one.veriph.sdk.ui.verification.composable.LoadingIndicator
import one.veriph.sdk.ui.verification.composable.TraditionalOTPAttempt

@Composable
internal fun AttemptExecutionScreen(
    viewModel: VerificationViewModel,
    onExpiration: () -> Unit,
    onEditNumber: () -> Unit,
    onCancellation: (reason: Int) -> Unit,
) {
    val attemptState = viewModel.attempt.collectAsState()

    if (attemptState.value.state?.attemptCreationResponse == null) {
        Column(
            modifier = Modifier
                .fillMaxSize(),
            verticalArrangement = Arrangement.Center,
            horizontalAlignment = Alignment.CenterHorizontally,
        ) {
            LoadingIndicator(R.string.attempt_execution_loading)
        }
    } else {
        val attempt: one.veriph.sdk.data.AttemptCreationResponse =
            attemptState.value.state?.attemptCreationResponse!!

        val attemptIsInverse = when (attempt.type) {
            VerificationMethod.INVERSE_OTP_SMS, VerificationMethod.INVERSE_OTP_WHATSAPP -> true
            else -> false
        }
        val sessionClosure = viewModel.sessionClosed.collectAsState()
        if (attemptIsInverse
            && !sessionClosure.value.isLoading
            && sessionClosure.value.state != null
            && !sessionClosure.value.state!!
        ) {
            viewModel.subscribeToStatus()
        }

        AttemptExecutor(
            viewModel,
            attempt,
            onExpiration,
            onEditNumber,
            onCancellation
        )
    }
}

@OptIn(ExperimentalMaterial3Api::class, ExperimentalLayoutApi::class)
@Composable
private fun AttemptExecutor(
    viewModel: VerificationViewModel,
    attempt: one.veriph.sdk.data.AttemptCreationResponse,
    onExpiration: () -> Unit,
    onEditNumber: () -> Unit,
    onCancellation: (reason: Int) -> Unit,
) {
    val helpCenterState = rememberModalBottomSheetState()
    val scope = rememberCoroutineScope()
    var helpCenterOpened by remember { mutableStateOf(false) }
    var cancellationRequested by rememberSaveable { mutableStateOf(false) }

    Column(
        modifier = Modifier
            .fillMaxSize(),
        verticalArrangement = Arrangement.Center,
        horizontalAlignment = Alignment.CenterHorizontally,
    ) {
        Column(
            modifier = Modifier
                .verticalScroll(rememberScrollState())
                .height(IntrinsicSize.Max)
                .padding(32.dp),
            verticalArrangement = Arrangement.Center,
            horizontalAlignment = Alignment.CenterHorizontally,
        ) {
            if (attempt.userPhoneNumberHint !== null)
                PhoneNumberHint(
                    attempt.userPhoneNumberHint,
                    attempt.userCanEditNumber,
                    onEditNumber,
                    Modifier
                        .padding(bottom = 24.dp)
                )

            when (attempt.type) {
                VerificationMethod.INVERSE_OTP_SMS,
                VerificationMethod.INVERSE_OTP_WHATSAPP -> {
                    InverseOTPAttempt(
                        attempt = attempt,
                        onExpiration = {
                            onExpiration()
                        }
                    )
                }

                VerificationMethod.OTP_SMS,
                VerificationMethod.OTP_WHATSAPP,
                VerificationMethod.ROBOCALL -> {
                    TraditionalOTPAttempt(
                        viewModel = viewModel,
                        attempt = attempt,
                        onExpiration = {
                            onExpiration()
                        }
                    )
                }
            }

            TextButton(
                onClick = { helpCenterOpened = true },
                modifier = Modifier.padding(top = 24.dp)
            ) {
                Text(stringResource(id = R.string.attempt_execution_btn_help))
            }
        }
    }

    if (helpCenterOpened) {
        ModalBottomSheet(
            onDismissRequest = {
                helpCenterOpened = false
            },
            sheetState = helpCenterState,
        ) {
            val reasons = one.veriph.sdk.data.getCancellationReasonsForMethod(attempt.type)
            reasons.forEach { item ->
                HelpCenterRow(item) {
                    scope.launch { helpCenterState.hide() }.invokeOnCompletion {
                        if (!helpCenterState.isVisible) {
                            helpCenterOpened = false
                        }
                    }
                    if (!cancellationRequested) {
                        cancellationRequested = true
                        onCancellation(item.value)
                    }
                }
            }
            Spacer(
                Modifier
                    .windowInsetsBottomHeight(
                        WindowInsets.navigationBarsIgnoringVisibility
                    )
                    .padding(bottom = 12.dp)
            )
        }
    }
}

@Composable
private fun HelpCenterRow(item: one.veriph.sdk.data.CancellationReason, onClick: () -> Unit) {
    Row(
        verticalAlignment = Alignment.CenterVertically,
        modifier = Modifier
            .fillMaxWidth()
            .clickable { onClick() }
            .padding(horizontal = 32.dp, vertical = 12.dp)
    ) {
        Icon(
            painter = painterResource(item.iconRes),
            contentDescription = stringResource(id = item.labelRes),
            tint = MaterialTheme.colorScheme.onSurface
        )
        Text(
            stringResource(id = item.labelRes),
            modifier = Modifier.padding(start = 32.dp)
        )
    }
}

@Composable
private fun PhoneNumberHint(
    numberHint: String,
    canEdit: Boolean,
    onEditNumber: () -> Unit,
    modifier: Modifier = Modifier
) {
    var str = stringResource(id = R.string.attempt_execution_hint, numberHint)
    if (canEdit)
        str += " ${stringResource(id = R.string.attempt_execution_hint_action)}"

    TextButton(
        enabled = canEdit,
        onClick = { onEditNumber() },
        modifier = modifier
    ) {
        Text(str, textAlign = TextAlign.Center)
    }
}