package one.veriph.sdk.ui.verification.screen

import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.fillMaxSize
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.text.KeyboardOptions
import androidx.compose.material.icons.Icons
import androidx.compose.material.icons.outlined.Phone
import androidx.compose.material3.Button
import androidx.compose.material3.Icon
import androidx.compose.material3.MaterialTheme
import androidx.compose.material3.OutlinedTextField
import androidx.compose.material3.Text
import androidx.compose.runtime.Composable
import androidx.compose.runtime.collectAsState
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.saveable.rememberSaveable
import androidx.compose.runtime.setValue
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.res.stringResource
import androidx.compose.ui.text.input.KeyboardType
import androidx.compose.ui.text.style.TextAlign
import androidx.compose.ui.unit.dp
import androidx.compose.ui.unit.sp
import one.veriph.sdk.R
import one.veriph.sdk.ui.util.getDefaultCountry
import one.veriph.sdk.ui.util.onlyContainsNumbers
import one.veriph.sdk.ui.verification.VerificationViewModel
import one.veriph.sdk.ui.verification.composable.CountryCodeSelector
import one.veriph.sdk.ui.verification.composable.LoadingIndicator

@Composable
internal fun PhoneFormScreen(
    viewModel: VerificationViewModel,
    onSubmit: () -> Unit
) {
    val countryCodes = viewModel.countryCodes.collectAsState()
    val attempt = viewModel.attempt.collectAsState()

    val countryCodeUuid: String? =
        attempt.value.state?.attemptCreationResponse?.helperData?.countryCodeUuid
    val cellphoneNumber: String? =
        attempt.value.state?.attemptCreationResponse?.helperData?.phoneNumber

    Column(
        modifier = Modifier
            .fillMaxSize()
            .padding(32.dp),
        verticalArrangement = Arrangement.Center,
        horizontalAlignment = Alignment.CenterHorizontally,
    ) {
        if (countryCodes.value.state == null) {
            if (!countryCodes.value.isLoading)
                viewModel.getCountryCodes()
            LoadingIndicator(R.string.form_phone_loading)
        } else {
            Text(
                stringResource(id = R.string.form_phone_title),
                modifier = Modifier.padding(bottom = 12.dp),
                textAlign = TextAlign.Center,
                color = MaterialTheme.colorScheme.secondary,
                fontSize = 24.sp,
            )
            Text(
                stringResource(id = R.string.form_phone_description),
                modifier = Modifier.padding(bottom = 24.dp),
                textAlign = TextAlign.Center,
            )
            PhoneForm(
                viewModel,
                countryCodes.value.state!!,
                countryCodeUuid,
                cellphoneNumber,
                onSubmit
            )
        }
    }
}

@Composable
private fun PhoneForm(
    viewModel: VerificationViewModel,
    countryCodes: List<one.veriph.sdk.data.CountryCode>,
    countryCodeUuid: String?,
    cellphoneNumber: String?,
    onSubmit: () -> Unit
) {
    var number by rememberSaveable { mutableStateOf(cellphoneNumber ?: "") }
    val defaultCc = countryCodes[0]
    val defaultCcUuid = getDefaultCountry(
        countryCodes,
        countryCodeUuid,
        defaultCc
    ).uuid
    var selCountryCodeUuid by rememberSaveable { mutableStateOf(defaultCcUuid) }
    var phoneError by rememberSaveable { mutableStateOf(false) }

    CountryCodeSelector(
        countryCodes = countryCodes,
        selectedCountryCodeUuid = defaultCcUuid,
        modifier = Modifier
            .padding(bottom = 12.dp)
            .fillMaxWidth(),
        defaultCC = defaultCc,
        onSelected = {
            selCountryCodeUuid = it.uuid
        }
    )

    OutlinedTextField(
        value = number,
        singleLine = true,
        leadingIcon = {
            Icon(
                Icons.Outlined.Phone,
                stringResource(id = R.string.form_phone_hint_number),
                tint = MaterialTheme.colorScheme.onSurface
            )
        },
        keyboardOptions = KeyboardOptions(keyboardType = KeyboardType.Number),
        modifier = Modifier
            .fillMaxWidth()
            .padding(bottom = 24.dp),
        onValueChange = {
            if (it.isEmpty() || onlyContainsNumbers(it) || it.length <= 20) {
                number = it
            }

            phoneError = number.length <= 3
        },
        label = { Text(stringResource(id = R.string.form_phone_hint_number)) },
        isError = phoneError
    )
    Button(
        onClick = {
            if (number.length > 3) {
                viewModel.setSessionPhoneNumber(selCountryCodeUuid, number)
                viewModel.invalidateAttempt()
                onSubmit()
            } else {
                phoneError = true
            }
        }) {
        Text(stringResource(id = R.string.v1_sdk_generic_next))
    }
}
