package one.veriph.sdk.network

import android.text.TextUtils
import dagger.Module
import dagger.Provides
import okhttp3.Interceptor
import okhttp3.OkHttpClient
import okhttp3.Request
import one.veriph.sdk.BuildConfig
import retrofit2.Retrofit
import retrofit2.converter.gson.GsonConverterFactory
import java.util.concurrent.TimeUnit
import javax.inject.Singleton

@Module
object RetrofitInstance {
    const val CONNECT_TIMEOUT = "CONNECT_TIMEOUT"
    const val READ_TIMEOUT = "READ_TIMEOUT"
    const val WRITE_TIMEOUT = "WRITE_TIMEOUT"

    @Provides
    @Singleton
    fun provideApi(): VerificationApi {
        val verificationApi: VerificationApi by lazy {

            val httpClient = OkHttpClient.Builder()
                .connectTimeout(20, TimeUnit.SECONDS)
                .writeTimeout(20, TimeUnit.SECONDS)
                .readTimeout(20, TimeUnit.SECONDS)
                .addInterceptor(timeoutInterceptor)
                .build()

            val retrofit = Retrofit.Builder()
                .baseUrl(BuildConfig.API_BASE_URL)
                .addConverterFactory(GsonConverterFactory.create())
                .client(httpClient)
                .build()
            retrofit.create(VerificationApi::class.java)
        }
        return verificationApi
    }

    private val timeoutInterceptor = Interceptor { chain ->
        val request: Request = chain.request()
        var connectTimeout = chain.connectTimeoutMillis()
        var readTimeout = chain.readTimeoutMillis()
        var writeTimeout = chain.writeTimeoutMillis()
        val connectNew = request.header(CONNECT_TIMEOUT)
        val readNew = request.header(READ_TIMEOUT)
        val writeNew = request.header(WRITE_TIMEOUT)
        if (!TextUtils.isEmpty(connectNew)) {
            if (connectNew != null) {
                connectTimeout = connectNew.toInt()
            }
        }
        if (!TextUtils.isEmpty(readNew)) {
            if (readNew != null) {
                readTimeout = readNew.toInt()
            }
        }
        if (!TextUtils.isEmpty(writeNew)) {
            if (writeNew != null) {
                writeTimeout = writeNew.toInt()
            }
        }
        chain
            .withConnectTimeout(connectTimeout, TimeUnit.MILLISECONDS)
            .withReadTimeout(readTimeout, TimeUnit.MILLISECONDS)
            .withWriteTimeout(writeTimeout, TimeUnit.MILLISECONDS)
            .proceed(request)
    }
}