package one.veriph.sdk.ui.verification

import androidx.lifecycle.ViewModel
import androidx.lifecycle.viewModelScope
import kotlinx.coroutines.CoroutineExceptionHandler
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.cancel
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.StateFlow
import kotlinx.coroutines.flow.asStateFlow
import kotlinx.coroutines.flow.update
import kotlinx.coroutines.isActive
import kotlinx.coroutines.launch
import one.veriph.sdk.R
import one.veriph.sdk.data.SessionStyle
import one.veriph.sdk.data.VerificationEventType
import one.veriph.sdk.data.VerificationMethod
import one.veriph.sdk.data.VerificationStatusParams
import one.veriph.sdk.repository.DaggerVerificationRepositoryFactory
import one.veriph.sdk.repository.VerificationRepository
import one.veriph.sdk.ui.util.FlowScreen
import one.veriph.sdk.ui.util.StateBundle
import one.veriph.sdk.util.InternalErrorCodes

class VerificationViewModel :
    ViewModel() {
    private val verificationRepository: VerificationRepository =
        DaggerVerificationRepositoryFactory.create().verificationRepo()

    private val _apiKey = MutableStateFlow<String?>(null)
    private val apiKey: StateFlow<String?> = _apiKey.asStateFlow()

    private val _sessionUuid = MutableStateFlow<String?>(null)
    val sessionUuid: StateFlow<String?> = _sessionUuid.asStateFlow()

    private val _fingerprint = MutableStateFlow<String?>(null)
    private val fingerprint: StateFlow<String?> = _fingerprint.asStateFlow()

    private val _currentScreen = MutableStateFlow<FlowScreen>(FlowScreen.Setup)
    private val currentScreen: StateFlow<FlowScreen> = _currentScreen.asStateFlow()

    private val _sessionStyle = MutableStateFlow<StateBundle<SessionStyle?>>(StateBundle())
    val sessionStyle: StateFlow<StateBundle<SessionStyle?>> = _sessionStyle.asStateFlow()

    private val _countryCodes =
        MutableStateFlow<StateBundle<List<one.veriph.sdk.data.CountryCode>?>>(StateBundle())
    val countryCodes: StateFlow<StateBundle<List<one.veriph.sdk.data.CountryCode>?>> =
        _countryCodes.asStateFlow()

    private val _sessionPhoneBundle = MutableStateFlow<one.veriph.sdk.data.PhoneBundle?>(null)
    private val sessionPhoneBundle: StateFlow<one.veriph.sdk.data.PhoneBundle?> =
        _sessionPhoneBundle.asStateFlow()

    private val _attempt =
        MutableStateFlow<StateBundle<one.veriph.sdk.data.AttemptBundle>>(StateBundle())
    val attempt: StateFlow<StateBundle<one.veriph.sdk.data.AttemptBundle>> = _attempt.asStateFlow()

    private val _cancellationReason = MutableStateFlow<Int?>(null)
    val cancellationReason: StateFlow<Int?> = _cancellationReason.asStateFlow()

    private val _submissionRes =
        MutableStateFlow<StateBundle<one.veriph.sdk.data.AttemptStatusResponse?>>(StateBundle())
    val submissionRes: StateFlow<StateBundle<one.veriph.sdk.data.AttemptStatusResponse?>> =
        _submissionRes.asStateFlow()

    private val _sessionClosed = MutableStateFlow(StateBundle(false))
    val sessionClosed = _sessionClosed.asStateFlow()

    private val coroutineHandler = Dispatchers.IO + CoroutineExceptionHandler { _, throwable ->
        throwable.printStackTrace()
        val update =
            one.veriph.sdk.data.ErrorBundle(
                InternalErrorCodes.ConnectionFailed.value,
                null,
                R.string.error_connection_failed
            )

        when (currentScreen.value) {
            FlowScreen.Setup -> _sessionStyle.update {
                StateBundle(error = update)
            }

            FlowScreen.AttemptCreation -> _attempt.update {
                StateBundle(error = update)
            }

            FlowScreen.AttemptExecution -> {
                when (attempt.value.state!!.attemptCreationResponse!!.type) {
                    VerificationMethod.INVERSE_OTP_SMS,
                    VerificationMethod.INVERSE_OTP_WHATSAPP ->
                        _sessionClosed.update {
                            StateBundle(error = update)
                        }

                    else -> _submissionRes.update { StateBundle(error = update) }
                }
            }

            FlowScreen.PhoneForm -> _countryCodes.update {
                StateBundle(error = update)
            }
        }
    }

    private fun resetState(screen: FlowScreen) {
        _sessionClosed.update { StateBundle(false) }

        if (_attempt.value.state != null || _attempt.value.error != null)
            _attempt.update { StateBundle() }

        _currentScreen.update { screen }

        if (coroutineHandler.isActive)
            coroutineHandler.cancel()
    }

    fun getSessionStyle(
        apiKey: String,
        sessionUuid: String,
        fingerprint: String,
    ) {
        _apiKey.update { apiKey }
        _sessionUuid.update { sessionUuid }
        _fingerprint.update { fingerprint }

        resetState(FlowScreen.Setup)
        _sessionStyle.update { StateBundle(null, null, true) }

        viewModelScope.launch(coroutineHandler) {
            val response: StateBundle<SessionStyle?> =
                verificationRepository.getSessionStyle(apiKey, sessionUuid)
            if (response.state != null) {
                _currentScreen.update { FlowScreen.AttemptCreation }
            }
            _sessionStyle.update { response }
        }
    }

    fun getCountryCodes() {
        _countryCodes.update { StateBundle(null, null, true) }
        viewModelScope.launch(coroutineHandler) {
            val response: StateBundle<List<one.veriph.sdk.data.CountryCode>?> =
                verificationRepository.getCountryCodes(apiKey.value!!)
            _countryCodes.update { response }
        }
    }

    fun setSessionPhoneNumber(
        countryCodeUuid: String,
        cellphoneNumber: String
    ) {
        _currentScreen.update { FlowScreen.AttemptCreation }
        _sessionPhoneBundle.update {
            one.veriph.sdk.data.PhoneBundle(
                countryCodeUuid,
                cellphoneNumber
            )
        }
    }

    @Synchronized
    fun createAttempt() {
        if (attempt.value.isLoading) {
            return
        }

        if (apiKey.value == null || sessionUuid.value == null || fingerprint.value == null) {
            _attempt.update {
                StateBundle(
                    error = one.veriph.sdk.data.ErrorBundle(
                        InternalErrorCodes.MissingParams.value,
                        null,
                        R.string.error_unexpected_flow
                    )
                )
            }
            return
        }

        _attempt.update { StateBundle(null, null, true) }
        val cancellationReason = cancellationReason.value
        if (cancellationReason != null)
            _cancellationReason.update { null }

        viewModelScope.launch(coroutineHandler) {
            val response: StateBundle<one.veriph.sdk.data.AttemptBundle> =
                verificationRepository.createAttempt(
                    apiKey.value!!,
                    sessionUuid.value!!,
                    fingerprint.value!!,
                    sessionPhoneBundle.value,
                    cancellationReason
                )
            if (response.state != null) {
                val destination = when (response.state.status) {
                    one.veriph.sdk.data.AttemptStatus.NEEDS_INPUT -> FlowScreen.PhoneForm
                    else -> FlowScreen.AttemptExecution
                }
                _currentScreen.update { destination }
            }
            _attempt.update { response }
        }
    }

    fun submitAttempt(input: String) {
        _submissionRes.update { StateBundle(null, null, true) }
        viewModelScope.launch(coroutineHandler) {
            val response: StateBundle<one.veriph.sdk.data.AttemptStatusResponse?> =
                verificationRepository.submitAttempt(
                    apiKey.value!!,
                    one.veriph.sdk.data.AttemptSubmission(
                        attempt.value.state!!.attemptCreationResponse!!.uuid,
                        input, fingerprint.value!!, null
                    )
                )
            if (response.error == null && response.state != null && response.state.sessionClosed) {
                _sessionClosed.update { StateBundle(true) }
            } else
                _submissionRes.update { response }
        }
    }

    fun resetAttemptSubmission() {
        _submissionRes.update { StateBundle() }
    }

    fun subscribeToStatus() {
        _sessionClosed.update { StateBundle(false, null, true) }

        viewModelScope.launch(coroutineHandler) {
            val params = VerificationStatusParams(
                attempt.value.state!!.attemptCreationResponse!!.uuid,
                fingerprint.value!!
            )
            verificationRepository.subscribeToStatus(
                sessionUuid.value!!,
                apiKey.value!!,
                params
            ) { event ->
                if (event.eventType == VerificationEventType.SessionClosed)
                    _sessionClosed.update { StateBundle(true) }
            }
        }
    }

    fun invalidateAttempt(cancellationReason: Int? = null) {
        if (coroutineHandler.isActive)
            coroutineHandler.cancel()

        if (cancellationReason != null)
            _cancellationReason.update { cancellationReason }

        _currentScreen.update { FlowScreen.AttemptCreation }
        _attempt.update { StateBundle(null) }
    }
}