package one.veriph.sdk.ui.verification.composable

import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.ExperimentalLayoutApi
import androidx.compose.foundation.layout.Spacer
import androidx.compose.foundation.layout.WindowInsets
import androidx.compose.foundation.layout.fillMaxSize
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.navigationBarsIgnoringVisibility
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.layout.sizeIn
import androidx.compose.foundation.layout.windowInsetsBottomHeight
import androidx.compose.foundation.rememberScrollState
import androidx.compose.foundation.verticalScroll
import androidx.compose.material3.ExperimentalMaterial3Api
import androidx.compose.material3.MaterialTheme
import androidx.compose.material3.ModalBottomSheet
import androidx.compose.material3.Text
import androidx.compose.material3.TextButton
import androidx.compose.material3.rememberModalBottomSheetState
import androidx.compose.runtime.Composable
import androidx.compose.runtime.collectAsState
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.rememberCoroutineScope
import androidx.compose.runtime.saveable.rememberSaveable
import androidx.compose.runtime.setValue
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.res.stringResource
import androidx.compose.ui.unit.dp
import coil.compose.AsyncImage
import kotlinx.coroutines.launch
import one.veriph.sdk.R
import one.veriph.sdk.data.VerificationMethod
import one.veriph.sdk.data.getCancellationReasonsForMethod
import one.veriph.sdk.ui.verification.VerificationViewModel

@OptIn(ExperimentalMaterial3Api::class, ExperimentalLayoutApi::class)
@Composable
fun AttemptExecutor(
    viewModel: VerificationViewModel,
    attempt: one.veriph.sdk.data.AttemptCreationResponse,
    onExpiration: () -> Unit,
    onEditNumber: () -> Unit,
    onCancellation: (reason: Int) -> Unit,
) {
    val style = viewModel.sessionStyle.collectAsState().value.state
    val helpCenterState = rememberModalBottomSheetState()
    val scope = rememberCoroutineScope()
    var helpCenterOpened by remember { mutableStateOf(false) }
    var cancellationRequested by rememberSaveable { mutableStateOf(false) }

    Column(
        modifier = Modifier
            .fillMaxSize(),
        verticalArrangement = Arrangement.Center,
        horizontalAlignment = Alignment.CenterHorizontally,
    ) {
        Column(
            modifier = Modifier
                .verticalScroll(rememberScrollState())
                .padding(32.dp),
            verticalArrangement = Arrangement.Center,
            horizontalAlignment = Alignment.CenterHorizontally,
        ) {
            style?.layout?.logo?.let {
                if (it.type == "url")
                    AsyncImage(
                        model = it.source,
                        contentDescription = null,
                        Modifier
                            .padding(bottom = 12.dp, start = 36.dp, end = 36.dp)
                            .fillMaxWidth()
                            .sizeIn(maxHeight = 76.dp)
                    )
            }

            if (attempt.userPhoneNumberHint !== null)
                PhoneNumberHint(
                    attempt.userPhoneNumberHint,
                    attempt.userCanEditNumber,
                    onEditNumber,
                    Modifier
                        .padding(bottom = 24.dp)
                )

            when (attempt.type) {
                VerificationMethod.INVERSE_OTP_SMS,
                VerificationMethod.INVERSE_OTP_WHATSAPP -> {
                    InverseOTPAttempt(
                        attempt = attempt,
                        onExpiration = {
                            onExpiration()
                        },
                        style = style
                    )
                }

                VerificationMethod.OTP_SMS,
                VerificationMethod.OTP_WHATSAPP,
                VerificationMethod.ROBOCALL -> {
                    TraditionalOTPAttempt(
                        viewModel = viewModel,
                        attempt = attempt,
                        onExpiration = {
                            onExpiration()
                        },
                        style = style
                    )
                }
            }

            TextButton(
                onClick = { helpCenterOpened = true },
                modifier = Modifier.padding(top = 24.dp)
            ) {
                Text(
                    stringResource(id = R.string.attempt_execution_btn_help)
                )
            }
        }
    }

    if (helpCenterOpened) {
        ModalBottomSheet(
            onDismissRequest = {
                helpCenterOpened = false
            },
            sheetState = helpCenterState,
            containerColor = MaterialTheme.colorScheme.surface,
        ) {
            val reasons = getCancellationReasonsForMethod(attempt.type)
            reasons.forEach { item ->
                HelpCenterRow(item, MaterialTheme.colorScheme.onSurface) {
                    scope.launch { helpCenterState.hide() }.invokeOnCompletion {
                        if (!helpCenterState.isVisible) {
                            helpCenterOpened = false
                        }
                    }
                    if (!cancellationRequested) {
                        cancellationRequested = true
                        onCancellation(item.value)
                    }
                }
            }
            Spacer(
                Modifier
                    .windowInsetsBottomHeight(
                        WindowInsets.navigationBarsIgnoringVisibility
                    )
                    .padding(bottom = 12.dp)
            )
        }
    }
}