package one.veriph.sdk.ui.verification.composable

import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.fillMaxSize
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.lazy.LazyColumn
import androidx.compose.foundation.lazy.items
import androidx.compose.foundation.shape.RoundedCornerShape
import androidx.compose.material.icons.Icons
import androidx.compose.material.icons.outlined.Search
import androidx.compose.material3.Card
import androidx.compose.material3.CardDefaults
import androidx.compose.material3.HorizontalDivider
import androidx.compose.material3.Icon
import androidx.compose.material3.MaterialTheme
import androidx.compose.material3.Text
import androidx.compose.material3.TextField
import androidx.compose.runtime.Composable
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.setValue
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.res.stringResource
import androidx.compose.ui.unit.dp
import androidx.compose.ui.window.Dialog
import one.veriph.sdk.R
import one.veriph.sdk.ui.util.getCountryName
import one.veriph.sdk.ui.util.getDefaultCountry
import one.veriph.sdk.ui.util.unaccent

@Composable
fun CountryCodeSelector(
    countryCodes: List<one.veriph.sdk.data.CountryCode>,
    modifier: Modifier = Modifier,
    selectedCountryCodeUuid: String,
    defaultCC: one.veriph.sdk.data.CountryCode,
    onSelected: (countryCode: one.veriph.sdk.data.CountryCode) -> Unit
) {
    var showSelector by remember { mutableStateOf(false) }
    var selected by remember {
        mutableStateOf(
            getDefaultCountry(
                countryCodes,
                selectedCountryCodeUuid,
                defaultCC
            )
        )
    }

    CountryCodeSelectorRow(modifier, selected, true) {
        showSelector = true
    }

    if (showSelector) {
        var searchQuery by remember { mutableStateOf("") }
        var filteredResults by remember { mutableStateOf(countryCodes) }

        Dialog(
            onDismissRequest = { showSelector = false },
        ) {
            Card(
                colors = CardDefaults.cardColors(
                    containerColor = MaterialTheme.colorScheme.surface,
                ),
                modifier = Modifier
                    .fillMaxWidth()
                    .padding(16.dp),
                shape = RoundedCornerShape(16.dp),
            ) {
                Column(
                    modifier = Modifier
                        .fillMaxSize(),
                    verticalArrangement = Arrangement.Top,
                    horizontalAlignment = Alignment.CenterHorizontally,
                ) {
                    TextField(
                        value = searchQuery,
                        singleLine = true,
                        leadingIcon = {
                            Icon(
                                Icons.Outlined.Search,
                                stringResource(id = R.string.form_phone_hint_search),
                                tint = MaterialTheme.colorScheme.onSurface
                            )
                        },
                        modifier = Modifier
                            .fillMaxWidth(),
                        onValueChange = {
                            searchQuery = it
                            filteredResults = countryCodes.filter { code ->
                                getCountryName(code)
                                    .lowercase()
                                    .unaccent()
                                    .contains(searchQuery.lowercase().unaccent())
                            }
                        },
                        label = { Text(stringResource(id = R.string.form_phone_hint_search)) }
                    )

                    LazyColumn {
                        items(filteredResults) { it ->
                            CountryCodeSelectorRow(item = it, onClick = { cc ->
                                selected = cc
                                showSelector = false
                                onSelected(selected)
                            })
                            HorizontalDivider()
                        }
                    }
                }
            }
        }
    }
}
