package one.veriph.sdk.ui.verification.screen

import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.fillMaxSize
import androidx.compose.foundation.layout.padding
import androidx.compose.material3.MaterialTheme
import androidx.compose.material3.Surface
import androidx.compose.runtime.Composable
import androidx.compose.runtime.collectAsState
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.setValue
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.unit.dp
import one.veriph.sdk.R
import one.veriph.sdk.ui.verification.VerificationViewModel
import one.veriph.sdk.ui.verification.composable.ErrorHolder
import one.veriph.sdk.ui.verification.composable.LoadingIndicator
import one.veriph.sdk.util.InternalErrorCodes

@Composable
internal fun AttemptCreationScreen(
    viewModel: VerificationViewModel,
    onExecution: () -> Unit,
    onFinish: () -> Unit,
    onInputNeeded: () -> Unit
) {
    val attemptBundle = viewModel.attempt.collectAsState()
    var isNavigating by remember { mutableStateOf(false) }

    val attempt = attemptBundle.value.state
    val error = attemptBundle.value.error
    val isNew = attempt == null && error == null

    if (isNew && !attemptBundle.value.isLoading)
        viewModel.createAttempt()

    Surface(
        color = MaterialTheme.colorScheme.background
    ) {
        Column(
            modifier = Modifier
                .fillMaxSize()
                .padding(32.dp),
            verticalArrangement = Arrangement.Center,
            horizontalAlignment = Alignment.CenterHorizontally,
        ) {
            if (isNew || attemptBundle.value.isLoading) {
                LoadingIndicator(R.string.attempt_execution_loading)
            } else {
                if (attempt != null) {
                    when (attempt.status) {
                        one.veriph.sdk.data.AttemptStatus.NEEDS_INPUT -> {
                            if (!isNavigating) {
                                isNavigating = true
                                onInputNeeded()
                            }
                        }

                        one.veriph.sdk.data.AttemptStatus.OPENED -> {
                            if (!isNavigating) {
                                isNavigating = true
                                onExecution()
                            }
                        }

                        one.veriph.sdk.data.AttemptStatus.CLOSED, one.veriph.sdk.data.AttemptStatus.EXCEEDED -> {
                            onFinish()
                        }

                        one.veriph.sdk.data.AttemptStatus.ERROR -> {
                            ErrorHolder(
                                errorBundle = one.veriph.sdk.data.ErrorBundle(
                                    InternalErrorCodes.UnknownAttemptError.value
                                ),
                                onRetry = {
                                    viewModel.createAttempt()
                                })
                        }
                    }
                } else if (error != null) {
                    ErrorHolder(errorBundle = error, onRetry = {
                        viewModel.createAttempt()
                    })
                }
            }
        }
    }
}