package one.veriph.sdk.network

import android.util.Log
import com.google.gson.Gson
import dagger.Module
import kotlinx.coroutines.cancel
import kotlinx.coroutines.coroutineScope
import kotlinx.coroutines.delay
import kotlinx.coroutines.flow.flow
import kotlinx.coroutines.isActive
import one.veriph.sdk.data.AttemptCreationResponse
import one.veriph.sdk.data.AttemptInput
import one.veriph.sdk.data.AttemptStatusResponse
import one.veriph.sdk.data.AttemptSubmission
import one.veriph.sdk.data.CountryCode
import one.veriph.sdk.data.SessionStyle
import one.veriph.sdk.data.VerificationEvent
import one.veriph.sdk.data.VerificationEventType
import one.veriph.sdk.data.VerificationStatusParams
import one.veriph.sdk.util.APIUtils
import retrofit2.HttpException
import retrofit2.Response
import java.io.IOException
import javax.inject.Inject

@Module
class VerificationDataSource @Inject constructor(
    private val verificationApi: VerificationApi
) {

    suspend fun getCountryCodes(apiKey: String): Response<List<CountryCode>> =
        verificationApi.getCountryCodes(apiKey)

    suspend fun getSessionStyle(
        apiKey: String,
        languages: String,
        sessionUuid: String
    ): Response<SessionStyle> =
        verificationApi.getSessionStyle(apiKey, languages, sessionUuid)

    suspend fun createAttempt(
        apiKey: String,
        languages: String, attemptInput: AttemptInput
    ): Response<AttemptCreationResponse> =
        verificationApi.createAttempt(apiKey, languages, attemptInput)

    suspend fun submitAttempt(
        apiKey: String,
        languages: String, input: AttemptSubmission
    ): Response<AttemptStatusResponse> =
        verificationApi.submitAttempt(apiKey, languages, input)

    fun subscribeToEvents(uuid: String) = flow {
        coroutineScope {
            val response = verificationApi.subscribeToEvents(uuid, 86400000).execute()
            if (response.isSuccessful) {
                Log.d("V1", "Successful connection")
                val input = response.body()?.byteStream()?.bufferedReader() ?: throw Exception()
                try {
                    while (isActive) {
                        Log.d("V1", "Connection active")
                        val line = input.readLine() ?: continue
                        Log.d("V1", "line: $line")
                        if (line.startsWith("data:")) {
                            try {
                                val gson = Gson()
                                val payload = gson.fromJson(
                                    line.substring(5).trim(),
                                    VerificationEvent::class.java
                                )
                                emit(payload)
                                input.close()
                                coroutineContext.cancel()
                            } catch (e: Exception) {
                                e.printStackTrace()
                            }
                        }
                    }
                } catch (e: IOException) {
                    throw Exception(e)
                } finally {
                    Log.d("V1", "Closed connection")
                    input.close()
                    coroutineContext.cancel()
                }
            } else {
                Log.d("V1", "Unsuccessful connection!")
                coroutineContext.cancel()
                throw HttpException(response)
            }
        }
    }

    fun subscribeViaPolling(
        apiKey: String, forcedLanguage: String?,
        input: VerificationStatusParams
    ) = flow {
        coroutineScope {
            while (isActive) {
                val languages = APIUtils.createLanguagesBundle(forcedLanguage)

                val response = verificationApi.getVerificationStatus(apiKey, languages, input)
                var shouldFinish = APIUtils.isSessionClosedFromStatus(response.code())
                if (response.isSuccessful) {
                    val result: AttemptStatusResponse? = response.body()
                    shouldFinish = result != null && result.sessionClosed
                }
                if (shouldFinish) {
                    emit(VerificationEvent(VerificationEventType.SessionClosed))
                    cancel()
                    break
                } else {
                    delay(5000)
                }
            }
        }
    }
}