package one.veriph.sdk.repository

import dagger.Component
import kotlinx.coroutines.flow.FlowCollector
import one.veriph.sdk.data.AttemptBundle
import one.veriph.sdk.data.AttemptStatus
import one.veriph.sdk.data.AttemptStatusResponse
import one.veriph.sdk.data.AttemptSubmission
import one.veriph.sdk.data.CountryCode
import one.veriph.sdk.data.PhoneBundle
import one.veriph.sdk.data.SessionStyle
import one.veriph.sdk.data.VerificationEvent
import one.veriph.sdk.data.VerificationStatusParams
import one.veriph.sdk.data.getErrorObject
import one.veriph.sdk.network.RetrofitInstance
import one.veriph.sdk.network.VerificationDataSource
import one.veriph.sdk.ui.util.StateBundle
import javax.inject.Inject
import javax.inject.Singleton

@Singleton
@Component(
    modules = [
        VerificationDataSource::class,
        RetrofitInstance::class
    ]
)
internal interface VerificationRepositoryFactory {
    fun verificationRepo(): VerificationRepository
}

class VerificationRepository @Inject constructor(
    private val verificationDataSource: VerificationDataSource
) {
    suspend fun getSessionStyle(
        apiKey: String,
        languages: String,
        sessionUuid: String
    ): StateBundle<SessionStyle?> {
        val result = verificationDataSource.getSessionStyle(apiKey, languages, sessionUuid)
        return if (result.isSuccessful)
            StateBundle(result.body())
        else
            StateBundle(error = result.errorBody()?.getErrorObject(result.code()))
    }

    suspend fun getCountryCodes(apiKey: String): StateBundle<List<CountryCode>?> {
        val result = verificationDataSource.getCountryCodes(apiKey)
        return if (result.isSuccessful)
            StateBundle(result.body())
        else
            StateBundle(error = result.errorBody()?.getErrorObject())
    }

    suspend fun createAttempt(
        apiKey: String, languages: String,
        sessionUuid: String,
        fingerprint: String,
        phoneNumber: PhoneBundle?,
        cancellationReason: Int?
    ): StateBundle<AttemptBundle> {
        val attemptInput =
            one.veriph.sdk.data.AttemptInput(
                fingerprint,
                sessionUuid,
                phoneNumber,
                cancellationReason
            )
        val result = verificationDataSource.createAttempt(apiKey, languages, attemptInput)

        val errorObj = if (!result.isSuccessful) result.errorBody()?.getErrorObject() else null
        val errorCode = errorObj?.internalErrorCode

        val bundle = AttemptBundle(result.body(), result.code(), errorCode)

        return when (bundle.status) {
            AttemptStatus.OPENED -> StateBundle(bundle)

            AttemptStatus.CLOSED,
            AttemptStatus.EXCEEDED,
            AttemptStatus.NEEDS_INPUT -> StateBundle(
                bundle
            )

            AttemptStatus.ERROR -> StateBundle(null, errorObj)
        }
    }

    suspend fun submitAttempt(
        apiKey: String, languages: String,
        input: AttemptSubmission
    ): StateBundle<AttemptStatusResponse?> {
        val result = verificationDataSource.submitAttempt(apiKey, languages, input)
        return if (result.isSuccessful)
            StateBundle(result.body())
        else
            StateBundle(error = result.errorBody()?.getErrorObject())
    }

    suspend fun subscribeToStatus(
        uuid: String,
        apiKey: String,
        forcedLanguage: String?,
        params: VerificationStatusParams,
        collector: FlowCollector<VerificationEvent>
    ) {
        // TODO Integrate Server-Sent Events properly and use both
        //verificationDataSource.subscribeToEvents(uuid).collect(collector)
        verificationDataSource.subscribeViaPolling(apiKey, forcedLanguage, params).collect(collector)
    }
}