package one.veriph.sdk.ui.util

import androidx.compose.material3.ColorScheme
import androidx.compose.material3.darkColorScheme
import androidx.compose.material3.lightColorScheme
import androidx.compose.ui.graphics.Color
import one.veriph.sdk.data.ClientUIPalette

private object LightPaletteDefaults {
    const val PRIMARY = 0xFF5765FC
    const val SECONDARY = 0xFFFC73E9
    const val BACKGROUND = 0xFFFFFBFE
    const val TEXT = 0xFF1C1B1F

    const val WHITE = 0xFFFFFFFF
}

private object DarkPaletteDefaults {
    const val PRIMARY = 0xFF5765FC
    const val SECONDARY = 0xFFFC73E9
    const val BACKGROUND = 0xFF000000
    const val TEXT = 0xFFFFFFFF

    const val WHITE = 0xFFFFFFFF
}

fun getDefaultPalette(isLight: Boolean): ColorScheme {
    return when {
        isLight -> lightColorScheme(
            primary = Color(LightPaletteDefaults.PRIMARY),
            secondary = Color(LightPaletteDefaults.SECONDARY),
            background = Color(LightPaletteDefaults.BACKGROUND),
            surface = Color(LightPaletteDefaults.BACKGROUND),
            onPrimary = Color(LightPaletteDefaults.WHITE),
            onSecondary = Color(LightPaletteDefaults.WHITE),
            onBackground = Color(LightPaletteDefaults.TEXT),
            onSurface = Color(LightPaletteDefaults.TEXT),
        )

        else -> darkColorScheme(
            primary = Color(DarkPaletteDefaults.PRIMARY),
            secondary = Color(DarkPaletteDefaults.SECONDARY),
            background = Color(DarkPaletteDefaults.BACKGROUND),
            surface = Color(DarkPaletteDefaults.BACKGROUND),
            onPrimary = Color(DarkPaletteDefaults.WHITE),
            onSecondary = Color(DarkPaletteDefaults.WHITE),
            onBackground = Color(DarkPaletteDefaults.TEXT),
            onSurface = Color(DarkPaletteDefaults.TEXT),
        )
    }
}

private fun parsePalette(palette: ClientUIPalette): Pair<Boolean, ColorScheme> {
    var primaryColor = Color(LightPaletteDefaults.PRIMARY)
    var onPrimaryColor = Color(LightPaletteDefaults.WHITE)
    var secondaryColor = Color(LightPaletteDefaults.SECONDARY)
    var onSecondaryColor = Color(LightPaletteDefaults.WHITE)
    var backgroundColor = Color(LightPaletteDefaults.BACKGROUND)
    var surfaceColor = Color(LightPaletteDefaults.BACKGROUND)
    var textColor = Color(LightPaletteDefaults.TEXT)

    val isLight = palette.mode == "light"

    palette.primary?.main?.let {
        primaryColor = Color.fromHex(it)
    }
    palette.primary?.contrastText?.let {
        onPrimaryColor = Color.fromHex(it)
    }
    palette.secondary?.main?.let {
        secondaryColor = Color.fromHex(it)
    }
    palette.secondary?.contrastText?.let {
        onSecondaryColor = Color.fromHex(it)
    }
    palette.background?.default?.let {
        backgroundColor = Color.fromHex(it)
    }
    palette.background?.paper?.let {
        surfaceColor = Color.fromHex(it)
    }
    palette.text?.primary?.let {
        textColor = Color.fromHex(it)
    }

    val colorScheme = when {
        isLight -> lightColorScheme(
            primary = primaryColor,
            secondary = secondaryColor,
            background = backgroundColor,
            surface = surfaceColor,
            onPrimary = onPrimaryColor,
            onSecondary = onSecondaryColor,
            onBackground = textColor,
            onSurface = textColor,
        )

        else -> darkColorScheme(
            primary = primaryColor,
            secondary = secondaryColor,
            background = backgroundColor,
            surface = surfaceColor,
            onPrimary = onPrimaryColor,
            onSecondary = onSecondaryColor,
            onBackground = textColor,
            onSurface = textColor,
        )
    }
    return Pair(isLight, colorScheme)
}

fun getPalettes(palettes: ArrayList<ClientUIPalette>): Pair<ColorScheme?, ColorScheme?> {
    var lightPalette: ColorScheme? = null
    var darkPalette: ColorScheme? = null
    for (palette in palettes) {
        val parsedPalette = parsePalette(palette)
        if (parsedPalette.first) // Is Light
            lightPalette = parsedPalette.second
        else
            darkPalette = parsedPalette.second
    }
    return Pair(lightPalette, darkPalette)
}