package one.veriph.sdk.ui.verification

import android.app.Application
import android.content.Intent
import android.os.Bundle
import android.widget.Toast
import androidx.activity.ComponentActivity
import androidx.activity.compose.setContent
import androidx.activity.viewModels
import androidx.lifecycle.Lifecycle
import androidx.lifecycle.lifecycleScope
import androidx.lifecycle.repeatOnLifecycle
import kotlinx.coroutines.launch
import one.veriph.sdk.R
import one.veriph.sdk.data.UIConfig
import one.veriph.sdk.ui.util.UIUtils
import one.veriph.sdk.ui.util.getFingerprint
import one.veriph.sdk.ui.verification.screen.MainScreen
import java.util.Locale

class VerificationActivity : ComponentActivity() {
    private val viewModel: VerificationViewModel by viewModels()

    private fun onVerificationEnded() {
        val i = Intent()
        val session = viewModel.sessionUuid.value
        if (session != null) i.putExtra(EXTRA_SESSION_UUID, session)
        setResult(RESULT_OK, i)
        finish()
    }

    private fun setLocaleIfForced(uiConfig: UIConfig) {

        uiConfig.forcedUIElements?.forcedLanguage?.let { forcedLanguage ->

            resources.let { res ->
                try {
                    val forceConfigLocal = res.configuration
                    val forcedLocal = Locale.forLanguageTag(forcedLanguage.language)
                    forceConfigLocal.setLocale(forcedLocal)
                    resources.updateConfiguration(forceConfigLocal, res.displayMetrics)
                    applicationContext.createConfigurationContext(forceConfigLocal)
                } catch (e: Exception) {
                    e.printStackTrace()
                }
            }
        }

    }

    override fun onCreate(savedInstanceState: Bundle?) {
        val extFingerprint = getFingerprint(applicationContext)
        val sessionUuid = intent.getStringExtra(EXTRA_SESSION_UUID)
        val apiKey = intent.getStringExtra(EXTRA_API_KEY)
        val uiConfig = UIUtils.getUIConfigFromIntent(intent)
        val appContainer = applicationContext as Application
        uiConfig?.let { setLocaleIfForced(it) }
        super.onCreate(savedInstanceState)

        if (sessionUuid == null || apiKey == null) {
            Toast.makeText(this, R.string.error_no_params, Toast.LENGTH_LONG).show()
            setResult(RESULT_CANCELED)
            finish()
            return
        }
        viewModel.initialize(
            apiKey,
            sessionUuid,
            extFingerprint,
            uiConfig?.forcedUIElements?.forcedLanguage?.language
        )
        lifecycleScope.launch {
            repeatOnLifecycle(Lifecycle.State.RESUMED) {
                viewModel.sessionClosed.collect {
                    if (it.state != null && it.state) {
                        onVerificationEnded()
                    }
                }
            }
        }
        setContent {
            MainScreen(
                appContainer, viewModel, uiConfig
            )
        }
    }


    companion object {
        const val EXTRA_SESSION_UUID: String = "sessionUuid"
        const val EXTRA_API_KEY: String = "apiKey"
        const val EXTRA_UI_CONFIG: String = "uiConfig"
    }
}
