package one.veriph.sdk.ui.verification

import androidx.lifecycle.ViewModel
import androidx.lifecycle.viewModelScope
import kotlinx.coroutines.CoroutineExceptionHandler
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.Job
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.StateFlow
import kotlinx.coroutines.flow.asStateFlow
import kotlinx.coroutines.flow.update
import kotlinx.coroutines.launch
import one.veriph.sdk.R
import one.veriph.sdk.data.AttemptBundle
import one.veriph.sdk.data.AttemptStatus
import one.veriph.sdk.data.AttemptStatusResponse
import one.veriph.sdk.data.AttemptSubmission
import one.veriph.sdk.data.CountryCode
import one.veriph.sdk.data.ErrorBundle
import one.veriph.sdk.data.PhoneBundle
import one.veriph.sdk.data.SessionStyle
import one.veriph.sdk.data.VerificationEventType
import one.veriph.sdk.data.VerificationMethod
import one.veriph.sdk.data.VerificationStatusParams
import one.veriph.sdk.repository.DaggerVerificationRepositoryFactory
import one.veriph.sdk.repository.VerificationRepository
import one.veriph.sdk.ui.util.FlowScreen
import one.veriph.sdk.ui.util.StateBundle
import one.veriph.sdk.util.APIUtils
import one.veriph.sdk.util.InternalErrorCodes
import one.veriph.sdk.util.SupportedLanguage


class VerificationViewModel :
    ViewModel() {
    private val verificationRepository: VerificationRepository =
        DaggerVerificationRepositoryFactory.create().verificationRepo()

    private val _apiKey = MutableStateFlow<String?>(null)
    private val apiKey: StateFlow<String?> = _apiKey.asStateFlow()

    private val _sessionUuid = MutableStateFlow<String?>(null)
    val sessionUuid: StateFlow<String?> = _sessionUuid.asStateFlow()

    private val _forcedLang = MutableStateFlow<String?>(null)
    private val forcedLang: StateFlow<String?> = _forcedLang.asStateFlow()

    private val _fingerprint = MutableStateFlow<String?>(null)
    private val fingerprint: StateFlow<String?> = _fingerprint.asStateFlow()

    private val _currentScreen = MutableStateFlow<FlowScreen>(FlowScreen.Setup)
    val currentScreen: StateFlow<FlowScreen> = _currentScreen.asStateFlow()

    private val _sessionStyle = MutableStateFlow<StateBundle<SessionStyle?>>(StateBundle())
    val sessionStyle: StateFlow<StateBundle<SessionStyle?>> = _sessionStyle.asStateFlow()

    private val _countryCodes =
        MutableStateFlow<StateBundle<List<CountryCode>?>>(StateBundle())
    val countryCodes: StateFlow<StateBundle<List<CountryCode>?>> =
        _countryCodes.asStateFlow()

    private val _sessionPhoneBundle = MutableStateFlow<PhoneBundle?>(null)
    private val sessionPhoneBundle: StateFlow<PhoneBundle?> =
        _sessionPhoneBundle.asStateFlow()

    private val _attempt =
        MutableStateFlow<StateBundle<AttemptBundle>>(StateBundle())
    val attempt: StateFlow<StateBundle<AttemptBundle>> = _attempt.asStateFlow()

    private val _submissionRes =
        MutableStateFlow<StateBundle<AttemptStatusResponse?>>(StateBundle())
    val submissionRes: StateFlow<StateBundle<AttemptStatusResponse?>> =
        _submissionRes.asStateFlow()

    private val _sessionClosed = MutableStateFlow(StateBundle(false))
    val sessionClosed = _sessionClosed.asStateFlow()

    private val coroutineHandler = Dispatchers.IO + CoroutineExceptionHandler { _, throwable ->
        throwable.printStackTrace()
        val update =
            ErrorBundle(
                InternalErrorCodes.ConnectionFailed.value,
                null,
                R.string.error_connection_failed
            )

        when (currentScreen.value) {
            FlowScreen.Setup -> _sessionStyle.update {
                StateBundle(error = update)
            }

            FlowScreen.AttemptCreation -> _attempt.update {
                StateBundle(error = update)
            }

            FlowScreen.AttemptExecution -> {
                when (attempt.value.state!!.attemptCreationResponse!!.type) {
                    VerificationMethod.INVERSE_OTP_SMS,
                    VerificationMethod.INVERSE_OTP_WHATSAPP ->
                        _sessionClosed.update {
                            StateBundle(error = update)
                        }

                    else -> _submissionRes.update { StateBundle(error = update) }
                }
            }

            FlowScreen.PhoneForm -> _countryCodes.update {
                StateBundle(error = update)
            }
        }
    }

    private var pollingJob: Job? = null

    private fun resetAttemptState() {
        if (pollingJob?.isActive == true)
            pollingJob!!.cancel()

        _attempt.update { StateBundle(null, null, true) }
        _submissionRes.update { StateBundle() }
        _currentScreen.update { FlowScreen.AttemptCreation }
    }

    private suspend fun getCountryCodes() {
        _countryCodes.update { StateBundle(null, null, true) }

        val response: StateBundle<List<CountryCode>?> =
            verificationRepository.getCountryCodes(apiKey.value!!)
        _countryCodes.update { response }
    }

    private fun subscribeToStatus() {
        _sessionClosed.update { StateBundle(false, null, true) }

        pollingJob = viewModelScope.launch(coroutineHandler) {
            val params = VerificationStatusParams(
                attempt.value.state!!.attemptCreationResponse!!.uuid,
                fingerprint.value!!
            )
            verificationRepository.subscribeToStatus(
                sessionUuid.value!!,
                apiKey.value!!,
                forcedLang.value,
                params
            ) { event ->
                if (event.eventType == VerificationEventType.SessionClosed)
                    _sessionClosed.update { StateBundle(true) }
            }
        }
    }

    private suspend fun requestNewAttempt(cancellationReason: Int?) {
        if (attempt.value.isLoading) {
            return
        }

        resetAttemptState()

        if (apiKey.value == null || sessionUuid.value == null || fingerprint.value == null) {
            _attempt.update {
                StateBundle(
                    error = ErrorBundle(
                        InternalErrorCodes.MissingParams.value,
                        null,
                        R.string.error_unexpected_flow
                    )
                )
            }
            return
        }

        val response: StateBundle<AttemptBundle> =
            verificationRepository.createAttempt(
                apiKey.value!!,
                APIUtils.createLanguagesBundle(forcedLang.value),
                sessionUuid.value!!,
                fingerprint.value!!,
                sessionPhoneBundle.value,
                cancellationReason
            )
        _attempt.update { response }

        if (response.state != null) {
            var destination: FlowScreen? = null
            when (response.state.status) {
                AttemptStatus.NEEDS_INPUT -> {
                    destination = FlowScreen.PhoneForm
                    getCountryCodes()
                }

                AttemptStatus.OPENED -> {
                    destination = FlowScreen.AttemptExecution
                    if (response.state.attemptCreationResponse!!.type == VerificationMethod.INVERSE_OTP_SMS
                        || response.state.attemptCreationResponse.type == VerificationMethod.INVERSE_OTP_WHATSAPP
                    )
                        subscribeToStatus()
                }

                AttemptStatus.CLOSED, AttemptStatus.EXCEEDED -> {
                    _sessionClosed.update { StateBundle(true) }
                }

                AttemptStatus.ERROR -> {
                    destination = null
                }
            }
            if (destination != null)
                _currentScreen.update { destination }
        }
    }

    private fun tryInitializing() {
        viewModelScope.launch(coroutineHandler) {
            val response: StateBundle<SessionStyle?> =
                verificationRepository.getSessionStyle(
                    apiKey.value!!,
                    APIUtils.createLanguagesBundle(forcedLang.value), sessionUuid.value!!
                )

            val statusCode = response.error?.httpResStatusCode
            if (statusCode != null && APIUtils.isSessionClosedFromStatus(statusCode)) {
                _sessionClosed.update { StateBundle(true) }
            } else {
                _sessionStyle.update { response }
                if (response.state != null) {
                    requestNewAttempt(null)
                }
            }
        }
    }

    fun getUsableLanguage(): SupportedLanguage {
        if (forcedLang.value != null) {
            return when (forcedLang.value) {
                SupportedLanguage.Spanish.value -> SupportedLanguage.Spanish
                else -> SupportedLanguage.English
            }
        }
        var deviceLang = APIUtils.getDeviceLang().lowercase()
        if (deviceLang.length > 2)
            deviceLang = deviceLang.substring(0, 2)

        return when (deviceLang) {
            SupportedLanguage.Spanish.value -> SupportedLanguage.Spanish
            else -> SupportedLanguage.English
        }
    }

    fun initialize(
        apiKey: String,
        sessionUuid: String,
        fingerprint: String,
        forcedLanguage: String?,

        ) {
        _apiKey.update { apiKey }
        _sessionUuid.update { sessionUuid }
        _fingerprint.update { fingerprint }
        _sessionStyle.update { StateBundle(null, null, true) }
        _forcedLang.update { forcedLanguage }

        tryInitializing()
    }

    fun retryInitialization() {
        tryInitializing()
    }

    fun setSessionPhoneNumber(
        countryCodeUuid: String,
        cellphoneNumber: String,
    ) {
        _sessionPhoneBundle.update {
            PhoneBundle(
                countryCodeUuid,
                cellphoneNumber
            )
        }
        invalidateAttempt(null)
    }

    fun submitAttempt(input: String) {
        _submissionRes.update { StateBundle(null, null, true) }
        viewModelScope.launch(coroutineHandler) {
            val response: StateBundle<AttemptStatusResponse?> =
                verificationRepository.submitAttempt(
                    apiKey.value!!,
                    APIUtils.createLanguagesBundle(forcedLang.value),
                    AttemptSubmission(
                        attempt.value.state!!.attemptCreationResponse!!.uuid,
                        input, fingerprint.value!!, null
                    )
                )
            if (response.error == null && response.state != null && response.state.sessionClosed) {
                _sessionClosed.update { StateBundle(true) }
            } else
                _submissionRes.update { response }
        }
    }

    fun resetAttemptSubmission() {
        _submissionRes.update { StateBundle() }
    }

    fun invalidateAttempt(cancellationReason: Int? = null) {
        viewModelScope.launch(coroutineHandler) {
            requestNewAttempt(cancellationReason)
        }
    }
}