package one.xingyi.profile;

import lombok.var;
import one.xingyi.helpers.ListHelpers;
import one.xingyi.interfaces.INanoTime;

import javax.management.*;
import java.lang.management.ManagementFactory;
import java.util.ArrayList;
import java.util.List;

public class Profiling implements ProfilingMBean {
    final ProfileImpl profile;
    final static List<Profiling> allProfiling = new ArrayList<>();

    public static String printAll() {
        return String.join("\n", ListHelpers.map(allProfiling, p -> p.profile.print()));
    }

    private INanoTime nanoTime;
    final String packageName;

    public Profiling(INanoTime nanoTime, String packageName) {
        this.nanoTime = nanoTime;
        this.packageName = packageName;
        profile = (ProfileImpl) IProfile.makeProfileMap(nanoTime).withPrefix(packageName);
        allProfiling.add(this);
    }

    public IProfileBuilder registerWithPrefix(String prefix) {
        var mBean = new Profiling(nanoTime, packageName);
        mBean.register(prefix);
        return mBean.profile;
    }

    public IProfileBuilder registerWithPrefixAndMain(String prefix, String main) {
        return registerWithPrefix(prefix).main(main);
    }

    @Override
    public String print() {
        return profile.print();
    }

    public void clear() {
        profile.clear();
    }

    public void register(String name) {
        try {
            MBeanServer mbs = ManagementFactory.getPlatformMBeanServer();
            mbs.registerMBean(this, new ObjectName(packageName + ":type=" + name));
        } catch (Exception e) {
            throw new RuntimeException(e);
        }
    }

    public static void main(String[] args) throws MalformedObjectNameException, NotCompliantMBeanException, InstanceAlreadyExistsException, MBeanRegistrationException, InterruptedException {
        Profiling mBean = new Profiling(INanoTime.realNanoTime, "one.xingyi");
        IProfile p1 = mBean.registerWithPrefixAndMain("op", "test11");
        IProfile p2 = mBean.registerWithPrefixAndMain("two", "test21");

        while (true) {
            Thread.sleep(1000);
            p1.add("test11", 100);
            p1.add("test12", 100);
            p2.add("test21", 100);
            p2.add("test22", 100);
        }
    }

    IProfileInfo mainProfileInfo() {
        return profile.mainProfileInfo();
    }

    @Override
    public int getCount() {
        return mainProfileInfo().getCount();
    }

    @Override
    public long getTotal() {
        return mainProfileInfo().getTotal()/1000000;
    }

    @Override
    public long getSnapshot() {
        return mainProfileInfo().getSnapshot()/1000000;
    }

    @Override
    public String getMain() {
        return profile.prefix + profile.main;
    }
}
