package one.zoop.sdk.scanner


import ScannerResult
import android.Manifest
import android.content.Intent
import android.content.pm.PackageManager
import android.net.Uri
import android.os.Bundle
import android.provider.Settings
import android.util.Log
import androidx.activity.ComponentActivity
import androidx.activity.compose.setContent
import androidx.activity.result.contract.ActivityResultContracts
import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.fillMaxSize
import androidx.compose.material3.Button
import androidx.compose.material3.Surface
import androidx.compose.material3.Text
import androidx.compose.runtime.Composable
import androidx.compose.runtime.SideEffect
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.graphics.Color
import androidx.core.content.ContextCompat
import androidx.lifecycle.viewmodel.compose.viewModel
import com.google.accompanist.systemuicontroller.rememberSystemUiController
import io.sentry.Sentry
import one.zoop.sdk.scanner.model.FileType
import one.zoop.sdk.scanner.model.ScanMode
import one.zoop.sdk.scanner.model.ScannerResultCallback
import one.zoop.sdk.scanner.ui.core.theme.CameraXComposeTheme
import one.zoop.sdk.scanner.ui.navigation.NavigationGraph
import one.zoop.sdk.scanner.utils.ScannerConfigManager
import one.zoop.sdk.scanner.utils.TxnManager
import one.zoop.sdk.scanner.viewmodel.CameraViewModel

class ScannerActivity : ComponentActivity(), ScannerResultCallback {
    private val requestPermissionLauncher =
        registerForActivityResult(ActivityResultContracts.RequestPermission()) { isGranted: Boolean ->
            if (isGranted) {
                Log.i("Permission", "Camera permission granted")
                setupUI()
            } else {
                Log.i("Permission", "Camera permission denied")
                showPermissionDeniedUI()
            }
        }

    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)
        val defaultHandler = Thread.getDefaultUncaughtExceptionHandler()

        Thread.setDefaultUncaughtExceptionHandler { thread, throwable ->
            Log.e("ScannerActivity", "Uncaught exception in thread: ${thread.name}", throwable)
            defaultHandler?.uncaughtException(thread, throwable)
        }
        if (checkCameraPermission()) {
            setupUI()
        } else {
            requestCameraPermission()
        }
    }

    private fun checkCameraPermission(): Boolean {
        return ContextCompat.checkSelfPermission(
            this,
            Manifest.permission.CAMERA
        ) == PackageManager.PERMISSION_GRANTED
    }

    private fun requestCameraPermission() {
        requestPermissionLauncher.launch(Manifest.permission.CAMERA)
    }

    private fun showPermissionDeniedUI() {
        setContent {
            CameraPermissionDeniedUI(
                onRequestPermissionAgain = { requestCameraPermission() },
                onOpenSettings = { openAppSettings() }
            )
        }
    }

    private fun setupUI() {
        setContent {
            val viewModel: CameraViewModel = viewModel()
            viewModel.setCallback(this)
            val systemUiController = rememberSystemUiController()
            CameraXComposeTheme {
                Surface(modifier = Modifier.fillMaxSize()) {
                    SideEffect {
                        systemUiController.setStatusBarColor(
                            color = Color.Black
                        )
                    }
                    NavigationGraph(viewModel = viewModel)
                }
            }
        }
    }

    private fun openAppSettings() {
        val intent = Intent(Settings.ACTION_APPLICATION_DETAILS_SETTINGS).apply {
            data = Uri.fromParts("package", packageName, null)
        }
        startActivity(intent)
    }

    override fun onSuccess(resultPaths: Map<String, Any?>) {
        val resultIntent = Intent()
        val currentTimestamp = System.currentTimeMillis()

        // Get the current scanner configuration
        val config = ScannerConfigManager.getConfig()

        // Create a map of configuration data to include in the result
        val configData = mapOf(
            "documentType" to when(config?.isIdCard){
                true ->"ID Card | " + if(config.isBackImageRequired) "Front & Back" else
                    "Front"
                    else -> "Document"
            },
            "captureType" to when(config?.mode) {
                ScanMode.BATCH -> "Batch"
                ScanMode.SINGLE -> "Single"
                ScanMode.OCR -> "OCR"
                else -> "Single"
            },
            "captureMode" to (if (config?.isAutoEnabled == true) "Auto" else "Manual"),
            "quality" to "${config?.quality}%",
            "format" to when(config?.fileTypes) {
                FileType.JPEG -> "JPEG"
                FileType.PNG -> "PNG"
                FileType.PDF -> "PDF"
                else -> "PNG"
            },
            "isIdCard" to config?.isIdCard,
            "isOcrEnabled" to (config?.ocrMode != null),
            "ocrMode" to config?.ocrMode?.title
        )


        // Create enhanced result map with both paths and config data
        val scannerData = mapOf(
            "imagePaths" to resultPaths.get("imagePath") as List<String>,
            "config" to configData
        )

        val scannerResult = ScannerResult(
            data = scannerData,
            createdAt = currentTimestamp,
            message = "Txn Complete",
            status = "Success",
            errorCode = null
        )

        TxnManager.completeTransaction(true, resultPaths.size)
        resultIntent.putExtra("scan_result", scannerResult)
        setResult(RESULT_OK, resultIntent)
        finish()
    }

    override fun onFailure(errorCode: String, errorMessage: String) {
        val resultIntent = Intent()
        val currentTimestamp = System.currentTimeMillis()
        val scannerResult = ScannerResult(
            data = null,
            createdAt = currentTimestamp,
            message = errorMessage,
            status = "Failure",
            errorCode = errorCode
        )
        TxnManager.completeTransaction(false, 0)
        Sentry.captureMessage("Scanner failed with errorCode: $errorCode, errorMessage: $errorMessage")
        resultIntent.putExtra("scan_result", scannerResult)
        setResult(RESULT_CANCELED, resultIntent)
        finish()
    }
}

@Composable
fun CameraPermissionDeniedUI(
    onRequestPermissionAgain: () -> Unit,
    onOpenSettings: () -> Unit
) {
    Column(
        modifier = Modifier.fillMaxSize(),
        verticalArrangement = Arrangement.Center,
        horizontalAlignment = Alignment.CenterHorizontally
    ) {
        Text("Camera permission is required to scan documents.")
        Button(onClick = onRequestPermissionAgain) {
            Text("Grant Permission")
        }
        Button(onClick = onOpenSettings) {
            Text("Open App Settings")
        }
    }
}