package one.zoop.sdk.scanner.ui.core.components.crop_handler

import android.graphics.PointF
import android.net.Uri
import androidx.compose.foundation.Canvas
import androidx.compose.foundation.background
import androidx.compose.foundation.gestures.detectDragGestures
import androidx.compose.foundation.layout.Box
import androidx.compose.foundation.layout.fillMaxSize
import androidx.compose.runtime.Composable
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateListOf
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.setValue
import androidx.compose.ui.Modifier
import androidx.compose.ui.geometry.Offset
import androidx.compose.ui.geometry.Rect
import androidx.compose.ui.geometry.Size
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.graphics.Paint
import androidx.compose.ui.graphics.PaintingStyle
import androidx.compose.ui.input.pointer.pointerInput
import androidx.compose.ui.res.colorResource
import androidx.compose.ui.tooling.preview.Preview
import androidx.core.graphics.minus
import androidx.core.graphics.plus
import one.zoop.sdk.scanner.R
import one.zoop.sdk.scanner.model.ScannedPage
import one.zoop.sdk.scanner.ui.core.theme.primaryColor
import one.zoop.sdk.scanner.utils.ImageProcessor
import kotlin.math.PI
import kotlin.math.abs
import kotlin.math.atan2
import kotlin.math.sqrt

@Composable
fun DrawViewComposable(
    modifier: Modifier = Modifier,
    pointsState: List<PointF> = listOf(),
    onPointsUpdated: (Array<PointF>, Array<PointF>) -> Unit,
    scannedPage: ScannedPage,
    accentColor: Color,
) {
    var indexMatch by remember { mutableStateOf<List<Int>?>(null) }
    val points = remember { mutableStateListOf(*pointsState.toTypedArray()) }
    val canvasDimension = remember {
        mutableStateOf<PointF?>(null)
    }

    // Paints
    val linePaint = Paint().apply {
        color = accentColor
        strokeWidth = 8f
    }
    val anchorPaint = Paint().apply {
        color = accentColor
        style = PaintingStyle.Fill
    }

    val handlePaintOutside = Paint().apply {
        color = accentColor
        style = PaintingStyle.Fill
        strokeWidth = 30f
    }
    val handlePaintInside = Paint().apply {
        color = Color.White
        style = PaintingStyle.Fill
        strokeWidth = 10f
    }

    Box(
        modifier = modifier
            .pointerInput(Unit) {
                detectDragGestures(
                    onDragStart = { offset ->
                        var isTouchAreaDetected = false
                        val localPosition = PointF(offset.x, offset.y)
                        // Check for anchor handler
                        for (i in points.indices) {
                            val globalCornerPoint = PointF(
                                points[i].x * canvasDimension.value!!.x,
                                points[i].y * canvasDimension.value!!.y
                            )
                            if (canvasDimension.value != null) {
                                val radCircle = sqrt(
                                    ((globalCornerPoint.x - localPosition.x) * (globalCornerPoint.x - localPosition.x) +
                                            (globalCornerPoint.y - localPosition.y) * (globalCornerPoint.y - localPosition.y)).toDouble()
                                )
                                if (radCircle <= 70F) { // Reduced touch area radius from 110F to 70F
                                    isTouchAreaDetected = true
                                    indexMatch = listOf(i)
                                    break
                                }
                            }
                        }
                        // Check for side handler
                        var i = 0
                        while (!isTouchAreaDetected && i < points.size) {
                            val startPoint = PointF(
                                points[i].x * canvasDimension.value!!.x,
                                points[i].y * canvasDimension.value!!.y
                            )
                            val endPoint = PointF(
                                points[(i + 1) % points.size].x * canvasDimension.value!!.x,
                                points[(i + 1) % points.size].y * canvasDimension.value!!.y
                            )
                            val lineCenter: PointF = PointsManipulation.midpoint(startPoint, endPoint)
                            val isAlongHeight: Boolean = i % 2 == 0
                            val width = if (isAlongHeight) 5 * 15 else 13 * 15 // Reduced from 30 to 15 and 130 to 65
                            val height = if (isAlongHeight) 13 * 15 else 5 * 15 // Reduced from 130 to 65 and 50 to 25

                            val rect = Rect(
                                left = lineCenter.x - width / 2,
                                top = lineCenter.y - height / 2,
                                right = lineCenter.x + width / 2,
                                bottom = lineCenter.y + height / 2
                            )

                            if (rect.contains(offset)) {
                                indexMatch = listOf(i, (i + 1) % points.size)
                                isTouchAreaDetected = true
                                break
                            }
                            i++
                        }
                    },
                    onDrag = { change, dragAmount ->
                        if (indexMatch?.size == 2) {
                            val startPoint = PointF(
                                points[indexMatch!![0]].x * canvasDimension.value!!.x,
                                points[indexMatch!![0]].y * canvasDimension.value!!.y
                            )
                            val endPoint = PointF(
                                points[indexMatch!![1]].x * canvasDimension.value!!.x,
                                points[indexMatch!![1]].y * canvasDimension.value!!.y
                            )
                            val delta = PointF(change.position.x, change.position.y) -
                                    PointsManipulation.midpoint(startPoint, endPoint)
                            val newPoint1 = startPoint + delta
                            val newPoint2 = endPoint + delta

                            if (PointsManipulation.checkPosition(newPoint1, canvasDimension.value!!.x.toInt(), canvasDimension.value!!.y.toInt()) &&
                                PointsManipulation.checkPosition(newPoint2, canvasDimension.value!!.x.toInt(), canvasDimension.value!!.y.toInt())
                            ) {
                                val scaledPoints = points.map { point ->
                                    PointF(point.x * canvasDimension.value!!.x, point.y * canvasDimension.value!!.y)
                                }
                                val limitsPt1 = PointsManipulation.isValidMove(indexMatch!![0], scaledPoints.toTypedArray(), newPoint1)
                                val limitsPt2 = PointsManipulation.isValidMove(indexMatch!![1], scaledPoints.toTypedArray(), newPoint2)

                                if (limitsPt1 && limitsPt2) {
                                    points[indexMatch!![0]] = PointF(newPoint1.x / canvasDimension.value!!.x, newPoint1.y / canvasDimension.value!!.y)
                                    points[indexMatch!![1]] = PointF(newPoint2.x / canvasDimension.value!!.x, newPoint2.y / canvasDimension.value!!.y)
                                }
                            }
                        } else if (indexMatch?.size == 1) {
                            if (PointsManipulation.checkPosition(PointF(change.position.x, change.position.y), canvasDimension.value!!.x.toInt(), canvasDimension.value!!.y.toInt())) {
                                val scaledPoints = points.map { point ->
                                    PointF(point.x * canvasDimension.value!!.x, point.y * canvasDimension.value!!.y)
                                }
                                val limits = PointsManipulation.isValidMove(indexMatch!![0], scaledPoints.toTypedArray(), PointF(change.position.x, change.position.y))

                                if (limits) {
                                    val newPosition = PointF(
                                        abs(change.position.x / canvasDimension.value!!.x),
                                        abs(change.position.y / canvasDimension.value!!.y)
                                    )
                                    points[indexMatch!![0]] = newPosition
                                }
                            }
                        }
                    },
                    onDragEnd = {
                        indexMatch = null
                        val originalPoints = ImageProcessor.pointProcessorNormalize(
                            points = points.toList(),
                            viewWidth = canvasDimension.value!!.x.toDouble(),
                            viewHeight = canvasDimension.value!!.y.toDouble(),
                            imageWidth = scannedPage.imageWidth!!.toDouble(),
                            imageHeight = scannedPage.imageHeight!!.toDouble(),
                        )
                        val tfliteModelCorner1 = PointF(originalPoints[0].x, originalPoints[0].y)
                        val tfliteModelCorner2 = PointF(originalPoints[3].x, originalPoints[3].y)
                        val tfliteModelCorner3 = PointF(originalPoints[2].x, originalPoints[2].y)
                        val tfliteModelCorner4 = PointF(originalPoints[1].x, originalPoints[1].y)
                        val reversedPoints = arrayOf(tfliteModelCorner1, tfliteModelCorner2, tfliteModelCorner3, tfliteModelCorner4)
                        onPointsUpdated(originalPoints, reversedPoints)
                    }
                )
            }
    ) {
        Canvas(modifier = Modifier.matchParentSize()) {
            val height = size.height
            val width = size.width
            canvasDimension.value = PointF(width, height)
            // Draw lines and corner circles
            for (i in points.indices) {
                val startPoint = PointF(points[i].x * width, points[i].y * height)
                val endPoint = PointF(points[(i + 1) % points.size].x * width, points[(i + 1) % points.size].y * height)

                drawLine(
                    start = Offset(startPoint.x, startPoint.y),
                    end = Offset(endPoint.x, endPoint.y),
                    color = linePaint.color,
                    strokeWidth = linePaint.strokeWidth
                )
                with(drawContext.canvas) {
                    val lineCenter: PointF = PointsManipulation.midpoint(startPoint, endPoint)
                    val isAlongHeight: Boolean = i % 2 == 0
                    save()

                    val angleRadian = atan2(
                        points[i].x * width - points[(i + 1) % points.size].x * width,
                        points[i].y * height - points[(i + 1) % points.size].y * height
                    )
                    val adjAngle = (if (i % 2 == 0) Math.PI else Math.PI / 2) - angleRadian
                    val angleInDegree = (adjAngle * (180.0 / PI)).toFloat()
                    translate(lineCenter.x, lineCenter.y)
                    rotate(angleInDegree)
                    translate(-lineCenter.x, -lineCenter.y)

                    // Scaled down rectangles
                    drawRect(
                        color = handlePaintOutside.color,
                        topLeft = Offset(
                            lineCenter.x - (if (isAlongHeight) 12.5f else 32.5f), // Reduced from 25f to 12.5f and 65f to 32.5f
                            lineCenter.y - (if (isAlongHeight) 32.5f else 12.5f) // Reduced from 65f to 32.5f and 25f to 12.5f
                        ),
                        size = Size(
                            if (isAlongHeight) 25f else 65f, // Reduced from 40f to 25f and 130f to 65f
                            if (isAlongHeight) 65f else 25f // Reduced from 130f to 65f and 50f to 25f
                        )
                    )

                    drawRect(
                        color = handlePaintInside.color,
                        topLeft = Offset(
                            lineCenter.x - (if (isAlongHeight) 10f else 30f), // Reduced from 20f to 10f and 60f to 30f
                            lineCenter.y - (if (isAlongHeight) 30f else 10f) // Reduced from 60f to 30f and 20f to 10f
                        ),
                        size = Size(
                            if (isAlongHeight) 20f else 60f, // Reduced from 40f to 20f and 120f to 60f
                            if (isAlongHeight) 60f else 20f // Reduced from 120f to 60f and 40f to 20f
                        )
                    )
                    restore()
                }
            }
            for (i in points.indices) {
                val startPoint = PointF(points[i].x * width, points[i].y * height)
                drawCircle(
                    color = anchorPaint.color,
                    radius = DrawViewConst.BLUE_CIRCLE_RADIUS, // Reduced below
                    center = Offset(startPoint.x, startPoint.y)
                )
            }
        }
    }
}

object DrawViewConst {
    const val BLUE_CIRCLE_RADIUS = 15f // Reduced from 25f to 15f
}

@Preview(showBackground = true)
@Composable
fun PreviewDrawViewComposable() {
    val mockPoints = listOf(
        PointF(0.1f, 0.1f),
        PointF(0.9f, 0.1f),
        PointF(0.9f, 0.9f),
        PointF(0.1f, 0.9f)
    )
    val mockScannedPage = ScannedPage(
        rawImagePath = Uri.parse("mock/path"),
        imageHeight = 1000.0,
        imageWidth = 1000.0,
        relativeImageWidth = 1.0,
        relativeImageHeight = 1.0,
        scaleFactorWidth = 1.0,
        scaleFactorHeight = 1.0,
        processedImagePath = "mock/processed/path",
        detectedCorners = listOf(),
        correctedCorners = listOf(),
        imageRotate = 0f
    )

    DrawViewComposable(
        modifier = Modifier
            .fillMaxSize()
            .background(Color.White),
        pointsState = mockPoints,
        scannedPage = mockScannedPage,
        onPointsUpdated = { _, _ -> },
        accentColor = primaryColor
    )
}