package one.zoop.sdk.scanner.ui.layout.Preview

import android.content.Context
import android.content.res.Configuration
import android.graphics.Bitmap
import android.graphics.BitmapFactory
import android.graphics.PointF
import android.graphics.Rect
import android.net.Uri
import android.util.DisplayMetrics
import androidx.activity.compose.BackHandler
import androidx.compose.foundation.Image
import androidx.compose.foundation.background
import androidx.compose.foundation.border
import androidx.compose.foundation.clickable
import androidx.compose.foundation.gestures.detectTransformGestures
import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.Box
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.PaddingValues
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.Spacer
import androidx.compose.foundation.layout.aspectRatio
import androidx.compose.foundation.layout.fillMaxHeight
import androidx.compose.foundation.layout.fillMaxSize
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.height
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.layout.size
import androidx.compose.foundation.layout.width
import androidx.compose.foundation.layout.wrapContentHeight
import androidx.compose.foundation.shape.CircleShape
import androidx.compose.foundation.shape.RoundedCornerShape
import androidx.compose.material3.Button
import androidx.compose.material3.ButtonDefaults
import androidx.compose.material3.CircularProgressIndicator
import androidx.compose.material3.Divider
import androidx.compose.material3.Icon
import androidx.compose.material3.IconButton
import androidx.compose.material3.MaterialTheme
import androidx.compose.material3.Scaffold
import androidx.compose.material3.Surface
import androidx.compose.material3.Text
import androidx.compose.runtime.Composable
import androidx.compose.runtime.LaunchedEffect
import androidx.compose.runtime.MutableIntState
import androidx.compose.runtime.MutableState
import androidx.compose.runtime.collectAsState
import androidx.compose.runtime.getValue
import androidx.compose.runtime.livedata.observeAsState
import androidx.compose.runtime.mutableFloatStateOf
import androidx.compose.runtime.mutableIntStateOf
import androidx.compose.runtime.mutableStateListOf
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.rememberCoroutineScope
import androidx.compose.runtime.setValue
import androidx.compose.runtime.snapshotFlow
import androidx.compose.runtime.snapshots.SnapshotStateList
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.draw.clip
import androidx.compose.ui.draw.clipToBounds
import androidx.compose.ui.draw.rotate
import androidx.compose.ui.draw.scale
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.graphics.asImageBitmap
import androidx.compose.ui.input.pointer.pointerInput
import androidx.compose.ui.layout.ContentScale
import androidx.compose.ui.platform.LocalConfiguration
import androidx.compose.ui.platform.LocalContext
import androidx.compose.ui.platform.LocalDensity
import androidx.compose.ui.res.colorResource
import androidx.compose.ui.res.painterResource
import androidx.compose.ui.text.font.FontWeight
import androidx.compose.ui.text.style.TextAlign
import androidx.compose.ui.unit.dp
import androidx.compose.ui.unit.sp
import androidx.core.content.ContextCompat
import androidx.navigation.NavHostController
import coil.compose.rememberAsyncImagePainter
import com.google.accompanist.pager.ExperimentalPagerApi
import com.google.accompanist.pager.HorizontalPager
import com.google.accompanist.pager.PagerState
import com.google.accompanist.pager.rememberPagerState
import io.sentry.Sentry
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.flow.distinctUntilChanged
import kotlinx.coroutines.launch
import kotlinx.coroutines.withContext
import one.zoop.sdk.scanner.R
import one.zoop.sdk.scanner.ui.core.components.crop_handler.DrawViewComposable
import one.zoop.sdk.scanner.model.DialogContent
import one.zoop.sdk.scanner.model.PreviewButtonControl
import one.zoop.sdk.scanner.model.ScannedPage
import one.zoop.sdk.scanner.ui.core.components.CropDialog
import one.zoop.sdk.scanner.ui.core.components.ExitScanDialog
import one.zoop.sdk.scanner.utils.ImageProcessor
import one.zoop.sdk.scanner.utils.ScannerConfigManager
import one.zoop.sdk.scanner.viewmodel.CameraViewModel
import java.io.File


@ExperimentalPagerApi
@Composable
internal fun ImagePreviewScreen(viewModel: CameraViewModel, navController: NavHostController) {
    val buttonControlRow = remember { mutableStateListOf<PreviewButtonControl>() }

    val pagerState = rememberPagerState()
    val currentImageDisplayedIndex = remember {
        mutableIntStateOf(0)
    }

    val scannedPages by viewModel.capturedImages.observeAsState(emptyList())
    val isIdCard = ScannerConfigManager.getConfig()?.isIdCard == true
    LaunchedEffect(buttonControlRow) {
        buttonControlRow.clear() // Clear existing items to avoid duplicates
        buttonControlRow.add(
            PreviewButtonControl(
                1, "Crop", R.drawable.crop_preview,
                isToggle = true,
                currentToggleState = true,
            )
        )
        buttonControlRow.add(
            PreviewButtonControl(
                3,
                "Rotate",
                R.drawable.rotate_preview,
                false,
                null,
            )
        )
        if (!isIdCard) {
            buttonControlRow.add(
                PreviewButtonControl(
                    6,
                    "Add",
                    R.drawable.add_document_preview,
                    false,
                    null,
                )
            )
        }
        if (!isIdCard) {
            buttonControlRow.add(
                PreviewButtonControl(
                    5,
                    "Delete",
                    R.drawable.delete_preview,
                    false,
                    null,
                )
            )
        }

    }

    //changing of crop toggle state if it is enabled, on device back button.
    BackHandler {
        if (buttonControlRow[0].isToggle && buttonControlRow[0].currentToggleState == true) {
            buttonControlRow[0] = buttonControlRow[0].copy(currentToggleState = false)
        } else {
            navController.popBackStack()
        }
    }
    LaunchedEffect(pagerState) {
        if (viewModel.indexToReplace != null) {
            pagerState.scrollToPage(viewModel.indexToReplace!!)
        } else {
            // need to optimize for singe
            if (pagerState.pageCount != 0) {
                pagerState.scrollToPage(pagerState.pageCount - 1)

            }
        }
        snapshotFlow { pagerState.currentPage }.distinctUntilChanged().collect { page ->
            currentImageDisplayedIndex.intValue = page
            println("Currently displaying page: $page")
        }
    }

    Scaffold(
        containerColor = colorResource(
            id = R.color.white
        ),
    ) { innerPadding ->
        Column(
            modifier = Modifier
                .padding(innerPadding)
                .fillMaxSize(),
            horizontalAlignment = Alignment.CenterHorizontally,
            verticalArrangement = Arrangement.SpaceBetween
        ) {
            when (viewModel.dialogContent.value) {
                is DialogContent.AbortScaDialog -> {
                    ExitScanDialog(
                        title = "Exit Scan?",
                        body = "Your changes will not be saved. Are you sure you wish to exit?",
                        onDismissRequest = { viewModel.dialogContent.value = null },
                        onConfirmation = {
                   
                                viewModel.sendErrorToActivity(
                                    errorCode = "CLS_BTN",
                                    errorMessage = "close button clicked-Camera"
                                )

                        },
                    )
                }

                DialogContent.ExitPreviewDialog -> {
                    ExitScanDialog(
                        title = "Delete Scan?",
                        body = "Are you sure?",
                        onDismissRequest = { viewModel.dialogContent.value = null },
                        onConfirmation = {
                            viewModel.deleteAllScannedPage()
                            viewModel.dialogContent.value = null
                            navController.popBackStack()

                        },
                    )
                }

                DialogContent.CropInfoDialog -> {
                    CropDialog(
                        onDismissRequest = {
                            viewModel.dialogContent.value = null
                        }
                    )
                }

                null -> {
                }

                DialogContent.DeletePageDialog ->{
                    ExitScanDialog(
                        title = "Delete Scan?",
                        body = "Are you sure?",
                        onDismissRequest = { viewModel.dialogContent.value = null },
                        onConfirmation = {
                            viewModel.deleteAllScannedPage()
                            viewModel.dialogContent.value = null
                            navController.popBackStack()

                        },
                    )
                }
            }
            TopAppBar(viewModel = viewModel, navController)
            Text(
                modifier = Modifier
                    .background(
                        Color(
                            ContextCompat.getColor(
                                LocalContext.current,
                                R.color.cSelectedPreviewColor
                            )
                        )
                    )
                    .padding(vertical = 10.dp)
                    .fillMaxWidth(),
                text = "Please ensure scans are not blurry.",
                textAlign = TextAlign.Center,
                fontSize = 12.sp,
                style = MaterialTheme.typography.titleMedium,
                color = Color(
                    ContextCompat.getColor(
                        LocalContext.current,
                        R.color.disabledArrowKeyColor
                    )
                ),
            )
            ImageSliderScreen(
                scannedPages,
                viewModel,
                if (buttonControlRow.isNotEmpty()) buttonControlRow[0].currentToggleState == true else false,
                pagerState,
                remember { mutableFloatStateOf(0.7f) },// Remember scale state
                Modifier.weight(1f),
            )
            ImageSliderControl(
                currentImageDisplayedIndex.intValue + 1,
                totalPages = if (scannedPages.isEmpty()) 0 else scannedPages.size, pagerState,
            )
            Row(
                horizontalArrangement = if (ScannerConfigManager.getConfig()?.isIdCard == true) Arrangement.Center else Arrangement.SpaceAround,
                modifier = Modifier
                    .fillMaxWidth()
                    .padding(top = 21.dp, end = 24.dp, start = 24.dp)
                    .wrapContentHeight(),

                ) {
                val deleteButtonControlState by viewModel.isDeleteInProgress.collectAsState()

                buttonControlRow.forEachIndexed { index, previewButtonControl ->
                    ImagePreviewButtonControl(previewButtonControl,
                        buttonControlRow,
                        index,
                        onToggle = { updatedButtonControl ->
                            buttonControlRow[index] = updatedButtonControl
                        },
                        rotationCallback = {
                            viewModel.rotateScannedPage(
                                90f,
                                pagerState.currentPage,

                                )
                        },
                        deleteButtonCallback = {
                            if (viewModel.capturedImages.value?.size == 1) {
                                viewModel.dialogContent.value = DialogContent.ExitPreviewDialog
                            } else {
                                viewModel.deleteScannedPage(currentImageDisplayedIndex.intValue)
                            }
                        },
                        isEnabled = if (previewButtonControl.id == 5) !deleteButtonControlState else true,
                        addButtonCallback = {
                            navController.popBackStack()
                        })

                }
            }
            Spacer(modifier = Modifier.height(20.dp))
            ImageOperationCTAs(
                navController,
                viewModel,
                Modifier.wrapContentHeight(),
                currentImageDisplayedIndex
            )
            Spacer(modifier = Modifier.height(20.dp))
        }
    }
}


@OptIn(ExperimentalPagerApi::class)
@Composable
fun ImageSliderControl(currentPageNumber: Int, totalPages: Int, pagerState: PagerState) {
    val coroutineScope = rememberCoroutineScope()

    Box(

        Modifier
            .fillMaxWidth()
            .background(
                Color(
                    ContextCompat.getColor(
                        LocalContext.current,
                        R.color.cSelectedPreviewColor
                    )
                )
            ), contentAlignment = Alignment.Center

    ) {
        Row(
            Modifier
                .padding(start = 16.dp, end = 16.dp, bottom = 16.dp, top = 14.dp)
                .background(
                    color = Color(
                        ContextCompat.getColor(
                            LocalContext.current,
                            R.color.secondaryGrey
                        ),
                    ), // Optional: adds a light background to see the rounded corners
                    shape = if (ScannerConfigManager.getConfig()?.isIdCard == true) RoundedCornerShape(
                        50.dp
                    ) else RoundedCornerShape(16.dp) // Adds rounded corners to the container
                )
                .padding(8.dp), // Inner padding to prevent content from touching the rounded edges
            horizontalArrangement = Arrangement.spacedBy(
                8.dp
            ), // Reduces space between elements
            verticalAlignment = Alignment.CenterVertically,

            )
        {
            IconButton(
                enabled = currentPageNumber != 1,
                onClick = {
                    coroutineScope.launch {
                        pagerState.animateScrollToPage((pagerState.currentPage - 1) % totalPages)
                    }
                },
                modifier = Modifier.size(20.dp) // Reduces IconButton size
            ) {
                Icon(
                    tint = if (currentPageNumber != 1) colorResource(id = R.color.EnabledArrowKeyColor)
                    else colorResource(id = R.color.disabledArrowKeyColor),
                    painter = painterResource(id = R.drawable.backward_arrow),
                    contentDescription = "backward_arrow",
                    modifier = Modifier
                        .width(8.dp)
                        .height(16.dp) // Reduces icon size
                )
            }
            if (ScannerConfigManager.getConfig()?.isIdCard == true) {
                Text(
//                    modifier = Modifier.padding(horizontal = 16.dp, vertical = 10.dp),
                    text = if (currentPageNumber == 1) "Front Side" else "Back Side", // Replace with your desired text
                    color = Color.Black,
                    fontSize = 12.sp,
                    fontWeight = FontWeight.Medium // Optional: Add weight for style
                )
            } else {
                Text(
                    text = "$currentPageNumber/$totalPages",
                    color = Color(
                        ContextCompat.getColor(
                            LocalContext.current,
                            R.color.colorPrimary
                        )
                    ),
                    fontSize = 12.sp,
                    // Adds minimal horizontal padding
                )

            }
            IconButton(
                enabled = currentPageNumber != totalPages,
                onClick = {
                    coroutineScope.launch {
                        pagerState.animateScrollToPage((pagerState.currentPage + 1) % totalPages)
                    }
                },
                modifier = Modifier.size(20.dp) // Reduces IconButton size
            ) {
                Icon(
                    tint = if (currentPageNumber != totalPages) colorResource(id = R.color.EnabledArrowKeyColor)
                    else colorResource(id = R.color.disabledArrowKeyColor),
                    painter = painterResource(id = R.drawable.forward_arrow),
                    contentDescription = "forward_arrow",
                    modifier = Modifier
                        .width(8.dp)
                        .height(16.dp)
                )
            }

        }
    }


}

@Composable
fun PreviewButton(
    icon: Int,
    onClick: () -> Unit,
    activeState: Boolean,
    iconColor: Color?,
    backgroundColor: Color?,
) {
    IconButton(
        onClick = onClick,
        Modifier
            .clip(CircleShape)
            .size(40.dp)
            .background(
                color = backgroundColor ?: colorResource(
                    R.color.colorPrimary
                )
            )
    ) {
        Icon(
            painter = painterResource(icon),
            contentDescription = null,
            tint = if (activeState) {
                colorResource(id = R.color.scannerActiveIconColor)
            } else {
                iconColor ?: Color.White
            }
        )
    }
}

@Composable
fun TopAppBar(viewModel: CameraViewModel, navController: NavHostController) {
    Row(
        modifier = Modifier
            .fillMaxWidth()
            .background(Color.White)
            .padding(top = 32.dp, end = 16.dp, start = 16.dp, bottom = 10.dp),
        horizontalArrangement = Arrangement.SpaceBetween,
        verticalAlignment = Alignment.CenterVertically,
    ) {
        Spacer(modifier = Modifier.width(36.dp))

        Text(
            text = viewModel.documentName,
            color = Color.Black,
            fontSize = 16.sp,
            fontWeight = FontWeight.W600,
            style = MaterialTheme.typography.titleMedium,        )
        PreviewButton(
            icon = R.drawable.ic_close,
            onClick = {
                viewModel.dialogContent.value = DialogContent.AbortScaDialog
            },
            activeState = false,
            iconColor = Color(
                ContextCompat.getColor(
                    LocalContext.current,
                    R.color.appBarBackButtonColor
                )
            ),
            backgroundColor = Color(
                ContextCompat.getColor(
                    LocalContext.current,
                    R.color.cSelectedPreviewColor
                )
            ),
        )
    }
}


@Composable
internal fun ImagePreviewButtonControl(
    previewButtonControl: PreviewButtonControl,
    buttonControlRow: SnapshotStateList<PreviewButtonControl>,
    index: Int,
    rotationCallback: () -> Unit,
    onToggle: (PreviewButtonControl) -> Unit,
    deleteButtonCallback: () -> Unit,
    addButtonCallback: () -> Unit,
    isEnabled: Boolean = true
) {
//    var isToggled by remember { mutableStateOf(previewButtonControl.currentToggleState) }
    Surface(
        shape = RoundedCornerShape(4.dp), // Apply rounded corners
        color = if (previewButtonControl.isToggle && buttonControlRow[index].currentToggleState == true) Color(
            color = ContextCompat.getColor(LocalContext.current, R.color.cSelectedPreviewColor),

            ) else
            Color.White,
        modifier = Modifier.clickable {
            if (previewButtonControl.id == 1 && buttonControlRow[index].currentToggleState != null) {
                onToggle(previewButtonControl.copy(currentToggleState = !previewButtonControl.currentToggleState!!))
            } else if (previewButtonControl.id == 3) {
                rotationCallback()
            } else if (previewButtonControl.id == 5) {
                deleteButtonCallback()
            } else if (previewButtonControl.id == 6) {
                addButtonCallback()
            }
        },
    ) {
        Column(
            horizontalAlignment = Alignment.CenterHorizontally,
            modifier = Modifier
                .width(72.dp)
        ) {
            Icon(
                painter = rememberAsyncImagePainter(previewButtonControl.iconRes),
                contentDescription = null,
                modifier = Modifier.padding(start = 24.dp, end = 24.dp, top = 6.dp),
                tint = if (isEnabled) colorResource(
                    id = R.color.promptColour
                ) else colorResource(
                    id = R.color.disabledGreyColor
                ),
            )
            Spacer(modifier = Modifier.height(4.dp))
            Text(
                text = previewButtonControl.name,
                color = if (isEnabled) colorResource(
                    id = R.color.promptColour
                ) else colorResource(
                    id = R.color.disabledGreyColor
                ),
                style = MaterialTheme.typography.titleMedium,
                fontSize = 12.sp,
                modifier = Modifier.padding(bottom = 6.dp)
            )
        }
    }
}

@Composable
fun ImageOperationCTAs(
    navController: NavHostController,
    viewModel: CameraViewModel,
    modifier: Modifier,
    currentImageDisplayedIndex: MutableIntState
) {
    val scannerConfig = ScannerConfigManager.getConfig();
    Row(horizontalArrangement = Arrangement.SpaceAround, modifier = modifier.fillMaxWidth()) {
        PrimaryButton(
            backgroundColor = Color.Transparent,
            contentColor = if (scannerConfig != null) Color(scannerConfig.primaryColor) else Color(
                ContextCompat.getColor(LocalContext.current, R.color.colorPrimary)
            ),
            buttonText = "Retake",
            onClick = {
                viewModel.indexToReplace = currentImageDisplayedIndex.intValue
                viewModel.setRetakeValue(true)
                navController.popBackStack()
            },
            isLoading = false
        )
        PrimaryButton(
            backgroundColor = Color(ScannerConfigManager.getConfig()!!.primaryColor),
            contentColor = if (scannerConfig != null) Color(scannerConfig.primaryTextColor) else Color(
                ContextCompat.getColor(LocalContext.current, R.color.textColorPrimary)
            ),
            buttonText = "Proceed",
            onClick =
            {
                viewModel.saveBtnCallback(
                    onFinish = {
                    }
                )
            },
            isLoading = false
        )
    }
}

@Composable
fun PrimaryButton(
    backgroundColor: Color,
    contentColor: Color,
    buttonText: String,
    onClick: () -> Unit,
    isLoading: Boolean
) {
    val configuration = LocalConfiguration.current
    val scannerConfig = ScannerConfigManager.getConfig()
    val localContext = LocalContext.current
    Button(
        contentPadding = PaddingValues(top = 16.dp, bottom = 16.dp),
        shape = RoundedCornerShape(16.dp),
        onClick = { if (!isLoading) onClick() },
        modifier = Modifier.width((configuration.screenWidthDp * 0.4).dp),
        colors = ButtonDefaults.buttonColors(
            containerColor = backgroundColor,
            contentColor = contentColor
        )
    ) {
        Box(
            contentAlignment = Alignment.Center,
        ) {
            if (isLoading) {
                CircularProgressIndicator(
                    color = if (scannerConfig != null) Color(scannerConfig.primaryTextColor) else Color(
                        ContextCompat.getColor(localContext, R.color.textColorPrimary)
                    ),
                    strokeWidth = 2.dp,
                    modifier = Modifier.size(18.dp) // Adjust size as needed
                )
            } else {
                Text(
                    text = buttonText,
                    fontSize = 16.sp,
                    fontWeight = FontWeight.W800,
                    style = MaterialTheme.typography.displayMedium,                )
            }
        }
    }
}

@Composable
fun adjustForRotation2(
    rotationAngle: Float,
    imageWidth: Float,
    imageHeight: Float,
    configuration: Configuration,
): List<Float> {
    val screenWidth = configuration.screenWidthDp.dp
    val screenHeight = (configuration.screenHeightDp * 0.65).dp

    // Calculate the aspect ratio of the image
    val imageAspectRatio = imageWidth / imageHeight

    // Calculate the width and height based on the aspect ratio
    val adjustedWidth: Float
    val adjustedHeight: Float

    if (rotationAngle == 0f || rotationAngle == 180f) {
        // Portrait orientation
        if (screenWidth.value / screenHeight.value > imageAspectRatio) {
            adjustedHeight = screenHeight.value
            adjustedWidth = screenHeight.value * imageAspectRatio
        } else {
            adjustedWidth = screenWidth.value
            adjustedHeight = screenWidth.value / imageAspectRatio
        }
    } else {
        // Landscape orientation
        if (screenHeight.value / screenWidth.value > imageAspectRatio) {
            adjustedHeight = screenWidth.value
            adjustedWidth = screenWidth.value * imageAspectRatio
        } else {
            adjustedWidth = screenHeight.value
            adjustedHeight = screenHeight.value / imageAspectRatio
        }
    }
    return listOf(adjustedWidth, adjustedHeight)
}


fun Modifier.adjustForRotation(rotationAngle: Float, configuration: Configuration): Modifier {
    val screenWidth = configuration.screenWidthDp.dp
    return if (rotationAngle == 0f || rotationAngle == 180f) {
        this
            .width(screenWidth)
            .height(screenWidth * 16 / 9)
//        this.width(400.dp).height(711.11.dp)
    } else {
        this
            .width(screenWidth * 16 / 9)
            .height(screenWidth)
//        this.width(711.11.dp).height(400.dp)
    }
}


@Composable
fun CropImageContainer(
    scannedPage: ScannedPage,
    context: Context,
    page: Int,
    viewModel: CameraViewModel,
) {
    val scannerConfig = ScannerConfigManager.getConfig()
    if (scannedPage.imageWidth != null && scannedPage.imageHeight != null && scannedPage.scaleFactorWidth != null && scannedPage.scaleFactorHeight != null) {
        val configuration = LocalConfiguration.current
        Box(
            modifier = Modifier
                .height((configuration.screenHeightDp * 0.65).dp)
                .aspectRatio(scannedPage.imageWidth!!.toFloat() / scannedPage.imageHeight!!.toFloat())
                .background(
                    color = Color(
                        ContextCompat.getColor(
                            context,
                            R.color.cSelectedPreviewColor
                        )
                    )
                ),
            contentAlignment = Alignment.Center // Align all children in the center of the Box
// Ensure the Box fills the available space

//                .aspectRatio(9f/16f ,matchHeightConstraintsFirst = true)
        ) {

//            var magnifiedBitmap by remember { mutableStateOf<Bitmap?>(null) }
//            var magnifyingGlassPosition by remember { mutableStateOf<Alignment?>(null) }
//            val scope = rememberCoroutineScope()
//            val currentJob = remember { mutableStateOf<Job?>(null) }
            var pointsState by remember {
                mutableStateOf(scannedPage.correctedCorners ?: scannedPage.detectedCorners!!)
            }
            val adjustDim = adjustForRotation2(
                scannedPage.imageRotate % 360,
                imageHeight = scannedPage.imageHeight!!.toFloat(),
                imageWidth = scannedPage.imageWidth!!.toFloat(),
                configuration = configuration,
            )

            Image(
                painter = rememberAsyncImagePainter(
                    model = scannedPage.rawImagePath
                ),
                contentDescription = "Slider Image",
                contentScale = ContentScale.Fit,
                modifier = Modifier
                    .height(adjustDim[1].dp)
                    .width(adjustDim[0].dp)

                    .rotate(scannedPage.imageRotate),
            )

            DrawViewComposable(
                modifier = Modifier
                    .height(adjustDim[1].dp)
                    .width(adjustDim[0].dp)
//
                    .rotate(scannedPage.imageRotate), pointsState = pointsState,

                scannedPage = scannedPage, onPointsUpdated = { originalPoints, updatedPoints ->
                    CoroutineScope(Dispatchers.Default).launch {
                        val minCropSize =
                            50 // Minimum acceptable width/height in pixels (adjust as needed)

                        // Calculate width and height of the cropping square
                        val width = calculateWidth(updatedPoints)
                        val height = calculateHeight(updatedPoints)

                        if (width < minCropSize || height < minCropSize) {
                            withContext(Dispatchers.Main) {
                                viewModel.dialogContent.value =
                                    DialogContent.CropInfoDialog // Show dialog if crop is too small
                            }
                        }
                        val targetPath = viewModel.initiateCrop(
                            uriRawImagePath = scannedPage.rawImagePath,
                            imageHeight = scannedPage.imageHeight!!,
                            imageWidth = scannedPage.imageWidth!!,
                            context = context,
                            srcPoints = updatedPoints
                        )

                        withContext(Dispatchers.Main) {
                            viewModel.updateCroppedPage(
                                correctedPoints = ImageProcessor.getRelativePoints(
                                    originalPoints.toList(),
                                    scannedPage.imageWidth!!,
                                    scannedPage.imageHeight!!
                                ),
                                processedPath = targetPath,
                                index = page,
                            )
                        }

                    }
                }, accentColor = Color(ContextCompat.getColor(context, R.color.colorPrimary))
            )


//            OverlayDrawViewPreview(
//                drawView,
//                scannedPage.correctedCorners ?: scannedPage.detectedCorners!!,
//                viewModel,
//                viewModel.capturedImages.value!![page],
//                resizedImageWidth,
//                resizedImageHeightDp, page,  modifier =
//                Modifier.adjustForRotation2(
//                    scannedPage.imageRotate % 360,
//                    imageHeight = scannedPage.imageHeight!!.toFloat(),
//                    imageWidth = scannedPage.imageWidth!!.toFloat(),
//                    configuration = LocalConfiguration.current
//                )
//            )
        }
    } else {
        CircularProgressIndicator(
            color = if (scannerConfig != null) Color(scannerConfig.primaryTextColor) else Color(
                ContextCompat.getColor(context, R.color.textColorPrimary)
            )
        )
    }
}

private fun scalePoints(points: List<PointF>, width: Float, height: Float): List<PointF> {
    val scaledPoints = points.map { point ->
        PointF(point.x * width, point.y * height)
    }
    return scaledPoints
}

// Helper function to calculate width of the cropping rectangle
fun calculateWidth(points: Array<PointF>): Float {
    val minX = points.minOf { it.x }
    val maxX = points.maxOf { it.x }
    return maxX - minX
}

// Helper function to calculate height of the cropping rectangle
fun calculateHeight(points: Array<PointF>): Float {
    val minY = points.minOf { it.y }
    val maxY = points.maxOf { it.y }
    return maxY - minY
}

@OptIn(ExperimentalPagerApi::class)
@Composable
fun ImageSliderScreen(
    capturedImages: List<ScannedPage>?,
    viewModel: CameraViewModel,
    currentCropState: Boolean,
    pagerState: PagerState,
    currentScaleState: MutableState<Float>,// Adding a state for scale
    modifier: Modifier,
) {
    val localContext = LocalContext.current
    val scannerConfig = ScannerConfigManager.getConfig()
    if (capturedImages.isNullOrEmpty()) {
        CircularProgressIndicator(
            color = if (scannerConfig != null) Color(scannerConfig.primaryTextColor) else Color(
                ContextCompat.getColor(localContext, R.color.textColorPrimary)
            )
        )
        return
    }
    HorizontalPager(
        count = capturedImages.size,
        state = pagerState,

        userScrollEnabled = !currentCropState,
        modifier = modifier.background(
            color = Color(
                ContextCompat.getColor(
                    LocalContext.current,
                    R.color.cSelectedPreviewColor
                ),
            ),
        )
    ) { page ->
        if (capturedImages[page].detectedCorners != null && currentCropState) {
            CropImageContainer(
                scannedPage = capturedImages[page],
                context = localContext,
                page = page,
                viewModel,
            )

        } else {
            ProcessedImageView(
                capturedImages = capturedImages,
                page = page,
                scaleState = currentScaleState // Pass the scale state
            )
        }
    }
}

@Composable
private fun ProcessedImageView(
    capturedImages: List<ScannedPage>,
    page: Int,
    scaleState: MutableState<Float> // Scale state is passed here
) {
    val configuration = LocalConfiguration.current
    val screenHeight = configuration.screenHeightDp * 0.65
    val screenWidth = configuration.screenWidthDp
    val width = with(LocalDensity.current) { screenWidth.dp }
    val height = with(LocalDensity.current) { screenHeight.dp }
    val scannerConfig = ScannerConfigManager.getConfig()
    val localContext = LocalContext.current
    if (capturedImages[page].processedImagePath != null) {


        Box(modifier = Modifier
            .width(width)
            .height(height)
            // Fixed size for the Box
            .pointerInput(Unit) {
                detectTransformGestures { _, _, zoom, _ ->
                    scaleState.value *= zoom
                    scaleState.value = scaleState.value.coerceIn(0.5f, 3f) // Limit the scale
                }
            }
            .background(
                color = Color(
                    ContextCompat.getColor(
                        LocalContext.current,
                        R.color.cSelectedPreviewColor
                    ),
                ),
            )
            .clipToBounds()) {
            Image(
                painter = rememberAsyncImagePainter(model = capturedImages[page].processedImagePath?.let {
                    File(it)
                }),
                contentDescription = "Slider Image",
                contentScale = ContentScale.Inside, // Ensures the image fits within the Box
                modifier = Modifier
                    .fillMaxSize()
                    .scale(scaleState.value) // Apply scaling
                    .rotate(capturedImages[page].imageRotate) // Apply rotation
            )
        }
    } else {
        CircularProgressIndicator(
            color = if (scannerConfig != null) Color(scannerConfig.primaryTextColor) else Color(
                ContextCompat.getColor(localContext, R.color.textColorPrimary)
            )
        )
    }
}


fun getBitmapFromUri(context: Context, uri: Uri): Bitmap? {
    return try {
        val contentResolver = context.contentResolver
        val inputStream = contentResolver.openInputStream(uri)
        inputStream?.use {
            BitmapFactory.decodeStream(it)
        }
    } catch (e: Exception) {
        Sentry.captureException(e);
        e.printStackTrace()
        null
    }
}


fun convertPixelsToDp(px: Float, context: Context): Float {
    return px / (context.resources.displayMetrics.densityDpi.toFloat() / DisplayMetrics.DENSITY_DEFAULT)
}

fun convertDpToPixel(dp: Float, context: Context): Float {
    return dp * (context.resources.displayMetrics.densityDpi.toFloat() / DisplayMetrics.DENSITY_DEFAULT)
}

//@Composable
//fun OverlayDrawViewPreview(
//    drawView: DrawView,
//    scannedPagePoints: List<PointF>,
//    viewModel: CameraViewModel,
//    scannedPageArg: ScannedPage,
//    imageWidth: Dp,
//    imageHeight: Dp,
//    currentIndex: Int,
//    modifier: Modifier
//) {
//    var rotation2 by remember { mutableFloatStateOf(scannedPageArg.imageRotate) }
//
////    var pointsState by remember {
////        mutableStateListOf<List<PointF>>( points)
////    }
//
//    AndroidView(
//        factory = {
//            drawView.apply {
//                initializeCropScreen()
//                initializeViewModel(viewModel = viewModel)
//                changeCurrentScannedPageIndex(currentIndex)
//                rotation = scannedPageArg.imageRotate
//                imageBitmapSize = PointF(
//                    scannedPageArg.imageWidth!!.toFloat(), scannedPageArg.imageHeight!!.toFloat()
//                )
//                setCornerPoints(pointsState.toTypedArray())
//                scannedPage = scannedPageArg
//            }
//        },
//        modifier = modifier.background(color = Color.Red),
//
////            .width(scannedPageArg.imageWidth!!.dp)
////            .height(scannedPageArg.imageHeight!!.dp),
//        update = {
//            it.rotation = scannedPageArg.imageRotate
//        },
//        // Add a key to trigger recomposition when rotation changes
////        key = rotation2
//    )
//}

@Composable
fun RotatableScalableImage(
    capturedImages: List<ScannedPage>, // Assuming ScannedPage is the data model
    page: Int,
    screenWidthDouble: Double,
    screenHeightDouble: Double,
    scaleState: MutableState<Float> = remember { mutableStateOf(1f) }
) {
    Box(
        Modifier
            .fillMaxWidth()
            .aspectRatio((screenWidthDouble / screenHeightDouble).toFloat())
            .pointerInput(Unit) {
                detectTransformGestures { _, _, zoom, _ ->
                    scaleState.value *= zoom
                    scaleState.value = scaleState.value.coerceIn(0.5f, 3f) // Limit the scale
                }
            }) {
        Image(
            painter = rememberAsyncImagePainter(model = capturedImages[page].processedImagePath?.let {
                File(it)
            }),
            contentDescription = "Slider Image",
            contentScale = ContentScale.Inside,
            modifier = Modifier
                .fillMaxSize()
                .scale(scaleState.value) // Apply the scale transformation
                .rotate(capturedImages[page].imageRotate) // Apply the rotation transformation
        )
    }
}

@Composable
fun MagnifyingGlassView(
    bitmap: Bitmap, modifier: Modifier = Modifier
) {
    Box(
        modifier = modifier.size(100.dp)
    ) {
        Image(
            bitmap.asImageBitmap(),
            contentDescription = null,
            modifier = Modifier
                .fillMaxSize()
                .border(
                    width = 2.dp,
                    color = Color.White,
                    shape = RoundedCornerShape(50) // Adjust to match the shape in your style
                )
                .clip(RoundedCornerShape(50))
        )
        // Horizontal
        Divider(
            color = Color.White, modifier = Modifier
                .align(Alignment.Center)
//                .fillMaxWidth()
                .height(2.dp)
        )

        // Vertical
        Divider(
            color = Color.White,
            modifier = Modifier
                .align(Alignment.Center)
                .fillMaxHeight()
                .width(2.dp)
        )
    }
}


fun getMagnifiedBitmap(
    centerX: Float,
    centerY: Float,
    originalBitmap: Bitmap,

    ): Bitmap {
    //TODO : check bitmap dispose and usability while threading and processing
    // + memory leaks
    val magnificationFactor = 2.0f
    val magnificationRadius = 100
    val left = (centerX - magnificationRadius).toInt()
    val top = (centerY - magnificationRadius).toInt()
    val right = (centerX + magnificationRadius).toInt()
    val bottom = (centerY + magnificationRadius).toInt()

    val sourceLeft = if (left < 0) 0 else left
    val sourceTop = if (top < 0) 0 else top
    val sourceRight = if (right > originalBitmap.width) originalBitmap.width else right
    val sourceBottom = if (bottom > originalBitmap.height) originalBitmap.height else bottom

    val sourceRect = Rect(sourceLeft, sourceTop, sourceRight, sourceBottom)

    return Bitmap.createBitmap(
        originalBitmap, sourceRect.left, sourceRect.top, sourceRect.width(), sourceRect.height()
    ).let { bitmap ->
        Bitmap.createScaledBitmap(
            bitmap,
            (bitmap.width * magnificationFactor).toInt(),
            (bitmap.height * magnificationFactor).toInt(),
            true
        )
    }

}