package one.zoop.sdk.scanner.utils

import android.content.Context
import android.util.Base64
import android.util.Log
import io.sentry.Sentry
import java.io.File
import java.io.FileInputStream
import java.io.FileNotFoundException
import java.io.FileOutputStream
import java.io.IOException
import java.io.InputStream
import java.io.OutputStream
import java.io.UnsupportedEncodingException
import java.nio.ByteBuffer
import java.nio.ByteOrder
import java.nio.MappedByteBuffer
import java.security.InvalidKeyException
import java.security.MessageDigest
import java.security.NoSuchAlgorithmException
import javax.crypto.BadPaddingException
import javax.crypto.Cipher
import javax.crypto.CipherInputStream
import javax.crypto.CipherOutputStream
import javax.crypto.IllegalBlockSizeException
import javax.crypto.NoSuchPaddingException
import javax.crypto.ShortBufferException
import javax.crypto.spec.IvParameterSpec
import javax.crypto.spec.SecretKeySpec

internal class EncryptionUtil() {
    companion object {
        const val TAG: String = "EncryptorUtil"
        const val AES: String = "AES/CBC/PKCS5PADDING"
    }

    // USE WITH LARGE FILE! AES Cipher
    fun encryptFile(password: String, fileToEncryptPath: String, outputFilePath: String): Boolean {
        // Generate secret key from password by calling the keyGen function
        try {
            val key = keyGen(password)
            val c = Cipher.getInstance(AES)
            val iv = ByteArray(c.blockSize)
//            SecureRandom().nextBytes(iv)
            val ivParams = IvParameterSpec(iv)
            c.init(Cipher.ENCRYPT_MODE, key, ivParams)
            val inStream = FileInputStream(fileToEncryptPath)
            CipherOutputStream(FileOutputStream(outputFilePath), c).use { outputStream ->
                inStream.copyTo(outputStream)
                inStream.close()
                outputStream.close()
            }
            return true
        } catch (e: UnsupportedEncodingException) {
            Sentry.captureException(e);
            Log.e(TAG, "Encrypt Error ${e.message.toString()}")
        } catch (e: IllegalBlockSizeException) {
            Sentry.captureException(e);
            Log.e(TAG, "Encrypt Error ${e.message.toString()}")
        } catch (e: BadPaddingException) {
            Sentry.captureException(e);
            Log.e(TAG, "Encrypt Error ${e.message.toString()}")
        } catch (e: InvalidKeyException) {
            Sentry.captureException(e);
            Log.e(TAG, "Encrypt Error ${e.message.toString()}")
        } catch (e: NoSuchPaddingException) {
            Sentry.captureException(e);
            Log.e(TAG, "Encrypt Error ${e.message.toString()}")
        } catch (e: NoSuchAlgorithmException) {
            Sentry.captureException(e);
            Log.e(TAG, "Encrypt Error ${e.message.toString()}")
        } catch (e: ShortBufferException) {
            Sentry.captureException(e);
            Log.e(TAG, "Encrypt Error ${e.message.toString()}")
        } catch (e: NullPointerException) {
            Sentry.captureException(e);
            Log.e(TAG, "Encrypt Error ${e.message.toString()}")
        } catch (e: IllegalArgumentException) {
            Sentry.captureException(e);
            Log.e(TAG, "Encrypt Error ${e.message.toString()}")
        } catch (e: IOException) {
            Sentry.captureException(e);
            Log.e(TAG, "Encrypt Error ${e.message.toString()}")
        } catch (e: IllegalArgumentException) {
            Sentry.captureException(e);
            Log.e(TAG, "Encrypt Error ${e.message.toString()}")
        } catch (e: IOException) {
            Sentry.captureException(e);
            Log.e(TAG, "Encrypt Error ${e.message.toString()}")
        } catch (e: FileNotFoundException) {
            Sentry.captureException(e);
            Log.e(TAG, "Encrypt Error ${e.message.toString()}")
        }
        return false
    }

    // USE WITH LARGE FILE! AES DeCipher
    fun decryptFile(password: String, fileToDecryptPath: String, outputFilePath: String): Boolean {
        try {
            val key = keyGen(password)
            val c = Cipher.getInstance(AES)
            val iv = ByteArray(c.blockSize)
            val ivParams = IvParameterSpec(iv)
            c.init(Cipher.DECRYPT_MODE, key, ivParams)
            val outputStream = FileOutputStream(outputFilePath)
            val inStream = FileInputStream(fileToDecryptPath)
            CipherInputStream(
                inStream,
                c
            ).use { inputStream ->
                inputStream.copyTo(outputStream)
                inputStream.close()
                outputStream.close()
                inStream.close()
            }
            return true
        } catch (e: IllegalBlockSizeException) {
            Sentry.captureException(e);
            Log.e(TAG, "Decrypt Error ${e.message.toString()}")
        } catch (e: BadPaddingException) {
            Sentry.captureException(e);
            Log.e(TAG, "Decrypt Error ${e.message.toString()}")
        } catch (e: InvalidKeyException) {
            Sentry.captureException(e);
            Log.e(TAG, "Decrypt Error ${e.message.toString()}")
        } catch (e: NoSuchPaddingException) {
            Sentry.captureException(e);
            Log.e(TAG, "Decrypt Error ${e.message.toString()}")
        } catch (e: NoSuchAlgorithmException) {
            Sentry.captureException(e);
            Log.e(TAG, "Decrypt Error ${e.message.toString()}")
        } catch (e: ShortBufferException) {
            Sentry.captureException(e);
            Log.e(TAG, "Decrypt Error ${e.message.toString()}")
        } catch (e: NullPointerException) {
            Sentry.captureException(e);
            Log.e(TAG, "Decrypt Error ${e.message.toString()}")
        } catch (e: IllegalArgumentException) {
            Sentry.captureException(e);
            Log.e(TAG, "Decrypt Error ${e.message.toString()}")
        } catch (e: FileNotFoundException) {
            Sentry.captureException(e);
            Log.e(TAG, "Decrypt Error ${e.message.toString()}")
        } catch (e: IOException) {
            Sentry.captureException(e);
            Log.e(TAG, "Decrypt Error ${e.message.toString()}")
        } catch (e: IOException) {
            Sentry.captureException(e);
            Log.e(TAG, "Decrypt Error ${e.message.toString()}")
        }
        return false
    }

    // USE WITH LARGE FILE! AES DeCipher
    fun decryptFileAsStream(
        password: String,
        fileToDecryptPath: String,
        outputStream: OutputStream
    ): Boolean {
        try {
            val key = keyGen(password)
            val c = Cipher.getInstance(AES)
            val iv = ByteArray(c.blockSize)
            val ivParams = IvParameterSpec(iv)
            c.init(Cipher.DECRYPT_MODE, key, ivParams)
            val inStream = FileInputStream(fileToDecryptPath)
            CipherInputStream(
                inStream,
                c
            ).use { inputStream ->
                inputStream.copyTo(outputStream)
                inputStream.close()
                outputStream.close()
                inStream.close()
            }
            return true
        } catch (e: IllegalBlockSizeException) {
            Sentry.captureException(e);
            Log.e(TAG, "Decrypt Error ${e.message.toString()}")
        } catch (e: BadPaddingException) {
            Sentry.captureException(e);
            Log.e(TAG, "Decrypt Error ${e.message.toString()}")
        } catch (e: InvalidKeyException) {
            Sentry.captureException(e);
            Log.e(TAG, "Decrypt Error ${e.message.toString()}")
        } catch (e: NoSuchPaddingException) {
            Sentry.captureException(e);
            Log.e(TAG, "Decrypt Error ${e.message.toString()}")
        } catch (e: NoSuchAlgorithmException) {
            Sentry.captureException(e);
            Log.e(TAG, "Decrypt Error ${e.message.toString()}")
        } catch (e: NoSuchAlgorithmException) {
            Sentry.captureException(e);
            Log.e(TAG, "Decrypt Error ${e.message.toString()}")
        } catch (e: ShortBufferException) {
            Sentry.captureException(e);
            Log.e(TAG, "Decrypt Error ${e.message.toString()}")
        } catch (e: NullPointerException) {
            Sentry.captureException(e);
            Log.e(TAG, "Decrypt Error ${e.message.toString()}")
        } catch (e: IllegalArgumentException) {
            Sentry.captureException(e);
            Log.e(TAG, "Decrypt Error ${e.message.toString()}")
        } catch (e: FileNotFoundException) {
            Sentry.captureException(e);
            Log.e(TAG, "Decrypt Error ${e.message.toString()}")
        } catch (e: IOException) {
            Sentry.captureException(e);
            Log.e(TAG, "Decrypt Error ${e.message.toString()}")
        }
        return false
    }

    fun decryptStreamAsByteBuffer(
        password: String,
        fileToDecryptInStream: InputStream,
    ): MappedByteBuffer? {
        try {
            val key = keyGen(password)
            val c = Cipher.getInstance(AES)
            val iv = ByteArray(c.blockSize)
            val ivParams = IvParameterSpec(iv)
            c.init(Cipher.DECRYPT_MODE, key, ivParams)

            val decryptedBytes =
                CipherInputStream(fileToDecryptInStream, c).use { cipherInputStream ->
                    cipherInputStream.readBytes()
                }
            val byteBuffer = ByteBuffer.allocateDirect(decryptedBytes.size)
            byteBuffer.put(decryptedBytes)
            byteBuffer.flip()
            byteBuffer.order(ByteOrder.nativeOrder())
            fileToDecryptInStream.close()
            return byteBuffer as MappedByteBuffer
        } catch (e: IllegalBlockSizeException) {
            Sentry.captureException(e);
            Log.e(TAG, "Decrypt Error ${e.message.toString()}")
        } catch (e: BadPaddingException) {
            Sentry.captureException(e);
            Log.e(TAG, "Decrypt Error ${e.message.toString()}")
        } catch (e: InvalidKeyException) {
            Sentry.captureException(e);
            Log.e(TAG, "Decrypt Error ${e.message.toString()}")
        } catch (e: NoSuchPaddingException) {
            Sentry.captureException(e);
            Log.e(TAG, "Decrypt Error ${e.message.toString()}")
        } catch (e: NoSuchAlgorithmException) {
            Sentry.captureException(e);
            Log.e(TAG, "Decrypt Error ${e.message.toString()}")
        } catch (e: NoSuchAlgorithmException) {
            Sentry.captureException(e);
            Log.e(TAG, "Decrypt Error ${e.message.toString()}")
        } catch (e: ShortBufferException) {
            Sentry.captureException(e);
            Log.e(TAG, "Decrypt Error ${e.message.toString()}")
        } catch (e: NullPointerException) {
            Sentry.captureException(e);
            Log.e(TAG, "Decrypt Error ${e.message.toString()}")
        } catch (e: IllegalArgumentException) {
            Sentry.captureException(e);
            Log.e(TAG, "Decrypt Error ${e.message.toString()}")
        } catch (e: FileNotFoundException) {
            Sentry.captureException(e);
            Log.e(TAG, "Decrypt Error ${e.message.toString()}")
        } catch (e: IOException) {
            Sentry.captureException(e);
            Log.e(TAG, "Decrypt Error ${e.message.toString()}")
        }
        return null
    }


    // AES DeCipher function with return function for String data
    fun decryptString(val1: String, password: String): String? {
        try {
            val key = keyGen(password)
            val c = Cipher.getInstance(AES)
            Log.d(TAG, c.blockSize.toString())
            val iv = ByteArray(c.blockSize)
            val ivParams = IvParameterSpec(iv)
            c.init(Cipher.DECRYPT_MODE, key, ivParams)
            val decodedValue = Base64.decode(val1, Base64.DEFAULT)
            val decValue = c.doFinal(decodedValue)
            return String(decValue)
        } catch (e: IllegalBlockSizeException) {
            Sentry.captureException(e);
            Log.e(TAG, "AesDecipherAnyString ${e.message.toString()}")
        } catch (e: BadPaddingException) {
            Sentry.captureException(e);
            Log.e(TAG, "AesDecipherAnyString ${e.message.toString()}")
        } catch (e: InvalidKeyException) {
            Sentry.captureException(e);
            Log.e(TAG, "AesDecipherAnyString ${e.message.toString()}")
        } catch (e: NoSuchPaddingException) {
            Sentry.captureException(e);
            Log.e(TAG, "AesDecipherAnyString ${e.message.toString()}")
        } catch (e: NoSuchAlgorithmException) {
            Sentry.captureException(e);
            Log.e(TAG, "AesDecipherAnyString ${e.message.toString()}")
        } catch (e: ShortBufferException) {
            Sentry.captureException(e);
            Log.e(TAG, "AesDecipherAnyString ${e.message.toString()}")
        } catch (e: NullPointerException) {
            Sentry.captureException(e);
            Log.e(TAG, "AesDecipherAnyString ${e.message.toString()}")
        } catch (e: IllegalArgumentException) {
            Sentry.captureException(e);
            Log.e(TAG, "AesDecipherAnyString ${e.message.toString()}")
        }
        return null
    }

    // AES DeCipher function with return function for String data
    fun encryptString(val1: String, password: String): String? {
        try {
            val key = keyGen(password)
            val c = Cipher.getInstance(AES)
            val iv = ByteArray(c.blockSize)
            val ivParams = IvParameterSpec(iv)
            c.init(Cipher.ENCRYPT_MODE, key, ivParams)
            val encValue = c.doFinal(val1.toByteArray())
            return Base64.encodeToString(encValue, Base64.DEFAULT)
        } catch (e: IllegalBlockSizeException) {
            Sentry.captureException(e);
            Log.e(TAG, "AesDecipherAnyString ${e.message.toString()}")
        } catch (e: BadPaddingException) {
            Sentry.captureException(e);
            Log.e(TAG, "AesDecipherAnyString ${e.message.toString()}")
        } catch (e: InvalidKeyException) {
            Sentry.captureException(e);
            Log.e(TAG, "AesDecipherAnyString ${e.message.toString()}")
        } catch (e: NoSuchPaddingException) {
            Sentry.captureException(e);
            Log.e(TAG, "AesDecipherAnyString ${e.message.toString()}")
        } catch (e: NoSuchAlgorithmException) {
            Sentry.captureException(e);
            Log.e(TAG, "AesDecipherAnyString ${e.message.toString()}")
        } catch (e: ShortBufferException) {
            Sentry.captureException(e);
            Log.e(TAG, "AesDecipherAnyString ${e.message.toString()}")
        } catch (e: NullPointerException) {
            Sentry.captureException(e);
            Log.e(TAG, "AesDecipherAnyString ${e.message.toString()}")
        } catch (e: IllegalArgumentException) {
            Sentry.captureException(e);
            Log.e(TAG, "AesDecipherAnyString ${e.message.toString()}")
        }
        return null
    }


    // Generate secret key function for SecretKeySpec data form
    private fun keyGen(password: String): SecretKeySpec {
        val digest = MessageDigest.getInstance("SHA-256")
        // Transform password to UTF-8 byte array format
        val bytes = password.toByteArray(Charsets.UTF_8)
        digest.update(bytes, 0, bytes.size)
        val key = digest.digest()
        return SecretKeySpec(key, "AES")
    }

    fun getPathFromAsset(context: Context, assetFileName: String): String {
        val tempFile = File(context.cacheDir, assetFileName)
        context.assets.open(assetFileName).use { inputStream ->
            FileOutputStream(tempFile).use { outputStream ->
                val buffer = ByteArray(1024)
                var length: Int
                while (inputStream.read(buffer).also { length = it } > 0) {
                    outputStream.write(buffer, 0, length)
                }
            }
        }
        return tempFile.path
    }

    fun test(context: Context) {
        val modelAssetPath =
            getPathFromAsset(context, "keypoint_320_256_float16_V5_1_output.tflite")
        val outputPath = context.cacheDir.path + "/enc_model.enc"
        val result = encryptFile(
            "zoop@security",
            modelAssetPath,
            outputPath
        )
        Log.d("EncryptorUtil", result.toString())
        val decFilePath = context.cacheDir.path + "/dec_model.tflite"
        val decPath = decryptFile(
            "zoop@security",
            outputPath,
            decFilePath
        )
        Log.d("EncryptorUtil", decPath.toString())
    }

    fun testByteEnc(context: Context) {
        val tempFile = File(context.cacheDir, "dec_model.tflite")
        val assetPath = getPathFromAsset(context, "enc_model.enc")
        val bytes = decryptStreamAsByteBuffer(
            "zoop@security",
            File(assetPath).inputStream()
        )
        tempFile.writeBytes(bytes!!.array())
    }
}