package one.zoop.sdk.scanner.utils

import android.util.Log
import com.google.firebase.firestore.FirebaseFirestore
import kotlinx.coroutines.tasks.await
import java.math.BigInteger
import java.security.MessageDigest

internal class FirestoreService(private val firestore: FirebaseFirestore) {

    fun String.toMD5(): String {
        val md = MessageDigest.getInstance("MD5")
        val digest = md.digest(this.toByteArray())
        val bigInt = BigInteger(1, digest)
        return String.format("%032x", bigInt)
    }

    suspend fun verifyLicense(clientId: String, secretKey: String): Boolean {
        val document = firestore.collection("keys").document(clientId).get()
            .await()
        val md5Hash = secretKey.toMD5()
        var secretKeyHash = document.get("secret_key_hash") as List<String>

        if (document.exists() && secretKeyHash.contains(md5Hash)) {
            return true
        } else {
            Log.v("License Verification failed", "Incorrect License Key has been passed")
            return false
        }
    }

    suspend fun getOrgDetails(clientId: String): OrgDetails? {
        val documentSnapshot = firestore.collection("keys").document(clientId).get().await()
        val orgPid = documentSnapshot.getString("org_pid") ?: return null

        val orgDocument = firestore.collection("orgs").document(orgPid).get().await()

        if (!orgDocument.exists()) {
            return null
        }

        val isSandbox = orgDocument.getBoolean("is_sandbox") ?: true

        return OrgDetails(orgPid = orgPid, isSandbox = isSandbox)
    }

    suspend fun verifyAndUpdateLicense(
        clientId: String,
        secretKey: String,
        orgPid: String?
    ): Boolean {
        try {
            val documentSnapshot = firestore.collection("keys").document(clientId).get().await()
            if (!documentSnapshot.exists()) {
                Log.v("License Verification", "Invalid client ID.")
                return false
            }

            val md5Hash = secretKey.toMD5()
            val secretKeyHash =
                documentSnapshot.get("secret_key_hash") as? List<String> ?: return false
            if (!secretKeyHash.contains(md5Hash)) {
                Log.v("License Verification", "Invalid secret key.")
                return false
            }

            val orgPid = documentSnapshot.getString("org_pid") ?: return false
            val orgDocument = firestore.collection("orgs").document(orgPid).get().await()

            if (!orgDocument.exists()) {
                Log.v("License Verification", "Organization not found.")
                return false
            }

            val isSandbox = orgDocument.getBoolean("is_sandbox") ?: true
            val txnCountField = if (isSandbox) "sandbox_txn_count" else "annual_txn_count"
            val txnLimitField = if (isSandbox) "sandbox_txn_limit" else "annual_txn_limit"
            val txnCount = orgDocument.getLong(txnCountField) ?: 0
            val txnLimit = orgDocument.getLong(txnLimitField) ?: 1000 // Default to 1000 if not set

            if (txnCount >= txnLimit) {
                Log.v(
                    "License Verification",
                    if (isSandbox) "Sandbox transaction limit exceeded." else "Annual transaction limit exceeded."
                )
                return false
            }

            val monthlyTxnCountField = if (isSandbox) "sandbox_txn_count" else "monthly_txn_count"
            val monthlyTxnLimitField = if (isSandbox) "sandbox_txn_limit" else "monthly_txn_limit"

            if (monthlyTxnCountField >= monthlyTxnLimitField) {
                Log.v("License Verification", "Monthly transaction limit exceeded.")
                return false
            }
            // Increment transaction count
//        firestore.collection("orgs").document(orgPid)
//            .update(txnCountField, txnCount + 1)
//            .await()

            // Log the transaction completion for analytics
            // Log the transaction in a separate transactions collection

            return true
        } catch (e: Exception) {
            return false;
        }
    }

    suspend fun logNewTransaction(clientId: String, orgPid: String, isSandbox: Boolean): String {
        val transactionData = mapOf(
            "timestamp" to System.currentTimeMillis(),
            "client_id" to clientId,
            "org_pid" to orgPid,
            "transaction_type" to if (isSandbox) "sandbox" else "production",
            "initialise" to true,
            "completed" to false
        )

        val transactionRef = firestore.collection("transactions").add(transactionData).await()
        return transactionRef.id
    }

    suspend fun markTransactionCompleted(
        transactionId: String,
        success: Boolean,
        size: Int
    ) {
        val transactionDocument =
            firestore.collection("transactions").document(transactionId).get().await()
        if (transactionDocument.exists()) {
            val orgPid = transactionDocument.getString("org_pid") ?: return
            val isSandbox = transactionDocument.getString("transaction_type") == "sandbox"
            val txnCountField = if (isSandbox) "sandbox_txn_count" else "annual_txn_count"

            firestore.collection("transactions").document(transactionId)
                .update("completed", success)
                .await()

            if (success) {
                val orgDocument = firestore.collection("orgs").document(orgPid).get().await()

                val txnCount = orgDocument.getLong(txnCountField) ?: 0
                firestore.collection("orgs").document(orgPid)
                    .update(txnCountField, txnCount + 1)
                    .await()
                firestore.collection("orgs").document(orgPid)
                    .update(txnCountField, txnCount + 1)
                    .await()
                firestore.collection("transactions").document(transactionId)
                    .update("pages_scanned", size)
                    .await()
                firestore.collection("orgs").document(orgPid)
                    .update("monthly_txn_count", txnCount + 1)
                    .await()
            }
        }
    }


}

internal data class OrgDetails(val orgPid: String, val isSandbox: Boolean)
