package one.zoop.sdk.scanner.ui.core.components.crop_handler
//Flutter Point Manipulation logic used here for handling crop handler logic to identify when to move the corners
//i.e if the corner are out of screen , crossing each other , too close to each other
import android.graphics.PointF
import android.util.Log
import kotlin.math.abs
import kotlin.math.acos
import kotlin.math.max
import kotlin.math.min
import kotlin.math.sqrt

object PointsManipulation {
    //basic utility functions
    private fun PointF.distanceTo(other: PointF): Float {
        val dx = this.x - other.x
        val dy = this.y - other.y
        return sqrt(dx * dx + dy * dy)
    }

    private fun PointF.subtract(other: PointF) {
        this.x -= other.x
        this.y -= other.y
    }

    private fun PointF.add(other: PointF) {
        this.x += other.x
        this.y += other.y
    }

    //helper function
    fun calculateLimit(
        vector1: PointF, vector2: Array<PointF>?, index: Int, diagonal1: Float, diagonal2: Float
    ): Boolean {
        var limitY = 0.0F
        var limitX = 0.0F

        when (index) {
            //for 0th point (TL) check dist between 0 and 3
            0 -> {
                limitX = vector1.x.minus(vector2?.get(3)?.x ?: 0F)
                limitY = vector2?.get(1)?.y?.minus(vector1.y) ?: 0F
            }

            1 -> {
                limitX = vector2?.get(3)?.x?.minus(vector1.x) ?: 0F
                limitY = vector1.y.minus(vector2?.get(0)?.y ?: 0F)
            }

            2 -> {
                limitX = vector1.x.minus(vector2?.get(1)?.x ?: 0F)
                limitY = vector1.y.minus(vector2?.get(3)?.y ?: 0F)
            }

            3 -> {
                limitX = vector1.x.minus(vector2?.get(0)?.x ?: 0F)
                limitY = vector2?.get(2)?.y?.minus(vector1.y) ?: 0F
            }
        }
        Log.d(DrawView.DEBUG_LOG, "X:$limitX Y:$limitY Diag1:$diagonal1 Diag:$diagonal2")
        return limitX > 20 && limitY > 20 && diagonal1 > 80 && diagonal2 > 80
    }

    //logic 2
    // Define the minimum and maximum diagonal thresholds
    private const val minDiagonalThreshold = 80
//        private val maxDiagonalThreshold = 200

    fun isValidMove2(
        index: Int, points: Array<PointF>, currentPoint: PointF
    ): Boolean {
        val oppositePointIndex = (index + 2) % 4
        val adjacentPointIndices = listOf((index + 1) % 4, (index + 3) % 4)
        val oppositePoint = points[oppositePointIndex]

        // Check the diagonal distance between the current point and the opposite point
        val diagonalDistance = calculateDistance(currentPoint, oppositePoint)
        Log.d(DrawView.DEBUG_LOG, diagonalDistance.toString())
        if (diagonalDistance < minDiagonalThreshold
//                || diagonalDistance > maxDiagonalThreshold
        ) {
            return false
        }

        var limitX = 0.0F
        var limitY = 0.0F
        when (index) {
            0 -> {
                limitX = min(points[1].x, points[2].x)
                limitY = min(points[2].y, points[3].y)
                if (currentPoint.x > limitX || currentPoint.y > limitY) {
                    return false
                }
            }

            1 -> {
                limitX = max(points[0].x, points[3].x)
                limitY = min(points[2].y, points[3].y)
                if (currentPoint.x < limitX || currentPoint.y > limitY) {
                    return false
                }
            }

            2 -> {
                limitX = max(points[0].x, points[3].x)
                limitY = max(points[0].y, points[1].y)
                if (currentPoint.x < limitX || currentPoint.y < limitY) {
                    return false
                }
            }

            3 -> {
                limitX = min(points[1].x, points[2].x)
                limitY = max(points[0].y, points[1].y)
                if (currentPoint.x > limitX || currentPoint.y < limitY) {
                    return false
                }
            }
        }

        // Check that the current point does not cross the line formed by the adjacent points
//            for (adjacentIndex in adjacentPointIndices) {
//                val adjacentStart = points[adjacentIndex]
//                val adjacentEnd = points[(adjacentIndex + 1) % 4]
//
//                if (isPointOnLine(currentPoint, adjacentStart, adjacentEnd)) {
//                    return false
//                }
//            }

        return true
    }

    private fun calculateEuclideanDistance(point1: PointF, point2: PointF): Float {
        val dx = point2.x - point1.x
        val dy = point2.y - point1.y
        return sqrt(dx * dx + dy * dy)
    }

    private const val ANGLE_THRESHOLD = 7
    fun isValidMove(
        index: Int, points: Array<PointF>, currentPoint: PointF
    ): Boolean {
        //create copy to avoid changing to original points
        val oldPoints = points.copyOf()
        oldPoints[index] = currentPoint

//        val oldAngles = calculateAngles(points)
        val newAngles = calculateAngles(oldPoints)
        Log.d("angle debug", "angle between " + newAngles.contentToString())
        if (calculateEuclideanDistance(
                points[(index + 1) % points.size],
                currentPoint
            ) < 185f || calculateEuclideanDistance(
                points[(index - 1 + points.size) % points.size],
                currentPoint
            ) < 185f
        )
            return false
        for (i in 0..3) {
            if (newAngles[i] < 0 || newAngles[i] < ANGLE_THRESHOLD || newAngles[i] > 140) {
                return false
            }
        }

        return true
    }

    private fun calculateAngles(points: Array<PointF>): DoubleArray {
        val angles = DoubleArray(4)

        //3 , 0 , 1
        //0 , 1 , 2
        //1 , 2 ,3
        //2 ,3 , 0
        for (p2 in 0..3) {
            val p1 = (p2 + 3) % 4
            val p3 = (p2 + 1) % 4

            // AB = (x1 - x0, y1 - y0)
            // BC = (x2 - x1, y2 - y1)
            val ab_vector =
                PointF(points[p1].x - points[p2].x, points[p1].y - points[p2].y)
            val bc_vector =
                PointF(points[p3].x - points[p2].x, points[p3].y - points[p2].y)

            val dotProduct = ab_vector.x * bc_vector.x + ab_vector.y * bc_vector.y

            // magnitudes of the vectors
            val ab_magnitude = sqrt(ab_vector.x * ab_vector.x + ab_vector.y * ab_vector.y)
            val bc_magnitude = sqrt(bc_vector.x * bc_vector.x + bc_vector.y * bc_vector.y)

            // cosine angle between vectors AB and BC
            val cosTheta = dotProduct / (ab_magnitude * bc_magnitude)

            // angle in radians
            val angleRadians = acos(cosTheta).toDouble()

            // sign of the angle based on vector orientation
            val orientation = ab_vector.x * bc_vector.y - ab_vector.y * bc_vector.x

            //rad to degree
            val angleDegrees =
                if (orientation >= 0) Math.toDegrees(angleRadians) else -Math.toDegrees(angleRadians)

            angles[p2] = angleDegrees

        }

        return angles
    }

    private fun calculateDistance(point1: PointF, point2: PointF): Float {
        val dx = point1.x - point2.x
        val dy = point1.y - point2.y
        return sqrt(dx * dx + dy * dy)
    }

    private fun isPointOnLine(point: PointF, lineStart: PointF, lineEnd: PointF): Boolean {
        // Check if the point is on the line formed by lineStart and lineEnd
        val distanceStartToPoint = calculateDistance(lineStart, point)
        val distanceStartToEnd = calculateDistance(lineStart, lineEnd)
        val distanceEndToPoint = calculateDistance(lineEnd, point)

        // Allow some tolerance (e.g., 5 pixels) for floating-point errors
        val tolerance = 5

        // The point is on the line if the sum of distances to lineStart and lineEnd is equal to the line's length
        return abs(distanceStartToPoint + distanceEndToPoint - distanceStartToEnd) <= tolerance
    }

    //0 -> TL ,1-> TR ,2->BR ,3->BL
    fun calculateDiagonals(points: Array<PointF>): PointF {
        val d1 = points[0].distanceTo(points[2])
        val d2 = points[1].distanceTo(points[3])
        return PointF(d1, d2)
    }

    fun checkPosition(localPosition: PointF, imageWidth: Int, imageHeight: Int): Boolean {
        return localPosition.x < imageWidth && localPosition.y < imageHeight && localPosition.x > 0 && localPosition.y > 0
    }

    fun clipPoints(diagonals: PointF, updatedPoints: Array<PointF>?) {
        val offset1 = PointF(30F, 20F)
        val offset2 = PointF(50F, 50F)

        diagonals.let {
            if (it.x < 82) {
                updatedPoints?.let { points ->
                    points[0].subtract(offset1)
                    points[2].add(offset1)
                }
            }

            if (it.y < 82) {
                updatedPoints?.let { points ->
                    points[1].add(offset2)
                    points[3].subtract(offset2)
                }
            }
        }
    }

    fun midpoint(a: PointF, b: PointF): PointF {
        return PointF(
            (a.x + b.x) / 2,
            (a.y + b.y) / 2,
        );
    }
}