package one.zoop.sdk.scanner.ui.layout.Scanner

import android.graphics.Bitmap
import android.graphics.BitmapFactory
import android.graphics.Matrix
import android.graphics.PointF
import android.media.ExifInterface
import android.net.Uri
import android.util.Log
import androidx.activity.compose.BackHandler
import androidx.activity.compose.rememberLauncherForActivityResult
import androidx.activity.result.PickVisualMediaRequest
import androidx.activity.result.contract.ActivityResultContracts
import androidx.compose.animation.core.Animatable
import androidx.compose.animation.core.FastOutSlowInEasing
import androidx.compose.animation.core.animateFloatAsState
import androidx.compose.animation.core.tween
import androidx.compose.foundation.Canvas
import androidx.compose.foundation.Image
import androidx.compose.foundation.background
import androidx.compose.foundation.border
import androidx.compose.foundation.clickable
import androidx.compose.foundation.interaction.MutableInteractionSource
import androidx.compose.foundation.interaction.collectIsHoveredAsState
import androidx.compose.foundation.interaction.collectIsPressedAsState
import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.Box
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.PaddingValues
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.Spacer
import androidx.compose.foundation.layout.fillMaxSize
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.height
import androidx.compose.foundation.layout.offset
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.layout.size
import androidx.compose.foundation.layout.width
import androidx.compose.foundation.shape.CircleShape
import androidx.compose.foundation.shape.RoundedCornerShape
import androidx.compose.material3.Button
import androidx.compose.material3.ButtonDefaults
import androidx.compose.material3.Card
import androidx.compose.material3.CardDefaults
import androidx.compose.material3.CircularProgressIndicator
import androidx.compose.material3.Icon
import androidx.compose.material3.IconButton
import androidx.compose.material3.MaterialTheme
import androidx.compose.material3.Text
import androidx.compose.runtime.Composable
import androidx.compose.runtime.LaunchedEffect
import androidx.compose.runtime.collectAsState
import androidx.compose.runtime.getValue
import androidx.compose.runtime.livedata.observeAsState
import androidx.compose.runtime.mutableIntStateOf
import androidx.compose.runtime.mutableStateListOf
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.setValue
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.draw.clip
import androidx.compose.ui.draw.shadow
import androidx.compose.ui.geometry.Offset
import androidx.compose.ui.geometry.Size
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.graphics.ImageBitmap
import androidx.compose.ui.graphics.asImageBitmap
import androidx.compose.ui.layout.ContentScale
import androidx.compose.ui.layout.onGloballyPositioned
import androidx.compose.ui.layout.positionInRoot
import androidx.compose.ui.platform.LocalConfiguration
import androidx.compose.ui.platform.LocalContext
import androidx.compose.ui.platform.LocalDensity
import androidx.compose.ui.res.colorResource
import androidx.compose.ui.res.painterResource
import androidx.compose.ui.res.stringResource
import androidx.compose.ui.text.font.FontWeight
import androidx.compose.ui.text.style.TextAlign
import androidx.compose.ui.text.style.TextOverflow
import androidx.compose.ui.unit.Dp
import androidx.compose.ui.unit.IntOffset
import androidx.compose.ui.unit.dp
import androidx.compose.ui.unit.sp
import androidx.compose.ui.unit.toSize
import androidx.compose.ui.viewinterop.AndroidView
import androidx.core.content.ContextCompat
import androidx.lifecycle.viewmodel.compose.viewModel
import androidx.navigation.NavHostController
import coil.compose.rememberAsyncImagePainter
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.delay
import kotlinx.coroutines.launch
import kotlinx.coroutines.withContext
import one.zoop.sdk.scanner.R
import one.zoop.sdk.scanner.ui.core.components.crop_handler.DrawView
import one.zoop.sdk.scanner.model.DialogContent
import one.zoop.sdk.scanner.model.ScanMode
import one.zoop.sdk.scanner.model.ScannerButtonControl
import one.zoop.sdk.scanner.ui.core.components.ExitScanDialog
import one.zoop.sdk.scanner.utils.ScannerConfigManager
import one.zoop.sdk.scanner.viewmodel.CameraViewModel
import one.zoop.sdk.scanner.viewmodel.CornerPointStatus


@Composable
fun CameraScannerUI(viewModel: CameraViewModel = viewModel(), navController: NavHostController) {
    var previewWidth by remember { mutableIntStateOf(0) }
    var previewHeight by remember { mutableIntStateOf(0) }
    val context = LocalContext.current

    val displayMetrics = context.resources.displayMetrics
    val screenWidthDouble = displayMetrics.widthPixels.toDouble()
    val screenHeightDouble = displayMetrics.heightPixels * 0.65
    val drawView = remember {
        DrawView(context)
    }

    BackHandler {
        if (viewModel.isRetake.value == true) {
            navController.navigate("previewScreen")

        } else {
            viewModel.dialogContent.value = DialogContent.AbortScaDialog
        }
    }

    Box(
        modifier = Modifier
            .fillMaxSize()
            .background(Color.Black)
    ) {
        val localDensity = LocalDensity.current
        var bottomControlSheetTop by remember { mutableStateOf<Dp?>(null) }
        when (viewModel.dialogContent.value) {
            is DialogContent.AbortScaDialog -> {
                ExitScanDialog(
                    title = "Exit Scan?",
                    body = "Your changes will not be saved. Are you sure you wish to exit?",
                    onDismissRequest = { viewModel.dialogContent.value = null },
                    onConfirmation = {
                        viewModel.sendErrorToActivity(
                            errorCode = "CLS_BTN", errorMessage = "close button clicked-Camera"
                        )

                    },
                )
            }

            DialogContent.ExitPreviewDialog -> {}

            else -> {}
        }

        CameraPreview(
            viewModel = viewModel,
            modifier = Modifier
                .fillMaxSize()
                .onGloballyPositioned { coordinates ->
                    val size = coordinates.size.toSize()
                    val width = size.width.toInt()
                    val height = size.height.toInt()
                    previewWidth = width
                    previewHeight = height

                },
            drawView = drawView, screenWidth = screenWidthDouble,
            screenHeight = screenHeightDouble, navController = navController,
        )
        // Top shadow and controls - specify a proper height for the gradient
        Box(
            modifier = Modifier
                .fillMaxWidth()
                .height(124.dp)
                .align(Alignment.TopCenter)
        ) {
            TopBar(
                viewModel = viewModel,
                navController = navController,
                modifier = Modifier
            )
        }

        Box(
            modifier = Modifier
                .fillMaxWidth()
                .padding(bottom = 80.dp)  // Position it above the bottom control sheet
              , contentAlignment = Alignment.BottomCenter
        ) {
            AutoCapturePrompt(
                viewModel,
                Modifier.align(Alignment.BottomCenter),
                navController = navController,
            )
        }
        OverlayDrawView(
            drawView
        )
        Box(
            modifier = Modifier
                .fillMaxWidth()
                .height(124.dp)
                .align(Alignment.BottomCenter)
        ) {
            BottomControlSheet(
                Modifier
                    .align(Alignment.BottomCenter)
                    .onGloballyPositioned { coordinates ->
                        bottomControlSheetTop =
                            with(localDensity) { coordinates.positionInRoot().y.toDp() }
                    },
                navController,
                viewModel,
                screenWidth = screenWidthDouble,
                screenHeight = screenHeightDouble,
            )
        }

        // Add the capture animation overlay
        if (viewModel.showCaptureAnimation.value) {
            CaptureAnimation(
                imageUri = viewModel.animationImageUri.value,
                onAnimationComplete = { viewModel.endCaptureAnimation() }
            )
        }
    }
}

@Composable
fun CaptureAnimation(
    imageUri: Uri?,
    onAnimationComplete: () -> Unit
) {
    val density = LocalDensity.current
    val context = LocalContext.current

    // Animation values
    val animationDuration = 1500 // milliseconds

    // Store final size
    val finalSize = 36.dp

    // Animation progress
    val animationProgress = remember { Animatable(0f) }

    // Get screen dimensions
    val screenWidth = LocalConfiguration.current.screenWidthDp.dp
    val screenHeight = LocalConfiguration.current.screenHeightDp.dp

    // Image dimensions state
    val imageBitmap = remember { mutableStateOf<ImageBitmap?>(null) }
    val imageAspectRatio = remember { mutableStateOf(1f) }
    val initialSize = remember { mutableStateOf(Size(screenWidth.value, screenHeight.value)) }

    // Load image and calculate aspect ratio
    LaunchedEffect(imageUri) {
        imageUri?.let {
            withContext(Dispatchers.IO) {
                try {
                    // Get EXIF orientation
                    val exifOrientation =
                        context.contentResolver.openInputStream(it)?.use { inputStream ->
                            val exifInterface = ExifInterface(inputStream)
                            exifInterface.getAttributeInt(
                                ExifInterface.TAG_ORIENTATION,
                                ExifInterface.ORIENTATION_UNDEFINED
                            )
                        } ?: ExifInterface.ORIENTATION_NORMAL

                    context.contentResolver.openInputStream(it)?.use { inputStream ->
                        val options = BitmapFactory.Options().apply {
                            inJustDecodeBounds = true
                        }
                        BitmapFactory.decodeStream(inputStream, null, options)

                        // Reset stream
                        inputStream.close()

                        context.contentResolver.openInputStream(it)?.use { resetStream ->
                            val bitmap = BitmapFactory.decodeStream(resetStream)

                            if (bitmap != null) {
                                // Apply EXIF rotation
                                val rotatedBitmap = rotateBitmapIfNeeded(bitmap, exifOrientation)

                                // Calculate aspect ratio
                                val aspectRatio =
                                    rotatedBitmap.width.toFloat() / rotatedBitmap.height.toFloat()
                                imageAspectRatio.value = aspectRatio

                                // Calculate initial size to fill screen while maintaining aspect ratio
                                val screenAspectRatio = screenWidth.value / screenHeight.value
                                initialSize.value = if (aspectRatio > screenAspectRatio) {
                                    // Image is wider than screen, fit to width
                                    Size(screenWidth.value, screenWidth.value / aspectRatio)
                                } else {
                                    // Image is taller than screen, fit to height
                                    Size(screenHeight.value * aspectRatio, screenHeight.value)
                                }

                                imageBitmap.value = rotatedBitmap.asImageBitmap()
                            }
                        }
                    }
                } catch (e: Exception) {
                    Log.e("CaptureAnimation", "Error loading image", e)
                }
            }
        }
    }


    // Animation effect
    LaunchedEffect(key1 = imageUri, key2 = imageBitmap.value) {
        if (imageBitmap.value != null) {
            animationProgress.animateTo(
                targetValue = 1f,
                animationSpec = tween(
                    durationMillis = animationDuration,
                    easing = FastOutSlowInEasing
                )
            )
            delay(50) // Small delay before removing the animation element
            onAnimationComplete()
        }
    }

    // Calculate current width and height
    val currentWidth = lerp(initialSize.value.width, finalSize.value, animationProgress.value).dp
    val currentHeight = if (imageAspectRatio.value > 0) {
        (currentWidth.value / imageAspectRatio.value).dp
    } else {
        lerp(initialSize.value.height, finalSize.value, animationProgress.value).dp
    }

    // X position: Move the CENTER of the image to the bottom-right corner
    val startX = screenWidth / 2 // Center of the image at start
    val endX =
        screenWidth - (finalSize / 2) - 16.dp // Center of the image at end (bottom-right with padding)
    val currentX = lerp(startX.value, endX.value, animationProgress.value).dp

    // Y position: Move the CENTER of the image to the bottom-right corner
    val startY = screenHeight / 2 // Center of the image at start
    val endY =
        screenHeight - (finalSize / 2) - 16.dp // Center of the image at end (bottom-right with padding)
    val currentY = lerp(startY.value, endY.value, animationProgress.value).dp

    // Current alpha/opacity
    val alpha = 1f - (0.3f * animationProgress.value)

    Box(
        modifier = Modifier.fillMaxSize()
    ) {
        imageUri?.let {
            imageBitmap.value?.let { bitmap ->
                Image(
                    bitmap = bitmap,
                    contentDescription = null,
                    contentScale = ContentScale.Fit,
                    modifier = Modifier
                        .width(currentWidth)
                        .height(currentHeight)
                        .offset {
                            // Offset is adjusted to position the center of the image at currentX, currentY
                            IntOffset(
                                x = with(density) { (currentX - (currentWidth / 2)).roundToPx() },
                                y = with(density) { (currentY - (currentHeight / 2)).roundToPx() }
                            )
                        }
                        .clip(RoundedCornerShape(percent = 5))
                        .background(Color.Black.copy(alpha = 0.5f * (1 - animationProgress.value)))
                )
            } ?: run {
                // Show placeholder while loading
                Box(
                    modifier = Modifier
                        .size(finalSize)
                        .offset {
                            IntOffset(
                                x = with(density) { (endX - (finalSize / 2)).roundToPx() },
                                y = with(density) { (endY - (finalSize / 2)).roundToPx() }
                            )
                        }
                        .clip(RoundedCornerShape(percent = 5))
                        .background(Color.Gray)
                )
            }
        }
    }
}

// Helper function for linear interpolation
private fun lerp(start: Float, end: Float, fraction: Float): Float {
    return start + (end - start) * fraction
}

// Helper function to rotate bitmap based on EXIF orientation
fun rotateBitmapIfNeeded(bitmap: Bitmap, orientation: Int): Bitmap {
    val matrix = Matrix()
    when (orientation) {
        ExifInterface.ORIENTATION_FLIP_HORIZONTAL -> matrix.setScale(-1f, 1f)
        ExifInterface.ORIENTATION_ROTATE_180 -> matrix.setRotate(180f)
        ExifInterface.ORIENTATION_FLIP_VERTICAL -> {
            matrix.setRotate(180f)
            matrix.postScale(-1f, 1f)
        }

        ExifInterface.ORIENTATION_TRANSPOSE -> {
            matrix.setRotate(90f)
            matrix.postScale(-1f, 1f)
        }

        ExifInterface.ORIENTATION_ROTATE_90 -> matrix.setRotate(90f)
        ExifInterface.ORIENTATION_TRANSVERSE -> {
            matrix.setRotate(-90f)
            matrix.postScale(-1f, 1f)
        }

        ExifInterface.ORIENTATION_ROTATE_270 -> matrix.setRotate(-90f)
        else -> return bitmap
    }

    try {
        val rotatedBitmap = Bitmap.createBitmap(
            bitmap, 0, 0, bitmap.width, bitmap.height, matrix, true
        )
        // Recycle the old bitmap if it's not the same as the new one
        if (rotatedBitmap != bitmap) {
            bitmap.recycle()
        }
        return rotatedBitmap
    } catch (e: OutOfMemoryError) {
        Log.e("CaptureAnimation", "Out of memory when rotating bitmap", e)
        return bitmap
    }
}

@Composable
fun AutoCapturePrompt(
    viewModel: CameraViewModel,
    modifier: Modifier,
    navController: NavHostController
) {
    val scannerConfig = ScannerConfigManager.getConfig()
    if (scannerConfig != null && scannerConfig.isAutoEnabled) {
        val autoCapturePromptState by viewModel.autoCapturePromptState.collectAsState()
        LaunchedEffect(autoCapturePromptState) {

            if (autoCapturePromptState == CornerPointStatus.DETECTED) {
                if (viewModel.isRetake.value == true || ScannerConfigManager.getConfig()?.mode == ScanMode.SINGLE || ScannerConfigManager.getConfig()?.mode == ScanMode.OCR) {
                    if (viewModel.isRetake.value == true && viewModel.indexToReplace != null) {
                        navController.navigate("previewScreen")

                    }
                }
            }


        }
        val autoCaptureText = getAutoPromptText(cornerPointStatus = autoCapturePromptState)
        if (autoCaptureText != null) Box(
            Modifier
                .padding(10.dp)
                .fillMaxSize()
        ) {
            Text(
                modifier = modifier
                    .clip(RoundedCornerShape(20.dp)) .align(Alignment.BottomCenter)

                    .background(color = colorResource(id = R.color.promptColour).copy(alpha = 0.95f))
                    .padding(12.dp),
                text = autoCaptureText,
                style = MaterialTheme.typography.titleLarge,
                fontSize = 16.sp,
                fontWeight = FontWeight.W700,
                textAlign = TextAlign.Center,
                color = Color.White
            )
        }

    }
}


@Composable
fun BottomControlSheet(
    modifier: Modifier = Modifier,
    navController: NavHostController,
    viewModel: CameraViewModel,
    screenHeight: Double,
    screenWidth: Double,
) {
    val context = LocalContext.current
    val capturedImages by viewModel.capturedImages.observeAsState(emptyList())
    val scannerConfig = ScannerConfigManager.getConfig()
    var isGalleryButtonEnabled by remember { mutableStateOf(true) }

    val singleImagePickerLauncher = rememberLauncherForActivityResult(
        contract = ActivityResultContracts.GetContent()
    ) { uri: Uri? ->
        uri?.let {
            viewModel.processGalleryImages(
                filenameFormat = "yyyy-MM-dd-HH-mm-ss-SSS",
                outputDirectory = viewModel.getOutputDirectory(context),
                onImageCaptured = {
                    val config = ScannerConfigManager.getConfig()

                    // Trigger animation if needed for gallery images
                    if (viewModel.isRetake.value != true && ((config?.mode == ScanMode.BATCH && !config.isIdCard) || (config?.isIdCard == true && (viewModel.capturedImages.value?.size
                            ?: 0) < 2))
                    ) {
                        viewModel.startCaptureAnimation(uri)
                    }

                    // Case 1: Handle retake mode
                    if (viewModel.isRetake.value == true) {
                        navController.navigate("previewScreen")
                    }
                    // Case 2: Handle ID card scanning
                    else if (config?.isIdCard == true) {
                        // Navigate if back image is not required OR if back image is required and we have both images
                        if (!config.isBackImageRequired || (viewModel.capturedImages.value?.size == 2)) {
                            navController.navigate("previewScreen")
                        }
                        // Otherwise no action needed (waiting for second image capture)
                    }
                    // Case 3: Handle non-ID card scanning
                    else if (config?.mode == ScanMode.SINGLE || config?.mode == ScanMode.OCR) {
                        navController.navigate("previewScreen")
                    }
                    // Case 4: All other scenarios (e.g., batch mode)
                    // No navigation needed - continue capturing images
                },
                screenHeight = screenHeight,
                screenWidth = screenWidth,
                onError = {

                },
                corners = null, context = context, isMultiImage = false, selectedUris = listOf(uri)
            )
        }
    }

    val multipleImagePickerLauncher = rememberLauncherForActivityResult(
        // Ensure maxItems is always at least 2 for the multi-picker
        contract = ActivityResultContracts.PickMultipleVisualMedia(
            maxItems = if (scannerConfig?.isIdCard == true)
                maxOf(2 - capturedImages.size, 2) // Never go below 2
            else
                20
        ),

        ) { uris: List<Uri> ->
        if (uris.isNotEmpty()) {
            // Trigger animation for the first selected image
            if (uris.isNotEmpty()) {
                if (viewModel.isRetake.value != true && ((scannerConfig?.mode == ScanMode.BATCH && !scannerConfig.isIdCard) || (scannerConfig?.isIdCard == true && (viewModel.capturedImages.value?.size
                        ?: 0) < 2))
                ) {
                    viewModel.startCaptureAnimation(uris.first())
                }
            }


            viewModel.processGalleryImages(
                filenameFormat = "yyyy-MM-dd-HH-mm-ss-SSS",
                outputDirectory = viewModel.getOutputDirectory(context),
                onImageCaptured =
                {
                    if (viewModel.isRetake.value == true) {
                        navController.navigate("previewScreen")
                    } else if (ScannerConfigManager.getConfig()?.mode == ScanMode.SINGLE) {
                        navController.navigate("previewScreen")
                    } else if (ScannerConfigManager.getConfig()?.mode == ScanMode.OCR) {
                        navController.navigate("previewScreen")
                    }
                    //batch cases
                    else if (ScannerConfigManager.getConfig()?.isIdCard == true) {
                        if (ScannerConfigManager.getConfig()?.isBackImageRequired == true) {
                            if (viewModel.capturedImages.value?.size == 2) {
                                navController.navigate("previewScreen")
                            }
                            //back required 1 image click no action needed
                            //back required 2 image route to preview
                        } else {
                            // batch id card back required == false , navigate the user
                            navController.navigate("previewScreen")
                        }
                    }
                },
                screenHeight = screenHeight,
                screenWidth = screenWidth,
                onError = { /* your existing error handling */ },
                corners = null,
                context = context,
                isMultiImage = true,
                selectedUris = uris
            )
        }
    }
    Box(
        modifier = modifier
            .fillMaxWidth()
            .background(
                brush = androidx.compose.ui.graphics.Brush.verticalGradient(
                    colors = listOf(
                        Color(0xFFE2E2E2).copy(alpha = 0f),  // Fully transparent gray at top (0%)
                        Color.Black.copy(alpha = 0.85f)  // 85% opacity black at bottom (100%)
                    )
                )
            )
    ) {
        Column(
            modifier = modifier
                .fillMaxWidth()
                .background(Color.Transparent),
            horizontalAlignment = Alignment.CenterHorizontally,
            verticalArrangement = Arrangement.Bottom,
        ) {
            Spacer(modifier = Modifier.height(16.dp))
            //capture and thumbnail row
            Row(
                modifier = Modifier
                    .fillMaxWidth()
                    .padding(horizontal = 16.dp),
                horizontalArrangement = Arrangement.SpaceBetween,
                verticalAlignment = Alignment.CenterVertically
            ) {

                ControlButton(
                    icon = R.drawable.gallery,
                    onClick = {
                        // Only proceed if the button is enabled
                        if (isGalleryButtonEnabled) {
                            // Disable the button immediately to prevent multiple clicks
                            isGalleryButtonEnabled = false

                            val config = ScannerConfigManager.getConfig()
                            val isRetakeMode = viewModel.isRetake.value == true

                            // Determine which picker to use based on document type and configuration
                            val shouldUseSinglePicker = when {
                                // Case 1: Retake mode always uses single selection
                                isRetakeMode -> true

                                // Case 2: ID card with back image requirement
                                config?.isIdCard == true -> {
                                    if (config.isBackImageRequired) {
                                        // Only allow single selection if we already have one image
                                        capturedImages.size == 1
                                    } else {
                                        // Back image not required, use single picker
                                        true
                                    }
                                }

                                // Case 3: Document type determines selection mode
                                config?.mode == ScanMode.SINGLE -> true
                                config?.mode == ScanMode.BATCH -> false

                                // Default to single picker for safety
                                else -> true
                            }

                            // Launch appropriate picker
                            if (shouldUseSinglePicker) {
                                singleImagePickerLauncher.launch("image/*")
                            } else {
                                multipleImagePickerLauncher.launch(
                                    PickVisualMediaRequest(ActivityResultContracts.PickVisualMedia.ImageOnly)
                                )
                            }

                            // Re-enable the button after a delay to prevent rapid clicks
                            kotlinx.coroutines.MainScope().launch {
                                delay(1000) // 1 second delay
                                isGalleryButtonEnabled = true
                            }
                        }
                    },
                    activeState = false,
                    iconColor = null,
                    backgroundColor = null,
                )
                RingCaptureButton(
                    onClick = {
                        val displayMetrics = context.resources.displayMetrics
                        val screenWidthDouble = displayMetrics.widthPixels.toDouble()
                        val screenHeightDouble = displayMetrics.heightPixels * 0.65
                        viewModel.takePhoto(
                            filenameFormat = "yyyy-MM-dd-HH-mm-ss-SSS",
                            outputDirectory = viewModel.getOutputDirectory(context),
                            onImageCaptured = { capturedUri ->
                                // Start the animation with the captured image URI
                                val config = ScannerConfigManager.getConfig()
                                if (viewModel.isRetake.value != true && ((config?.mode == ScanMode.BATCH && !config.isIdCard) || (config?.isIdCard == true && (viewModel.capturedImages.value?.size
                                        ?: 0) < 2))
                                ) {
                                    viewModel.startCaptureAnimation(capturedUri as Uri)
                                }

                                // Case 1: Handle retake mode
                                if (viewModel.isRetake.value == true) {
                                    navController.navigate("previewScreen")

                                }
                                // Case 2: Handle ID card scanning
                                else if (config?.isIdCard == true) {
                                    // Navigate if back image is not required OR if back image is required and we have both images
                                    if (!config.isBackImageRequired || (viewModel.capturedImages.value?.size == 2)) {
                                        navController.navigate("previewScreen")
                                    }
                                    // Otherwise no action needed (waiting for second image capture)
                                }
                                // Case 3: Handle non-ID card scanning
                                else if (config?.mode == ScanMode.SINGLE || config?.mode == ScanMode.OCR) {
                                    navController.navigate("previewScreen")
                                }
                                // Case 4: All other scenarios (e.g., batch mode)
                                // No navigation needed - continue capturing images
                            },
                            screenHeight = screenHeightDouble,
                            screenWidth = screenWidthDouble,
                            onError = {
                            },
                            corners = null, context = context,
                        )
                    }, isLoading = ScannerConfigManager.getConfig()?.isAutoEnabled == true
                )
                ImagePreviewButton(
                    onClick = {
                        if (!(scannerConfig?.isIdCard == true && scannerConfig.isBackImageRequired && (viewModel.capturedImages.value?.size
                                ?: 0) < 2)
                        ) {
                            navController.navigate("previewScreen")
                        }
                    },
                    imageUri = capturedImages.lastOrNull()?.rawImagePath,
                    imageCount = capturedImages.size,
                    viewModel = viewModel
                )
            }
        }
    }
}

@Composable
fun TopBar(viewModel: CameraViewModel, navController: NavHostController, modifier: Modifier) {
    val buttonStates by viewModel.buttonStates.collectAsState()
    val scannerButtonControl = remember { mutableStateListOf<ScannerButtonControl>() }

    LaunchedEffect(scannerButtonControl) {
        scannerButtonControl.addAll(
            mutableStateListOf(
                ScannerButtonControl(
                    id = "CLOSE", iconRes = R.drawable.backward_arrow,
                ),
                ScannerButtonControl(
                    id = "FLASH", iconRes = R.drawable.flash_off,
                ),
            )
        )
    }
    Box(
        modifier = Modifier
            .fillMaxWidth()
            .background(
                brush = androidx.compose.ui.graphics.Brush.verticalGradient(
                    colors = listOf(
                        Color.Black.copy(alpha = 0.7f),
                        Color.Black.copy(alpha = 0.4f),
                        Color.Black.copy(alpha = 0.0f)
                    )
                )
            )
    ) {


        Row(
            modifier = Modifier
                .fillMaxWidth()
                .background(
                    color = Color.Transparent,
                    shape = RoundedCornerShape(10.dp)
                )
                .padding(horizontal = 16.dp, vertical = 32.dp),
            horizontalArrangement = Arrangement.SpaceBetween,
            verticalAlignment = Alignment.CenterVertically
        ) {
            // First button (assuming this is the first item in the list)
            if (scannerButtonControl.isNotEmpty()) {
                val firstControl = scannerButtonControl.first()
                ControlButton(
                    icon = firstControl.iconRes,
                    onClick = {
                        viewModel.toggleButtonState(
                            firstControl.id,
                            navController
                        )
                    },
                    iconColor = null,
                    backgroundColor = null,
                    activeState = if (buttonStates[0].isToggle) {
                        buttonStates[0].isEnabled
                    } else {
                        false
                    }
                )
            }

            // Text in the middle with overflow handling
            Text(
                modifier = Modifier
                    .weight(1f)
                    .padding(horizontal = 8.dp),
                text = viewModel.documentName,
                fontSize = 16.sp,
                color = Color(
                    color = ContextCompat.getColor(
                        LocalContext.current,
                        R.color.cSelectedPreviewColor
                    ),
                ),
                textAlign = TextAlign.Center,
                style = MaterialTheme.typography.titleMedium,
                maxLines = 1,
                overflow = TextOverflow.Ellipsis
            )

            // Last button (assuming this is the last item in the list)
            if (scannerButtonControl.size > 1) {
                val lastControl = scannerButtonControl.last()
                val lastIndex = scannerButtonControl.size - 1
                ControlButton(
                    icon = lastControl.iconRes,
                    onClick = {
                        viewModel.toggleButtonState(
                            lastControl.id,
                            navController
                        )
                    },
                    activeIcon = if (lastControl.id == "FLASH") R.drawable.flash else null,
                    iconColor = null,
                    backgroundColor = null,
                    activeState = if (buttonStates[lastIndex].isToggle) {
                        buttonStates[lastIndex].isEnabled
                    } else {
                        false
                    }
                )
            }
        }
    }

}

@Composable
fun ImagePreviewButton(
    modifier: Modifier = Modifier,
    onClick: () -> Unit,
    imageUri: Uri?,
    imageCount: Int,
    viewModel: CameraViewModel,
) {
    val containerHeight = 36.dp
    val containerWidth = 36.dp
    val scannerConfig = ScannerConfigManager.getConfig()

    val isRetake by viewModel.isRetake.collectAsState()
    if (isRetake == true) Spacer(
        modifier = Modifier
            .height(containerHeight)
            .width(containerWidth)
    )
    else {
        if (imageCount == 0) {
            Box(
                modifier = Modifier
                    .height(containerHeight)
                    .width(containerWidth)
                    .padding(end = 10.dp)
                    .background(color = Color.Transparent)
            )
        } else {
            Box(
                modifier = modifier.clickable { onClick() }, contentAlignment = Alignment.Center
            ) {
                imageUri?.let {
                    Image(
                        painter = rememberAsyncImagePainter(it),
                        contentDescription = null,
                        modifier = Modifier
                            .height(containerHeight)
                            .width(containerWidth)
                            .border(
                                width = 2.dp,
                                color = Color(ScannerConfigManager.getConfig()!!.primaryColor)
                            )
                    )
                } ?: Box(
                    modifier = Modifier
                        .height(containerHeight)
                        .width(containerWidth)
                        .background(Color.Black)
                )
                Box(
                    contentAlignment = Alignment.Center,
                    modifier = Modifier
                        .align(Alignment.TopEnd)
                        .offset(
                            x = 5.dp, y = (-10).dp
                        ) // Shifted 40.dp to the right, outside the box
                        .size(if (imageCount >= 100) 22.dp else 18.dp) // Size of the circle
                        .background(
                            Color(ScannerConfigManager.getConfig()!!.primaryColor), CircleShape
                        ) // Blue circle
                ) {
                    Text(
                        text = imageCount.toString(),
                        color =
                        if (scannerConfig != null) Color(scannerConfig.primaryTextColor) else Color(
                            ContextCompat.getColor(LocalContext.current, R.color.textColorPrimary)
                        ),

                        fontSize = 10.sp,
                        textAlign = TextAlign.Center
                    )
                }

            }
        }
    }
}

@Composable
fun ControlButton(
    icon: Int,
    onClick: () -> Unit,
    activeState: Boolean,
    iconColor: Color?,
    activeIcon: Int? = null,
    backgroundColor: Color?,
) {
    IconButton(
        onClick = onClick,
        Modifier
            .clip(CircleShape)
            .size(40.dp)
            .background(
                color = backgroundColor ?: colorResource(
                    R.color.colorPrimary
                )
            )
    ) {
        if (activeState)
            if (activeIcon != null)
                Icon(
                    painter = painterResource(activeIcon),
                    contentDescription = null,
                    tint = colorResource(id = R.color.scannerActiveIconColor)
                )
            else
                Icon(
                    painter = painterResource(icon),
                    contentDescription = null,
                    tint =
                    colorResource(id = R.color.scannerActiveIconColor)

                )
        else
            Icon(
                painter = painterResource(icon),
                contentDescription = null,
                tint =
                iconColor ?: Color.White
            )
    }
}

@Composable
fun RingCaptureButton(
    modifier: Modifier = Modifier,
    onClick: () -> Unit,
    isLoading: Boolean,
    primaryColor: Color = Color(0xFF2196F3) // Default to blue, replace with your app's primary color
) {
    Box(
        contentAlignment = Alignment.Center,
        modifier = modifier.size(80.dp)
    ) {
        // Ring around the button
        Box(
            modifier = Modifier
                .size(64.dp)
                .border(
                    width = 3.dp,
                    color = Color.White,
                    shape = CircleShape
                )
        )

        // The main button
        Button(
            onClick = onClick,
            modifier = Modifier.size(54.dp),

            colors = ButtonDefaults.buttonColors(containerColor = Color.White),
            shape = CircleShape,
            contentPadding = PaddingValues(0.dp)
        ) {
            //empty button content
        }

        // Loading indicator when isLoading is true
        if (isLoading) {
            CircularProgressIndicator(
                modifier = Modifier
                    .align(Alignment.Center)
                    .size(70.dp),
                color = primaryColor,
                strokeWidth = 4.dp
            )
        }
    }
}

@Composable
fun RingButton(
    onClick: () -> Unit,
    modifier: Modifier = Modifier,
    buttonSize: Dp = 48.dp,
    ringSize: Dp = 64.dp,
    buttonContent: @Composable () -> Unit
) {
    val interactionSource = remember { MutableInteractionSource() }
    val isPressed by interactionSource.collectIsPressedAsState()
    val isHovered by interactionSource.collectIsHoveredAsState()

    // Animate shadow elevation based on press/hover state
    val elevation by animateFloatAsState(
        targetValue = when {
            isPressed -> 1f
            isHovered -> 6f
            else -> 4f
        }, label = "elevation"
    )

    // Ring color based on interaction state
    val ringColor = when {
        isPressed -> Color.Gray.copy(alpha = 0.5f)
        isHovered -> Color.Gray.copy(alpha = 0.3f)
        else -> Color.LightGray.copy(alpha = 0.3f)
    }

    Box(
        modifier = modifier,
        contentAlignment = Alignment.Center
    ) {
        // Outer ring
        Box(
            modifier = Modifier
                .size(ringSize)
                .border(
                    width = 2.dp,
                    color = ringColor,
                    shape = CircleShape
                )
        )

        // Button
        Card(
            modifier = Modifier
                .size(buttonSize)
                .clip(CircleShape)
                .shadow(elevation.dp)
                .clickable(
                    interactionSource = interactionSource,
                    indication = null,
                    onClick = onClick
                ),
            shape = CircleShape,
            colors = CardDefaults.cardColors(
                containerColor = Color.White
            )
        ) {
            Box(
                modifier = Modifier.fillMaxSize(),
                contentAlignment = Alignment.Center
            ) {
                buttonContent()
            }
        }
    }
}

@Composable
fun OverlayDrawView(
    drawView: DrawView
) {
    val scannerConfig = ScannerConfigManager.getConfig()
    if (scannerConfig != null && scannerConfig.isAutoEnabled) AndroidView(
        factory = {
            drawView.apply {
                initializeForCameraScreen()
                resetCornerPoints()
            }
        },

        modifier = Modifier.fillMaxSize()

    )
}

@Composable
private fun getAutoPromptText(cornerPointStatus: CornerPointStatus?): String? {
    when (cornerPointStatus) {
        CornerPointStatus.CAPTURING -> {
            return stringResource(R.string.detectedCornerStateLabel)
        }

        CornerPointStatus.LOW_CONF_SCORE -> {

            return stringResource(R.string.moveCameraCloserCornerStateLabel)
        }

        CornerPointStatus.FINDING -> {
            return stringResource(R.string.detectingCornerStateLabel)
        }

        CornerPointStatus.NOT_FOUND -> {

            return stringResource(R.string.noCornerFoundCornerStateLabel)
        }

        else -> return null
    }
}

private fun copyPointsArray(points: Array<PointF>): Array<PointF> {
    val copiedArray = Array(points.size) { PointF(0f, 0f) }
    for (i in points.indices) {
        copiedArray[i].x = points[i].x
        copiedArray[i].y = points[i].y
    }
    return copiedArray
}

@Composable
fun GridLineOverlay(modifier: Modifier = Modifier) {
    Canvas(modifier = modifier) {
        val width = size.width
        val height = size.height
        val lineColor = Color.White
        val strokeWidth = 3f

        // Vertical lines
        drawLine(
            color = lineColor,
            start = Offset(x = width / 3f, y = 0f),
            end = Offset(x = width / 3f, y = height),
            strokeWidth = strokeWidth
        )
        drawLine(
            color = lineColor,
            start = Offset(x = (width / 3) * 2f, y = 0f),
            end = Offset(x = (width / 3) * 2f, y = height),
            strokeWidth = strokeWidth
        )

        // Horizontal lines
        drawLine(
            color = lineColor,
            start = Offset(x = 0f, y = height / 3f),
            end = Offset(x = width, y = height / 3f),
            strokeWidth = strokeWidth
        )
        drawLine(
            color = lineColor,
            start = Offset(x = 0f, y = (height / 3) * 2f),
            end = Offset(x = width, y = (height / 3) * 2f),
            strokeWidth = strokeWidth
        )
    }
}
