package one.zoop.sdk.scanner.viewmodel

import OcrRepository
import android.app.Application
import android.content.ContentValues.TAG
import android.content.Context
import android.content.res.AssetManager
import android.graphics.Bitmap
import android.graphics.BitmapFactory
import android.graphics.Canvas
import android.graphics.Matrix
import android.graphics.Paint
import android.graphics.PointF
import android.icu.text.SimpleDateFormat
import android.net.Uri
import android.os.Build
import android.util.Base64
import android.util.Log
import android.util.Size
import androidx.annotation.OptIn
import androidx.camera.core.CameraControl
import androidx.camera.core.CameraInfo
import androidx.camera.core.ExperimentalGetImage
import androidx.camera.core.FocusMeteringAction
import androidx.camera.core.ImageAnalysis
import androidx.camera.core.ImageCapture
import androidx.camera.core.ImageCaptureException
import androidx.camera.core.ImageProxy
import androidx.camera.view.PreviewView
import androidx.compose.runtime.mutableStateOf
import androidx.core.content.ContextCompat
import androidx.exifinterface.media.ExifInterface
import androidx.lifecycle.AndroidViewModel
import androidx.lifecycle.LiveData
import androidx.lifecycle.MutableLiveData
import androidx.lifecycle.viewModelScope
import androidx.navigation.NavHostController
import io.sentry.Sentry
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.StateFlow
import kotlinx.coroutines.flow.asStateFlow
import kotlinx.coroutines.launch
import kotlinx.coroutines.withContext
import one.zoop.sdk.scanner.R
import one.zoop.sdk.scanner.data.models.CardInfo
import one.zoop.sdk.scanner.data.models.DocumentResult
import one.zoop.sdk.scanner.data.models.toMap
import one.zoop.sdk.scanner.model.ButtonState
import one.zoop.sdk.scanner.model.DialogContent
import one.zoop.sdk.scanner.model.DrawViewState
import one.zoop.sdk.scanner.model.FileType
import one.zoop.sdk.scanner.model.OcrMode
import one.zoop.sdk.scanner.model.ScanMode
import one.zoop.sdk.scanner.model.ScannedPage
import one.zoop.sdk.scanner.model.ScannerResultCallback
import one.zoop.sdk.scanner.ui.layout.Preview.getBitmapFromUri
import one.zoop.sdk.scanner.utility.TfliteHelper
import one.zoop.sdk.scanner.utils.ImageProcessor.scaleImageAccordingToAspectRatio2
import one.zoop.sdk.scanner.utils.ScannerConfigManager
import org.tensorflow.lite.DataType
import org.tensorflow.lite.Interpreter
import org.tensorflow.lite.support.image.ImageProcessor
import org.tensorflow.lite.support.image.TensorImage
import org.tensorflow.lite.support.image.ops.ResizeOp
import org.tensorflow.lite.support.image.ops.Rot90Op
import java.io.ByteArrayOutputStream
import java.io.File
import java.io.FileInputStream
import java.io.FileOutputStream
import java.io.IOException
import java.nio.MappedByteBuffer
import java.nio.channels.FileChannel
import java.time.Instant
import java.util.Date
import java.util.Locale
import kotlin.math.abs

enum class CornerPointStatus {
    FINDING, DETECTED, CAPTURING, LOW_CONF_SCORE, NOT_FOUND
}


class CameraViewModel(application: Application) : AndroidViewModel(application) {

    private val repository: OcrRepository = OcrRepository()
    private lateinit var imageCapture: ImageCapture
    private lateinit var imageAnalyzer: ImageAnalysis
    private val _capturedImages = MutableLiveData<List<ScannedPage>>()
    private var scannerResultCallback: ScannerResultCallback? = null
    var rotateDrawViewCallback: ((Float) -> Unit)? = null
    val capturedImages: LiveData<List<ScannedPage>> = _capturedImages
    private lateinit var interpreter: Interpreter
    private lateinit var tfInputSize: Size
    var aspectRatio: String? = null
    private val _isRetake = MutableStateFlow<Boolean?>(null)
    var isRetake = _isRetake.asStateFlow()
    var indexToReplace: Int? = null
    private var _autoCapturePromptState: MutableStateFlow<CornerPointStatus?> =
        MutableStateFlow(null)
    var autoCapturePromptState: StateFlow<CornerPointStatus?> = _autoCapturePromptState
    private lateinit var mCameraControl: CameraControl
    private lateinit var mCameraInfo: CameraInfo
    var isAutoCaptureTimeout: Boolean = false
    private val drawViewStates: MutableMap<Int, DrawViewState> = mutableMapOf()
    private var _isDeleteInProgress: MutableStateFlow<Boolean> =
        MutableStateFlow(false)
    val isDeleteInProgress: StateFlow<Boolean> = _isDeleteInProgress

    private val _showCaptureAnimation = mutableStateOf(false)
    val showCaptureAnimation = _showCaptureAnimation

    private val _animationImageUri = mutableStateOf<Uri?>(null)
    val animationImageUri = _animationImageUri

    // Method to trigger animation
    fun startCaptureAnimation(imageUri: Uri) {
        _animationImageUri.value = imageUri
        _showCaptureAnimation.value = true
    }

    fun endCaptureAnimation() {
        _showCaptureAnimation.value = false
    }


    private val _scannerButtonStates = MutableStateFlow(
        listOf(
            ButtonState("CLOSE", false, isToggle = false),
            ButtonState("FLASH", false, isToggle = true),
        )
    )

    val dialogContent = mutableStateOf<DialogContent?>(null)
    lateinit var documentName: String
    val tfliteHelper: TfliteHelper = TfliteHelper(application);
    val buttonStates: StateFlow<List<ButtonState>> = _scannerButtonStates

    fun toggleButtonState(id: String, navController: NavHostController) {
        viewModelScope.launch {
            _scannerButtonStates.value = _scannerButtonStates.value.map { buttonState ->
                if (buttonState.id == id) {
                    if (buttonState.id == "CLOSE") {
                        if (isRetake.value == true) {
                            navController.navigate("previewScreen")
                            setRetakeValue(false)
                        } else {
                            dialogContent.value = DialogContent.AbortScaDialog
                        }
                    }
                    if (buttonState.isToggle) {
                        buttonState.copy(isEnabled = !buttonState.isEnabled)
                    } else {
                        buttonState
                    }
                } else {
                    buttonState
                }

            }
        }
    }


//    private lateinit var warpTransform: WarpTransform

    companion object {
        private const val ACCURACY_THRESHOLD = 0.5f
        private const val NO_OF_THREADS = 4
        private const val MODEL_NAME = "keypoint_320_256_float16_V5_1_output.tflite"
        private const val REQUEST_CODE_GALLERY = 1
    }

    init {
//        viewModelScope.launch(Dispatchers.IO) {
//            // Load the TensorFlow Lite model
//            interpreter = Interpreter(
//                loadModelFile(
//                    application.assets, MODEL_NAME
//                )
//            )
//
//            Interpreter.Options().setNumThreads(NO_OF_THREADS)
//        }
        _isRetake.value = false
    }

    fun saveDrawViewState(pageIndex: Int, state: DrawViewState) {
        drawViewStates[pageIndex] = state
    }

    fun getDrawViewState(pageIndex: Int): DrawViewState? {
        return drawViewStates[pageIndex]
    }

    fun setScanName(documentName: String?) {
        this.documentName = documentName ?: generateName()
    }

    fun getImageAnalyzer(): ImageAnalysis {
        return imageAnalyzer
    }

    fun setCameraController(cameraControl: CameraControl) {
        mCameraControl = cameraControl
    }

    fun setCameraInfo(cameraInfo: CameraInfo) {
        mCameraInfo = cameraInfo
    }

    private fun turnFlashOn() {
        mCameraControl.enableTorch(true)
    }

    fun changeAutoCapturePromptState(cornerPointStatus: CornerPointStatus?) {
        _autoCapturePromptState.value = cornerPointStatus
    }

    private fun turnFlashOff() {
        mCameraControl.enableTorch(false)
    }

    fun setRetakeValue(isRetake: Boolean) {
        _isRetake.value = isRetake
    }

    fun addScannedPage(newScannedPage: ScannedPage) {
        val currentList = _capturedImages.value.orEmpty().toMutableList()
        currentList.add(newScannedPage)
        _capturedImages.value = currentList
    }

    fun deleteAllScannedPage() {
        _capturedImages.value = emptyList()
    }

    private fun generateName(): String {
        // Define the date pattern
        val dateFormat = SimpleDateFormat("dd-MM-yyyy - HH:mm", Locale.getDefault())

        // Get the current date and time
        val currentDateAndTime: String = dateFormat.format(Date())
        // Concatenate with the desired prefix
            return "ZoopDoc - $currentDateAndTime"
    }

    fun setCallback(scannerResultCallback: ScannerResultCallback) {
        this.scannerResultCallback = scannerResultCallback
    }

    fun updateCroppedPage(correctedPoints: List<PointF>?, processedPath: String, index: Int) {
        val allScannedPages = _capturedImages.value ?: return
        if (index !in allScannedPages.indices) return //O(1)
        val currPage = allScannedPages[index]
        val updatedPage =
            currPage.copy(correctedCorners = correctedPoints, processedImagePath = processedPath)
        val updatedPages = allScannedPages.toMutableList().apply {
            set(index, updatedPage)
        }
        _capturedImages.value = updatedPages

    }

    fun replaceScannedPage(
        newScannedPage: ScannedPage,
        indexToReplace: Int,
        isRetake: Boolean = false
    ) {
        val currentList = _capturedImages.value.orEmpty().toMutableList()
        val oldPage = currentList[indexToReplace]
        currentList[indexToReplace] = newScannedPage
        _capturedImages.value = currentList
        if (isRetake) {
            deleteScannedPageFilePaths(oldPage)

        }
    }

    fun rotateScannedPage(rotation: Float, index: Int) {
        val allScannedPages = _capturedImages.value ?: return
        if (index !in allScannedPages.indices) return //O(1)
        val currPage = allScannedPages[index]
        val updatedPage = currPage.copy(imageRotate = rotation + currPage.imageRotate)
        val updatedPages = allScannedPages.toMutableList().apply {
            set(index, updatedPage)
        }
        _capturedImages.value = updatedPages
    }

    fun deleteScannedPage(index: Int) {
        if (!_isDeleteInProgress.value) {
            _isDeleteInProgress.value = true
            val currentList = _capturedImages.value.orEmpty().toMutableList()
            if (currentList.size > index) {
                val scannedPageToDelete = currentList[index]
                currentList.removeAt(index)
                deleteScannedPageFilePaths(scannedPageToDelete)
            }
            _capturedImages.value = currentList
            _isDeleteInProgress.value = false

        }
    }


    private fun flushBitmap(bitmap: Bitmap) {
        bitmap.recycle()
    }

    private fun deleteScannedPageFilePaths(scannedPage: ScannedPage) {
        val rawFileUriToDelete = scannedPage.rawImagePath
        val processedFilePathToDelete = scannedPage.processedImagePath

        val res1 = deleteImageFromUri(uri = rawFileUriToDelete)
        val res2 = deleteImageFromFilePath(filePath = processedFilePathToDelete)
    }


    fun getExifRotationFromImagePath(imagePath: String): Int {
        val exif = ExifInterface(imagePath)
        return when (exif.getAttributeInt(
            ExifInterface.TAG_ORIENTATION, ExifInterface.ORIENTATION_NORMAL
        )) {
            ExifInterface.ORIENTATION_ROTATE_90 -> 90
            ExifInterface.ORIENTATION_ROTATE_180 -> 180
            ExifInterface.ORIENTATION_ROTATE_270 -> 270
            else -> 0
        }
    }

    fun getCurrentScannedPage(index: Int): ScannedPage? {
        if (!capturedImages.value.isNullOrEmpty() && capturedImages.value!!.size > index) {
            return capturedImages.value!![index]
        }
        return null
    }

    fun addCroppedImageToScannedPage(uri: Uri, processedUri: String, corners: List<PointF>) {
        val currentList = _capturedImages.value.orEmpty().toMutableList()
        val pageIndex = currentList.indexOfFirst { it.rawImagePath == uri }
        if (pageIndex != -1) {
            currentList[pageIndex] = currentList[pageIndex].copy(
                processedImagePath = processedUri, detectedCorners = corners,
            )
        }
        _capturedImages.value = currentList
    }


    fun reRunInference(
        uri: Uri,
        rotatedBitmap: Bitmap,
        context: Context,
        autoCaptureCorners: Array<PointF>?,
        onCornersDetected: (List<PointF>, String) -> Unit,
    ) {
        viewModelScope.launch(Dispatchers.IO) {
            val width = rotatedBitmap.width
            val height = rotatedBitmap.height

            // Define default corners with margin of 100px from edges
            val defaultCorners = arrayOf(
                PointF(100f / width, 100f / height),                          // Top-left corner
                PointF(100f / width, (height.toFloat() - 100) / height),      // Bottom-left corner
                PointF((width.toFloat() - 100) / width, (height.toFloat() - 100) / height), // Bottom-right corner
                PointF((width.toFloat() - 100) / width, 100f / height)        // Top-right corner
            )

            val corners: Array<PointF>
            val score: Float?

            if (autoCaptureCorners == null) {
                val result = tfliteHelper.runInference(rotatedBitmap, 0)
                var detectedCorners = result["corner_points"] as Array<*>?

                if (detectedCorners.isNullOrEmpty()) {
                    corners = defaultCorners
                } else {
                    // Check if any detected corner is outside the image bounds
                    val tempCorners = detectedCorners as Array<PointF>
                    val isOutOfBounds = tempCorners.any { point ->
                        point.x < 0f || point.x > 1f || point.y < 0f || point.y > 1f
                    }

                    // If any corner is out of bounds, use default corners
                    corners = if (isOutOfBounds) {
                        Log.d("CornerDetection", "Detected corners outside image bounds, using defaults")
                        defaultCorners
                    } else {
                        tempCorners
                    }
                }

                score = result["conf_score"] as Float
                Log.d("CornerDetection", "Confidence score: $score")
            } else {
                // Check if any provided corners are outside the image bounds
                val isOutOfBounds = autoCaptureCorners.any { point ->
                    point.x < 0f || point.x > 1f || point.y < 0f || point.y > 1f
                }

                // If any corner is out of bounds, use default corners
                corners = if (isOutOfBounds) {
                    Log.d("CornerDetection", "Provided corners outside image bounds, using defaults")
                    defaultCorners
                } else {
                    autoCaptureCorners
                }
            }

            // Scale points to the image dimensions
            val tfliteModelCorner1 = PointF(corners[0].x * width, corners[0].y * height) // Top-left
            val tfliteModelCorner2 = PointF(corners[3].x * width, corners[3].y * height) // Top-right
            val tfliteModelCorner3 = PointF(corners[2].x * width, corners[2].y * height) // Bottom-right
            val tfliteModelCorner4 = PointF(corners[1].x * width, corners[1].y * height) // Bottom-left

            val srcPoints = arrayOf(
                tfliteModelCorner1,
                tfliteModelCorner2,
                tfliteModelCorner3,
                tfliteModelCorner4
            )

            // Define the destination points for perspective transformation
            val transformedPath = initiateCrop(
                uri,
                imageHeight = height.toDouble(),
                imageWidth = width.toDouble(),
                srcPoints,
                context
            )

            withContext(Dispatchers.Main) {
                onCornersDetected(corners.toList(), transformedPath)
            }
        }
    }

    fun getFilePath(): String {
        return if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.O) {
            val scanFileType = ScannerConfigManager.getConfig()?.fileTypes
            File(
                getApplication<Application>().filesDir,
                Date.from(Instant.now())
                    .toString() + if (scanFileType == FileType.JPEG) ".jpg" else ".png"
            ).absolutePath
        } else {
            // For SDK versions lower than O
            val dateFormatter = SimpleDateFormat("yyyyMMdd_HHmmss", Locale.US)
            val date = Date()
            File(getApplication<Application>().cacheDir, dateFormatter.format(date)).absolutePath
        }
    }

    private fun processImagePayload(): ArrayList<String> {
        val imagePathsToSend: ArrayList<String> = ArrayList()
        for (scannedPages in capturedImages.value!!) {
            val isSuccess = saveRotatedImageToFile(
                scannedPages.processedImagePath ?: scannedPages.rawImagePath.path!!,
                rotationAngle = scannedPages.imageRotate,
                outputFilePath = scannedPages.processedImagePath ?: scannedPages.rawImagePath.path!!
            );
            if (isSuccess) {
                imagePathsToSend.add(
                    scannedPages.processedImagePath ?: scannedPages.rawImagePath.path.toString()
                )
            }

        }
        return imagePathsToSend
    }

    fun saveBtnCallback(onFinish: () -> Unit) {
        val imagePathsToSend = processImagePayload()
        val scanMode = ScannerConfigManager.getConfig()?.mode
        val ocrMode = ScannerConfigManager.getConfig()?.ocrMode

        val imageFrontPath = imagePathsToSend.getOrNull(0)
        val imageBackPath = imagePathsToSend.getOrNull(1)

        if (imageFrontPath == null) {
            onFinish()
            scannerResultCallback?.onFailure("100", "No Image Scanned for OCR")
            return
        }

        val response: MutableMap<String, Any?> = mutableMapOf("imagePath" to imagePathsToSend)
        scannerResultCallback?.onSuccess(response)
        //remove for ocr at btnClick
        return
        viewModelScope.launch(Dispatchers.IO) {
            var ocrResponse: Map<String, Any?>? = null
            if (scanMode == ScanMode.OCR && ocrMode != null) {
                ocrResponse = processOcrRequest(imageFrontPath, imageBackPath, ocrMode)
            }
            ocrResponse?.let { response.putAll(it) }
            withContext(Dispatchers.Main) {
                onFinish()
                if (response["error"] == null) {
                    scannerResultCallback?.onSuccess(response)
                } else {
                    scannerResultCallback?.onFailure("100", response["error"].toString())
                }
            }
        }
    }

    fun sendErrorToActivity(errorCode: String, errorMessage: String) {
        scannerResultCallback?.onFailure(errorCode = errorCode, errorMessage = errorMessage)

    }


    fun getImageCapture(): ImageCapture {
        return imageCapture
    }

    fun saveBitmapAsPng(bitmap: Bitmap, file: File): Boolean {
        return try {
            FileOutputStream(file).use { out ->
                bitmap.compress(Bitmap.CompressFormat.PNG, 100, out)
                out.flush()
            }
            true
        } catch (e: IOException) {
            Sentry.captureException(e);
            e.printStackTrace()
            false
        }
    }

    fun setImageCapture(imageCapture: ImageCapture) {
        this.imageCapture = imageCapture
    }

    fun setImageAnalyzer(imageAnalyzer: ImageAnalysis) {
        this.imageAnalyzer = imageAnalyzer
    }

    fun onZoomChange(zoomFactor: Float) {
        mCameraControl.apply {
            val currentZoomRatio = mCameraInfo.zoomState.value?.zoomRatio ?: 1f
            val newZoomRatio = currentZoomRatio * zoomFactor
            setZoomRatio(newZoomRatio)
        }
    }

    fun onFocus(x: Float, y: Float, previewView: PreviewView) {
        mCameraControl.let { control ->
            val factory = previewView.meteringPointFactory
            val point = factory.createPoint(x, y)
            val action = FocusMeteringAction.Builder(point).build()
            control.startFocusAndMetering(action)
        }
    }

    fun processGalleryImages(
        selectedUris: List<Uri>,
        isMultiImage: Boolean,
        filenameFormat: String,
        outputDirectory: File,
        onImageCaptured: () -> Unit,
        onError: (ImageCaptureException) -> Unit,
        screenHeight: Double,
        screenWidth: Double,
        corners: Array<PointF>?,
        context: Context
    ) {
        viewModelScope.launch(Dispatchers.Main) {
            val urisToProcess =
                if (isMultiImage) selectedUris else listOfNotNull(selectedUris.firstOrNull())

            // Process images sequentially to maintain order
            for ((index, selectedUri) in urisToProcess.withIndex()) {
                try {
                    val scanFileType = ScannerConfigManager.getConfig()?.fileTypes
                    val photoFile = File(
                        outputDirectory,
                        SimpleDateFormat(
                            filenameFormat,
                            Locale.US
                        ).format(System.currentTimeMillis() + index) + // Add index to ensure unique names
                                if (scanFileType == FileType.JPEG) ".jpg" else ".png"
                    )

                    val bitmap = withContext(Dispatchers.IO) {
                        loadBitmapWithRotation(context, selectedUri)
                            ?: throw Exception("Failed to load bitmap from URI")
                    }

                    withContext(Dispatchers.IO) {
                        FileOutputStream(photoFile).use { outputStream ->
                            bitmap.compress(
                                Bitmap.CompressFormat.JPEG,
                                if (_scannerButtonStates.value.size > 2 && _scannerButtonStates.value[2].isEnabled)
                                    100 else ScannerConfigManager.getConfig()?.quality ?: 100,
                                outputStream
                            )
                            outputStream.flush()
                        }
                    }

                    val savedUri = Uri.fromFile(photoFile)
                    val initialScannedPage = ScannedPage(rawImagePath = savedUri)

                    // Handle page addition or replacement in the main thread
                    if (!isMultiImage && isRetake.value == true && indexToReplace != null) {
                        replaceScannedPage(initialScannedPage, indexToReplace!!, isRetake = true)
                    } else {
                        addScannedPage(initialScannedPage)
                    }

                    val currentScannedPageIndex =
                        if (isRetake.value == true) indexToReplace
                        else _capturedImages.value?.size?.minus(1)

                    // For first image, notify immediately to update UI
                    if (index == 0) {
                        onImageCaptured()
                    }

                    // Process each image in sequence
                    try {
                        val result = withContext(Dispatchers.IO) {
                            scaleImageAccordingToAspectRatio2(
                                bitmap.height.toDouble(),
                                bitmap.width.toDouble(),
                                screenHeight,
                                screenWidth
                            )
                        }

                        val newHeight = result[0]
                        val newWidth = result[1]
                        val scaleFactorHeight = result[2]
                        val scaleFactorWidth = result[3]

                        val newScannedPage = ScannedPage(
                            rawImagePath = savedUri,
                            imageHeight = bitmap.height.toDouble(),
                            imageWidth = bitmap.width.toDouble(),
                            relativeImageWidth = newWidth,
                            relativeImageHeight = newHeight,
                            scaleFactorWidth = scaleFactorWidth,
                            scaleFactorHeight = scaleFactorHeight,
                        )

                        // Update the page with the scaled information
                        if (!isMultiImage && isRetake.value == true && indexToReplace != null) {
                            replaceScannedPage(newScannedPage, indexToReplace!!)
                            indexToReplace = null
                            setRetakeValue(false)
                        } else {
                            replaceScannedPage(newScannedPage, currentScannedPageIndex ?: 0)
                        }

                        val adjustedCorners = corners?.let {
                            adjustCorners(it, scaleFactorWidth, scaleFactorHeight)
                        }

                        // Process inference and get cropped image, ensuring we wait for completion
                        val pageIndex = currentScannedPageIndex ?: 0
                        withContext(Dispatchers.IO) {
                            reRunInference(
                                savedUri,
                                bitmap,
                                context,
                                adjustedCorners
                            ) { updatedCorners, croppedPath ->
                                // Ensure we associate the corners with the correct page
                                addCroppedImageToScannedPage(savedUri, croppedPath, updatedCorners)
                            }
                        }

                    } catch (e: Exception) {
                        Sentry.captureException(e)
                        Log.e("CameraViewModel", "Exception during image processing:", e)
                        onError(
                            ImageCaptureException(
                                ImageCapture.ERROR_UNKNOWN,
                                "Exception during image processing",
                                e
                            )
                        )
                    }
                } catch (e: Exception) {
                    Sentry.captureException(e)
                    Log.e("CameraViewModel", "Exception during gallery image processing:", e)
                    onError(
                        ImageCaptureException(
                            ImageCapture.ERROR_UNKNOWN,
                            "Exception during gallery image processing",
                            e
                        )
                    )
                }
            }

            // If multiple images were processed, ensure navigation happens after all are done
            if (urisToProcess.size > 1) {
                onImageCaptured()
            }
        }
    }

    fun adjustCorners(
        corners: Array<PointF>,
        scaleFactorWidth: Double,
        scaleFactorHeight: Double
    ): Array<PointF> {
        return corners.map { point ->
            PointF(
                (point.x * scaleFactorWidth).toFloat(),
                (point.y * scaleFactorHeight).toFloat()
            )
        }.toTypedArray()
    }

    private fun loadBitmapWithRotation(context: Context, uri: Uri): Bitmap? {
        return try {
            // Get EXIF orientation
            val inputStream = context.contentResolver.openInputStream(uri)
            val exif = inputStream?.let { ExifInterface(it) }
            val orientation = exif?.getAttributeInt(
                ExifInterface.TAG_ORIENTATION,
                ExifInterface.ORIENTATION_NORMAL
            )
            inputStream?.close()

            // Load bitmap
            val bitmap = context.contentResolver.openInputStream(uri).use { stream ->
                BitmapFactory.decodeStream(stream)
            }

            // Apply rotation if necessary
            val rotation = when (orientation) {
                ExifInterface.ORIENTATION_ROTATE_90 -> 90f
                ExifInterface.ORIENTATION_ROTATE_180 -> 180f
                ExifInterface.ORIENTATION_ROTATE_270 -> 270f
                else -> 0f
            }
            if (rotation != 0f) {
                val matrix = Matrix().apply { postRotate(rotation) }
                Bitmap.createBitmap(bitmap, 0, 0, bitmap.width, bitmap.height, matrix, true)
            } else {
                bitmap
            }
        } catch (e: Exception) {
            Log.e("CameraViewModel", "Error loading bitmap with rotation:", e)
            null
        }
    }

    fun takePhoto(
        filenameFormat: String,
        outputDirectory: File,
        onImageCaptured: (Any?) -> Unit,
        onError: (ImageCaptureException) -> Unit,
        screenHeight: Double,
        screenWidth: Double,
        corners: Array<PointF>?,
        context: Context
    ) {
        val scanFileType = ScannerConfigManager.getConfig()?.fileTypes
        val photoFile = File(
            outputDirectory,
            SimpleDateFormat(
                filenameFormat,
                Locale.US
            ).format(System.currentTimeMillis()) + if (scanFileType == FileType.JPEG) ".jpg" else ".png"
        )

        val outputOptions = ImageCapture.OutputFileOptions.Builder(photoFile).build()

        //TODO : require refactoring the saving ,model detection ,run inference code here
        viewModelScope.launch(Dispatchers.Main) {
            try {
                if ( _scannerButtonStates.value[1].id == "FLASH") {
                    if (_scannerButtonStates.value[1].isEnabled) {
                        turnFlashOn()
                    }
                }
                imageCapture.takePicture(
                    outputOptions,
                    ContextCompat.getMainExecutor(context),
                    object : ImageCapture.OnImageSavedCallback {
                        override fun onError(exception: ImageCaptureException) {
                            Log.e("CameraViewModel", "Take photo error:", exception)
                            onError(exception)
                        }

                        override fun onImageSaved(outputFileResults: ImageCapture.OutputFileResults) {
                            val savedUri = Uri.fromFile(photoFile)

                            val initialScannedPage = ScannedPage(rawImagePath = savedUri)
                            if (isRetake.value == true && indexToReplace != null) {
                                replaceScannedPage(
                                    initialScannedPage,
                                    indexToReplace!!,
                                    isRetake = true
                                )
                            } else {
                                addScannedPage(initialScannedPage)
                            }
                            val currentScannedPageIndex =
                                if (isRetake.value == true) indexToReplace else _capturedImages.value?.size?.minus(
                                    1
                                )
                            if ( _scannerButtonStates.value[1].id == "FLASH") {
                                if (_scannerButtonStates.value[1].isEnabled) {
                                    turnFlashOff()
                                }
                            }
                            onImageCaptured(savedUri)
                            // Process image in the background
                            viewModelScope.launch(Dispatchers.IO) {
                                try {
                                    val bitmap = getBitmapFromUri(context, savedUri)
                                    if (bitmap != null) {
                                        val rotation = getExifRotationFromImagePath(photoFile.path)
                                        val matrix =
                                            Matrix().apply { postRotate(rotation.toFloat()) }
                                        val rotatedBitmap = Bitmap.createBitmap(
                                            bitmap, 0, 0, bitmap.width, bitmap.height, matrix, true
                                        )
                                        FileOutputStream(photoFile).use { outputStream ->
                                            rotatedBitmap.compress(
                                                Bitmap.CompressFormat.JPEG,
                                                if (_scannerButtonStates.value.size > 2 && _scannerButtonStates.value[2].isEnabled) 100 else ScannerConfigManager.getConfig()?.quality
                                                    ?: 100,
                                                outputStream
                                            )
                                            outputStream.flush()
                                            outputStream.close()
                                        }
                                        val result = scaleImageAccordingToAspectRatio2(
                                            rotatedBitmap.height.toDouble(),
                                            rotatedBitmap.width.toDouble(),
                                            screenHeight,
                                            screenWidth
                                        )
                                        val newScannedPage = ScannedPage(
                                            rawImagePath = savedUri,
                                            imageHeight = rotatedBitmap.height.toDouble(),
                                            imageWidth = rotatedBitmap.width.toDouble(),
                                            relativeImageWidth = result[1],
                                            relativeImageHeight = result[0],
                                            scaleFactorWidth = result[3],
                                            scaleFactorHeight = result[2],
                                        )
                                        withContext(Dispatchers.Main) {
                                            if (isRetake.value == true && indexToReplace != null) {
                                                replaceScannedPage(
                                                    newScannedPage,
                                                    indexToReplace!!,
                                                )
                                                indexToReplace = null
                                                setRetakeValue(false)
                                            } else {
                                                replaceScannedPage(
                                                    newScannedPage,
                                                    currentScannedPageIndex ?: 0
                                                )
                                            }
                                        }
                                        reRunInference(
                                            savedUri, rotatedBitmap, context, corners
                                        ) { corners, croppedPath ->
                                            addCroppedImageToScannedPage(
                                                savedUri,
                                                croppedPath,
                                                corners
                                            )
                                        }
                                    }
                                } catch (e: Exception) {
                                    Sentry.captureException(e);
                                    Log.e(
                                        "CameraViewModel",
                                        "Exception during image processing:",
                                        e
                                    )
                                    withContext(Dispatchers.Main) {
                                        onError(
                                            ImageCaptureException(
                                                ImageCapture.ERROR_UNKNOWN,
                                                "Exception during image processing",
                                                e
                                            )
                                        )
                                    }
                                }
                            }
                        }
                    })
            } catch (e: Exception) {
                Sentry.captureException(e);
                Log.e("CameraViewModel", "Exception during photo capture:", e)
                viewModelScope.launch {
                    onError(
                        ImageCaptureException(
                            ImageCapture.ERROR_UNKNOWN,
                            "Exception during photo capture",
                            e
                        )
                    )
                }
            }
        }
    }

    private fun createFile(context: Context): File {
        val fileName = "IMG_${System.currentTimeMillis()}.png"
        val storageDir = context.getExternalFilesDir(null)
        return File(storageDir, fileName)
    }

    /**
     * Deletes an image file from the given URI path.
     *
     * @param context The context used to access the content resolver (if needed).
     * @param uri The URI pointing to the image file to delete.
     * @return True if the file was successfully deleted, false otherwise.
     */
    fun deleteImageFromUri(uri: Uri?): Boolean {
        if (uri == null || uri.path == null) {
            Log.e(TAG, "URI is null. Cannot perform deletion.")
            return false
        }

        // Get the file path from the URI
        val file = File(uri.path)
        var deleted = false

        try {
            if (file.exists()) {
                deleted = file.delete()

                if (deleted) {
                    Log.d(TAG, "Image file deleted successfully: ${uri.path}")
                } else {
                    Log.e(TAG, "Failed to delete image file: ${uri.path}")
                }
            } else {
                Log.e(TAG, "File does not exist: ${uri.path}")
            }
        } catch (e: Exception) {
            Log.e(TAG, "Error while deleting image file: ${e.message}", e)
        }

        return deleted
    }


    fun getBitmapFromUri(context: Context, uri: Uri): Bitmap? {
        return try {
            val contentResolver = context.contentResolver
            val inputStream = contentResolver.openInputStream(uri)
            inputStream?.use {
                BitmapFactory.decodeStream(it)
            }
        } catch (e: Exception) {
            Sentry.captureException(e);
            e.printStackTrace()
            null
        }
    }

    /**
     * Deletes an image file from the given file path.
     *
     * @param filePath The path of the file to delete.
     * @return True if the file was successfully deleted, false otherwise.
     */
    fun deleteImageFromFilePath(filePath: String?): Boolean {
        if (filePath.isNullOrEmpty()) {
            Log.e(TAG, "File path is null or empty. Cannot perform deletion.")
            return false
        }

        val file = File(filePath)
        var deleted = false

        try {
            if (file.exists()) {
                deleted = file.delete()

                if (deleted) {
                    Log.d(TAG, "Image file deleted successfully: $filePath")
                } else {
                    Log.e(TAG, "Failed to delete image file: $filePath")
                }
            } else {
                Log.e(TAG, "File does not exist: $filePath")
            }
        } catch (e: Exception) {
            Log.e(TAG, "Error while deleting image file: ${e.message}", e)
        }

        return deleted
    }


    private fun saveRotatedImageToFile(
        filePath: String,
        rotationAngle: Float,
        outputFilePath: String
    ): Boolean {
        // Load the image from the file path
        val originalBitmap = BitmapFactory.decodeFile(filePath) ?: return false

        // Rotate the bitmap
        val matrix = Matrix().apply {
            postRotate(rotationAngle)
        }
        val rotatedBitmap = Bitmap.createBitmap(
            originalBitmap,
            0,
            0,
            originalBitmap.width,
            originalBitmap.height,
            matrix,
            true
        )
        // Save the rotated bitmap to the output file path
        return try {
            FileOutputStream(File(outputFilePath)).use { out ->
                rotatedBitmap.compress(Bitmap.CompressFormat.JPEG, 100, out)
            }
            // Encode the byte array to Base64
            rotatedBitmap.recycle()
            true
        } catch (e: IOException) {
            Sentry.captureException(e);
            e.printStackTrace()
            rotatedBitmap.recycle()
            false
        }
    }

    fun areArraysEqualWithThreshold(
        arr1: List<PointF>, arr2: List<PointF>, threshold: Float
    ): Boolean {
        if (arr1.size != arr2.size) {
            return false
        }
        for (i in arr1.indices) {
            if (!approxEquals(arr1[i], arr2[i], threshold)) {
                return false
            }
        }
        return true
    }

    private fun approxEquals(
        currentCorners: PointF, other: PointF, threshold: Float = 0.01f
    ): Boolean {
        return abs(currentCorners.x - other.x) <= threshold && abs(currentCorners.y - other.y) <= threshold
    }

    fun getOutputDirectory(context: Context): File {
        // Attempt to use the app-specific external storage directory which does not require permissions
        val mediaDir = context.getExternalFilesDir(null)?.let {
            File(it, context.resources.getString(R.string.app_name)).apply { mkdirs() }
        }
        // Fallback to internal storage if the external directory is not available
        return if (mediaDir != null && mediaDir.exists()) mediaDir else context.filesDir
    }

    private fun initOutputMap(interpreter: Interpreter): HashMap<Int, Any> {
        val outputMap = HashMap<Int, Any>()
        //model v3
//         outputMap[0] = List.filled(1 * 4 * 4, 0).reshape([1, 4, 4]);
        // outputMap[1] = List.filled(1 * 4, 0).reshape([1, 4]);
        // outputMap[2] = List.filled(1 * 4, 0).reshape([1, 4]);
        // outputMap[3] = List.filled(1, 0).reshape([1]);
        // outputMap[4] = List.filled(1 * 4 * 4 * 2, 0).reshape([1, 4, 4, 2]);
        // outputMap[5] = List.filled(1 * 4 * 4, 0).reshape([1, 4, 4]);
        // 1 * 10 * 4 contains heatmaps
        val boxes = interpreter.getOutputTensor(0).shape()
//        val output0 = TensorBuffer.createFixedSize(intArrayOf(boxes[0], boxes[1],boxes[2]), DataType.FLOAT32)

        outputMap[0] = Array(boxes[0]) {
            FloatArray(boxes[1])
        }

        // 1 * 10 contains offsets
        val classId = interpreter.getOutputTensor(1).shape()
        outputMap[1] = Array(classId[0]) {
            Array(classId[1]) {
                Array(classId[2]) {
                    FloatArray(classId[3])
                }
            }
        }

        // 1 * 10  contains forward displacements
        val boxScore = interpreter.getOutputTensor(2).shape()
        outputMap[2] = Array(boxScore[0]) {
            Array(boxScore[1]) {
                FloatArray(boxScore[2])
            }
        }

        // 1 contains backward displacements
        val count = interpreter.getOutputTensor(3).shape()
        outputMap[3] = Array(count[0]) {
            FloatArray(count[1])
        }

        //1 * 10 * 4 *2
        val keypoint = interpreter.getOutputTensor(4).shape()
        outputMap[4] = Array(keypoint[0]) {
            Array(keypoint[1]) {
                FloatArray(keypoint[2])
            }
        }

        //1 * 10 * 4
        val keypointScore = interpreter.getOutputTensor(5).shape()
        outputMap[5] = FloatArray(keypointScore[0])
//        warpTransform = WarpTransform()

        return outputMap
    }

    private fun loadModelFile(assets: AssetManager, modelPath: String): MappedByteBuffer {
        val fileDescriptor = assets.openFd(modelPath)
        val inputStream = FileInputStream(fileDescriptor.fileDescriptor)
        val fileChannel = inputStream.channel
        val startOffset = fileDescriptor.startOffset
        val declaredLength = fileDescriptor.declaredLength
        return fileChannel.map(FileChannel.MapMode.READ_ONLY, startOffset, declaredLength)
    }

    private fun runInference(image: Bitmap, isImageRotated: Boolean): Map<String, Any> {
        try {
            val inputIndex = 0
            val start = System.currentTimeMillis()
            val inputShape = interpreter.getInputTensor(inputIndex).shape()
            Log.d(TAG, inputShape.toString())
            val output = Array(1) { FloatArray(8) } // Adjust output size according to your model

            tfInputSize = Size(inputShape[2], inputShape[1])
            val tensorImage2 = TensorImage(DataType.FLOAT32).also { it.load(image) }

            val tfImageProcessor: ImageProcessor = if (isImageRotated) {
                ImageProcessor.Builder().add(Rot90Op(-90 / 90)).add(
                    ResizeOp(
                        tfInputSize.height,
                        tfInputSize.width,
                        ResizeOp.ResizeMethod.NEAREST_NEIGHBOR
                    )
                ).build()
            } else {
                ImageProcessor.Builder().add(
                    ResizeOp(
                        tfInputSize.height,
                        tfInputSize.width,
                        ResizeOp.ResizeMethod.NEAREST_NEIGHBOR
                    )
                ).build()
            }

            val tensorImage = tfImageProcessor.process(tensorImage2);
            val inputTensor = arrayOf(tensorImage.buffer)
            Log.d("inputTensor", tensorImage.buffer.capacity().toString())
            val outputTensor = initOutputMap(interpreter)
            interpreter.runForMultipleInputsOutputs(inputTensor, outputTensor)

            val score = outputTensor[4] as Array<Array<FloatArray>> //v5

            // Process output

            var maxSum = Float.NEGATIVE_INFINITY
            var maxIndex = mutableListOf<Int>()
            val xCords = mutableListOf<Float>()
            val yCords = mutableListOf<Float>()

            for (i in score.indices) {
                for (j in score[i].indices) {
                    var sum = 0f
                    for (k in score[i][j].indices) {
                        sum += score[i][j][k]
                    }
                    if (sum > maxSum) {
                        maxSum = sum
                        maxIndex = mutableListOf(j)
                    } else if (sum == maxSum) {
                        maxIndex.add(j)
                    }
                }
            }

            val keyPoints = outputTensor[1] as Array<Array<Array<FloatArray>>> //v5
            for (i in keyPoints[0][maxIndex[0]].indices) {
                xCords.add(keyPoints[0][maxIndex[0]][i][1])
                yCords.add(keyPoints[0][maxIndex[0]][i][0])
            }

            // MAKE SURE PREVIEW AND INPUT IMAGE SIZES ARE THE SAME OTHERWISE CORNER POINTS WILL MISMATCH

            val combinedList: Array<PointF> = if (maxSum > 0.8) {
                arrayOf(
                    PointF(xCords[0], yCords[0]),
                    PointF(xCords[1], yCords[1]),
                    PointF(xCords[2], yCords[2]),
                    PointF(xCords[3], yCords[3])
                )
            } else {
                var count = 0
                for (i in score[0][maxIndex[0]].indices) {
                    val checkThreshForAllPoints = score[0][maxIndex[0]][i]
                    if (checkThreshForAllPoints > ACCURACY_THRESHOLD) {
                        count++
                    }
                }

//                if (count > 2) {

                arrayOf(
                    PointF(xCords[0], yCords[0]),
                    PointF(xCords[1], yCords[1]),
                    PointF(xCords[2], yCords[2]),
                    PointF(xCords[3], yCords[3])
                )

//                } else {
//                    arrayOf(
//                        PointF(100f, 100f),                                 // Top-left corner
//                        PointF(100f, image.height.toFloat() - 100),            // Bottom-left corner
//                        PointF(
//                            image.width.toFloat() - 100,
//                            image.height.toFloat() - 100
//                        ), // Bottom-right corner
//                        PointF(image.width.toFloat() - 100, 100f),             // Top-right corner
//                    )
//
//                }
            }
            val end = System.currentTimeMillis()
            val inferenceTime = (end - start)
            Log.d(TAG, "Inference Time $inferenceTime")
            return mapOf(
                "corner_points" to combinedList,
                "conf_score" to maxSum,
                "inference_time" to inferenceTime
            )
        } catch (e: Exception) {
            Sentry.captureException(e);
            Log.e(TAG, "Something went wrong : ${e.message}")
            return mapOf(
                "corner_points" to arrayOf<PointF>(), "conf_score" to 0.0F, "inference_time" to 0.0F
            )
        }
    }

    /**
     * Process Ocr Request via API call as per `ocrMode`
     * @param fileUri : `Uri` of file
     * @param ocrMode : enum type to specify the ocr mode
     * @param onResult : result callback with response or exception as `String`
     */
    suspend fun processOcrRequest(
        filePathFront: String,
        filePathBack: String?,
        ocrMode: OcrMode
    ): Map<String, Any?> {
        val result: MutableMap<String, Any?> = mutableMapOf()
        when (ocrMode) {
            OcrMode.INVOICE_OCR -> {
                processImageForInvoiceOcr(
                    filePath = filePathFront,
                    onSuccess = { ocrResponse ->
                        result["ocrResponse"] = toMap(ocrResponse)
                    },
                    onError = { error ->
                        result["error"] = error
                    }
                )
            }

            OcrMode.PAN_OCR -> {
                uploadImageForOcr(
                    filePath = filePathFront,
                    filePathBack = filePathBack,
                    taskId = "f26eb21e-4c35-4491-b2d5-41fa0e545a34",
                    cardType = "PAN",
                    onSuccess = { ocrResponse ->
                        result["ocrResponse"] = toMap(ocrResponse)
                    },
                    onError = { error ->
                        result["error"] = error
                    }
                )
            }

            OcrMode.AADHAAR_OCR -> {
                uploadImageForOcr(
                    filePath = filePathFront,
                    filePathBack = filePathBack,
                    taskId = "f26eb21e-4c35-4491-b2d5-41fa0e545a34",
                    cardType = "AADHAAR",
                    onSuccess = { ocrResponse ->
                        result["ocrResponse"] = toMap(ocrResponse)
                    },
                    onError = { error ->
                        result["error"] = error
                    }
                )
            }


        }

        return result ?: throw Exception("Cant process the ocr request")
    }

    /**
     * Function to upload an image for OCR API in IO coroutine
     * @param filePath : filepath of document
     * @param taskId : task id of coroutine / job
     * @param cardType : type of card : AADHAAR , PAN
     * @param onSuccess : success callback with `OcrResponse`
     * @param onError : error callback with exception string
     */
    private suspend fun uploadImageForOcr(
        filePath: String,
        filePathBack: String? = null,
        taskId: String,
        cardType: String,
        onSuccess: (CardInfo) -> Unit,
        onError: (String) -> Unit
    ) {
        //front
        val base64Image = convertImageToBase64(filePath)
        val cleanedBase64 = base64Image.replace("\n", "").replace("\r", "")

        if (base64Image.isEmpty()) {
            onError("Failed to convert image to Base64.")
            return
        }

        //back
        var cleanedBase64Back: String? = null
        if (filePathBack != null) {
            val base64ImageBack = convertImageToBase64(filePathBack)
            cleanedBase64Back = base64ImageBack.replace("\n", "").replace("\r", "")

            if (cleanedBase64Back.isEmpty()) {
                onError("Failed to convert back image to Base64.")
                return
            }
        }

        try {
            val response = repository.uploadImageForOcr(
                cleanedBase64,
                taskId,
                cardType,
                cleanedBase64Back ?: ""
            )
            val cardInfo = response.body()?.result?.cardInfo
            if (response.isSuccessful && cardInfo != null) {
                onSuccess(cardInfo)
            } else {
                onError("OCR API Exception:${response.code()} ${response.message()} ${response.body()?.responseMessage.toString()} ${response.body()?.responseCode.toString()}")
            }
        } catch (e: Exception) {
            onError("OCR Exception: ${e.message}")
        }
    }

    /**
     * Function to process document using Document Processor API in IO coroutine
     * @param filePath filepath of image to process
     * @param onSuccess callback with `DocumentProcessorResponse` as response
     * @param onError callback with `String` as response for exception
     */
    private suspend fun processImageForInvoiceOcr(
        filePath: String,
        onSuccess: (DocumentResult) -> Unit,
        onError: (String) -> Unit
    ) {
        val base64Image = convertImageToBase64(filePath)
        val cleanedBase64 = base64Image.replace("\n", "").replace("\r", "")
        if (base64Image.isEmpty()) {
            onError("Failed to convert image to Base64.")
            return
        }
        try {
            val response = repository.processDocument(cleanedBase64)
            if (response.isSuccessful && response.body() != null) {

                val invoiceData = response.body()?.result
                if (invoiceData == null) {
                    onError("OCR API Exception:${response.code()} ${response.message()} ${response.body()?.responseMessage.toString()} ${response.body()?.responseCode.toString()}")
                } else {
                    onSuccess(invoiceData)
                }
            } else {
                onError("Document Processor API Error: ${response.message()}")
            }
        } catch (e: Exception) {
            onError("Document Processor API Exception: ${e.message}")
        }
    }

    /**
     * Converts image to base64
     * @param filePath : accepts absolute file path of image
     * @return `String` : returns the base64 string
     */
    private fun convertImageToBase64(filePath: String): String {
        return try {
            val file = File(filePath)
            val fis = FileInputStream(file)
            val byteArrayOutputStream = ByteArrayOutputStream()
            val buffer = ByteArray(1024)
            var bytesRead: Int
            while (fis.read(buffer).also { bytesRead = it } != -1) {
                byteArrayOutputStream.write(buffer, 0, bytesRead)
            }
            fis.close()
            val imageBytes = byteArrayOutputStream.toByteArray()
            Base64.encodeToString(imageBytes, Base64.DEFAULT)
        } catch (e: IOException) {
            Sentry.captureException(e);
            Log.e(TAG, "Error converting image to base64: ${e.message}")
            ""
        }
    }

    private fun saveBitmap(bitmap: Bitmap, outputPath: String): Boolean {
        return try {
            val outputFile = File(outputPath)
            val outputStream = FileOutputStream(outputFile)
            bitmap.compress(Bitmap.CompressFormat.JPEG, 100, outputStream)
            outputStream.flush()
            outputStream.close()
            true
        } catch (e: IOException) {
            Sentry.captureException(e);
            Log.e(TAG, "Error saving image: ${e.message}")
            false
        }
    }

    suspend fun initiateCrop(
        uriRawImagePath: Uri,
        imageHeight: Double,
        imageWidth: Double,
        srcPoints: Array<PointF>,
        context: Context
    ): String {
        val dstPoints = arrayOf(
            PointF(0f, 0f),                // Top-left corner
            PointF(imageWidth.toFloat(), 0f),   // Top-right corner
            PointF(imageWidth.toFloat(), imageHeight.toFloat()),  // Bottom-right corner
            PointF(0f, imageHeight.toFloat())   // Bottom-left corner
        )
        val imageBitmap = getBitmapFromUri(context, uriRawImagePath)
        val transformedPath = getFilePath()

        withContext(Dispatchers.IO) {
            cropAndPerspectiveTransform(
                uriRawImagePath,
                imageBitmap,
                srcPoints,
                transformedPath,
                context = context
            )
            imageBitmap?.recycle()
        }

        return transformedPath
    }

    fun cropAndPerspectiveTransform(
        rawImageUri: Uri,
        bitmap: Bitmap?,
        srcPoints: Array<PointF>,
        outputPath: String,
        context: Context
    ): String? {
        // Ensure that srcPoints have exactly 4 points
        if (srcPoints.size != 4) {
            throw IllegalArgumentException("srcPoints must have exactly 4 points")
        }

        val finalBitmapToProcess: Bitmap? =
            bitmap ?: getBitmapFromUri(context = context, rawImageUri)
        if (finalBitmapToProcess == null) {
            return null
        }

        // Calculate the bounding box of the srcPoints
        val minX = abs(srcPoints.minOf { it.x }).coerceAtLeast(0f)
        val maxX = abs(srcPoints.maxOf { it.x }).coerceAtMost(finalBitmapToProcess.width.toFloat())
        val minY = abs(srcPoints.minOf { it.y }).coerceAtLeast(0f)
        val maxY = abs(srcPoints.maxOf { it.y }).coerceAtMost(finalBitmapToProcess.height.toFloat())

        // Ensure the width and height of the cropped area are positive
        val width = (maxX - minX).coerceAtLeast(1f)
        val height = (maxY - minY).coerceAtLeast(1f)

        // Crop the image to the bounding box
        val croppedBitmap = Bitmap.createBitmap(
            finalBitmapToProcess,
            minX.toInt(),
            minY.toInt(),
            width.toInt(),
            height.toInt()
        )

        // Adjust srcPoints relative to the cropped bitmap
        val adjustedSrcPoints = arrayOf(
            PointF(srcPoints[0].x - minX, srcPoints[0].y - minY),
            PointF(srcPoints[1].x - minX, srcPoints[1].y - minY),
            PointF(srcPoints[2].x - minX, srcPoints[2].y - minY),
            PointF(srcPoints[3].x - minX, srcPoints[3].y - minY)
        )

        // Define the destination points as a rectangle the size of the cropped bitmap
        val dstPoints = arrayOf(
            PointF(0f, 0f),                          // Top-left corner
            PointF(croppedBitmap.width.toFloat(), 0f),   // Top-right corner
            PointF(
                croppedBitmap.width.toFloat(),
                croppedBitmap.height.toFloat()
            ), // Bottom-right corner
            PointF(0f, croppedBitmap.height.toFloat())  // Bottom-left corner
        )

        // Perform perspective transform on the cropped bitmap
        return perspectiveTransform(
            rawImageUri,
            croppedBitmap,
            adjustedSrcPoints,
            dstPoints,
            outputPath,
            context
        )
    }


}


fun perspectiveTransform(
    rawImageUri: Uri,
    bitmap: Bitmap?,
    srcPoints: Array<PointF>,
    dstPoints: Array<PointF>,
    outputPath: String,
    context: Context
): String? {
    // Ensure that srcPoints and dstPoints have exactly 4 points each
    if (srcPoints.size != 4 || dstPoints.size != 4) {
        throw IllegalArgumentException("srcPoints and dstPoints must have exactly 4 points")
    }

    // Create a matrix for the perspective transformation
    val matrix = Matrix()
    val src = floatArrayOf(
        srcPoints[0].x, srcPoints[0].y,
        srcPoints[1].x, srcPoints[1].y,
        srcPoints[2].x, srcPoints[2].y,
        srcPoints[3].x, srcPoints[3].y
    )
    val dst = floatArrayOf(
        dstPoints[0].x, dstPoints[0].y,
        dstPoints[1].x, dstPoints[1].y,
        dstPoints[2].x, dstPoints[2].y,
        dstPoints[3].x, dstPoints[3].y
    )

    // Set the perspective transformation on the matrix
    matrix.setPolyToPoly(src, 0, dst, 0, 4)
    val finalBitmapToProcess: Bitmap?
    if (bitmap != null) {
        finalBitmapToProcess = bitmap
    } else {
        finalBitmapToProcess = getBitmapFromUri(context = context, rawImageUri)

    }
    if (finalBitmapToProcess == null) {
        return null
    }

    // Create a new bitmap to hold the transformed image
    val transformedBitmap = Bitmap.createBitmap(
        finalBitmapToProcess.width, finalBitmapToProcess.height, Bitmap.Config.ARGB_8888
    )
    val paint = Paint(Paint.ANTI_ALIAS_FLAG).apply {
        isAntiAlias = true
        isFilterBitmap = true
        isDither = true
    }
    // Draw the transformed bitmap using a canvas and the transformation matrix
    val canvas = Canvas(transformedBitmap)
    canvas.drawBitmap(finalBitmapToProcess, matrix, paint)

    // Save the transformed bitmap to the specified output path
    return try {
        val outputFile = File(outputPath)
        val outputStream = FileOutputStream(outputFile)
        transformedBitmap.compress(Bitmap.CompressFormat.JPEG, 100, outputStream)
        outputStream.flush()
        outputStream.close()
        outputPath
    } catch (e: IOException) {
        Sentry.captureException(e);
        Log.e(TAG, "Error saving transformed image: ${e.message}")
        null
    }
}

@OptIn(ExperimentalGetImage::class)
fun ImageProxy.toBitmap(): Bitmap? {
    val image = this.image ?: return null
    val plane = image.planes[0]
    val buffer = plane.buffer
    val data = ByteArray(buffer.remaining())
    buffer.get(data)
    // Create Bitmap from byte array
    // ...
    return null // Replace with the actual implementation to create a bitmap from the data
}