package one.zoop.sdk.scanner

import ScannerResult
import android.app.Activity
import android.content.Context
import android.content.Intent
import android.util.Log
import androidx.activity.ComponentActivity
import androidx.activity.result.ActivityResultLauncher
import androidx.activity.result.contract.ActivityResultContracts
import com.google.firebase.firestore.FirebaseFirestore
import io.sentry.Sentry
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch
import one.zoop.sdk.scanner.interfaces.ILicenseManager
import one.zoop.sdk.scanner.interfaces.IScanner
import one.zoop.sdk.scanner.interfaces.IZoopScannerSdk
import one.zoop.sdk.scanner.model.ScannerConfig
import one.zoop.sdk.scanner.model.ScannerResultData
import one.zoop.sdk.scanner.utils.DefaultScannerLauncher
import one.zoop.sdk.scanner.utils.FirestoreService
import one.zoop.sdk.scanner.utils.LicenseManager
import one.zoop.sdk.scanner.utils.OrgDetails
import one.zoop.sdk.scanner.utils.ScannerConfigManager
import one.zoop.sdk.scanner.utils.TxnManager

sealed class ZoopScannerSdkBase(
    protected val activity: ComponentActivity,
    protected val licenseManager: ILicenseManager,
    protected val scanner: IScanner
) {
    protected abstract var startScannerForResult: ActivityResultLauncher<Intent>
    protected abstract var resultCallback: ((Result<ScannerResultData<*>>) -> Unit)?

    protected abstract fun handleScannerResult(resultCode: Int, data: Intent?)


    fun start(callback: (Result<ScannerResultData<*>>) -> Unit) {
        startScanner(callback)
    }

    protected abstract fun startScanner(callback: (Result<ScannerResultData<*>>) -> Unit)

    // Configuration and License Key setup must be done internally and not exposed
    protected abstract fun configureScanner(config: ScannerConfig)
    protected abstract fun initializeLicenseKey(
        context: Context,
        licenseKey: String,
        orgPid: String?
    )
}


internal class ZoopScannerSdkImpl(
    activity: ComponentActivity,
    licenseManager: ILicenseManager,
    scanner: IScanner = DefaultScannerLauncher()
) : ZoopScannerSdkBase(activity, licenseManager, scanner), IZoopScannerSdk {
    private var clientId: String? = null
    private var orgId: String? = null
    private var isSandbox: Boolean = true
    private lateinit var firestoreSchema: FirestoreService
    override var startScannerForResult: ActivityResultLauncher<Intent> =
        activity.registerForActivityResult(
            ActivityResultContracts.StartActivityForResult()
        ) { result ->
            handleScannerResult(result.resultCode, result.data)
        }

    override var resultCallback: ((Result<ScannerResultData<*>>) -> Unit)? = null

    override fun handleScannerResult(resultCode: Int, data: Intent?) {
        if (resultCode == Activity.RESULT_OK) {
            val scanResult = data?.getSerializableExtra("scan_result") as ScannerResult?
            if (scanResult != null) {
                val scanResultData = ScannerResultData(
                    data = scanResult.data,
                    status = scanResult.status,
                    message = scanResult.message,
                    createdAt = scanResult.createdAt
                )
                resultCallback?.invoke(Result.success(scanResultData))
            }
        } else {
            val scanResult = data?.getSerializableExtra("scan_result") as ScannerResult?
            if (scanResult != null) {
                val scanResultData = ScannerResultData(
                    data = null,
                    status = scanResult.status,
                    message = scanResult.message,
                    createdAt = scanResult.createdAt
                )
                resultCallback?.invoke(Result.failure(Exception(scanResult.message)))
            }
        }
    }

    override fun startScanner(callback: (Result<ScannerResultData<*>>) -> Unit) {
        try {
            resultCallback = callback
            when {
                !licenseManager.isLicenseValid() -> {
                    val errorMessage = "License is not initialized yet."
                    Log.e(TAG, errorMessage)
                    resultCallback?.invoke(Result.failure(Exception(errorMessage)))
                }

                ScannerConfigManager.getConfig() == null -> {
                    val errorMessage = "ScannerConfig is not initialized yet."
                    Log.e(TAG, errorMessage)
                    resultCallback?.invoke(Result.failure(Exception(errorMessage)))
                }

                else -> {
                    clientId?.let {
                        orgId?.let { it1 ->
                            TxnManager.startNewTransaction(
                                it,
                                it1, isSandbox
                            )
                        }
                    }
                    scanner.launchScanner(activity, startScannerForResult)
                }
            }
        } catch (e: Exception) {
            Log.e(TAG, "Failed to start ScannerActivity", e)
            resultCallback?.invoke(Result.failure(e))
            Sentry.captureException(e);
        }
    }

    override fun configureScanner(config: ScannerConfig) {
        ScannerConfigManager.setConfig(config)
    }

    override fun initializeLicenseKey(context: Context, licenseKey: String, orgPid: String?) {
        licenseManager.setLicenseKey(context, licenseKey, orgPid) {
            val licenseParts = licenseKey.split(":")
            if (licenseParts.size == 2) {
                clientId = licenseParts[0]
                // Verify the license to retrieve orgPid and isSandbox
                if (it) {
                    // Assuming that these details can be fetched after license verification
                    CoroutineScope(Dispatchers.IO).launch {
                        val orgDetails = fetchOrgDetailsFromLicense(clientId ?: "")
                        orgId = orgPid ?: orgDetails?.orgPid
                        isSandbox = orgDetails?.isSandbox ?: true
                    }

                }
            }
        }

    }

    private suspend fun fetchOrgDetailsFromLicense(clientId: String): OrgDetails? {
        return try {
            firestoreSchema = FirestoreService(FirebaseFirestore.getInstance())
            val orgDetails = firestoreSchema.getOrgDetails(clientId)
            orgDetails
        } catch (e: Exception) {
            Sentry.captureException(e);
            Log.e(TAG, "Error fetching org details: ${e.message}")
            null
        }
    }


    companion object {
        private const val TAG = "ZoopScannerSdk"
    }
}
