package one.zoop.sdk.scanner.data.network

import okhttp3.Interceptor
import okhttp3.OkHttpClient
import okhttp3.Response
import okhttp3.logging.HttpLoggingInterceptor
import one.zoop.sdk.scanner.ndk.NdkConnector
import retrofit2.Retrofit
import retrofit2.converter.gson.GsonConverterFactory

object RetrofitClient {

    private const val BASE_URL = "https://live.zoop.one/api/"

    // Custom Interceptor to add headers
    private class AuthInterceptor : Interceptor {
        override fun intercept(chain: Interceptor.Chain): Response {
            val originalRequest = chain.request()

            // Fetch app-id and api-key securely from NDK
            val appId = NdkConnector.getAppId()
            val ocrApiKey = NdkConnector.getOcrApiKey()

            // Add headers
            val modifiedRequest = originalRequest.newBuilder()
                .addHeader("app-id", appId)
                .addHeader("api-key", ocrApiKey)
                .addHeader("Content-Type", "application/json")
                .build()

            return chain.proceed(modifiedRequest)
        }
    }

    // Logging Interceptor for debugging
    private val loggingInterceptor: HttpLoggingInterceptor by lazy {
        HttpLoggingInterceptor().apply {
            level = HttpLoggingInterceptor.Level.BODY
        }
    }

    // OkHttpClient with multiple interceptors
    private val client: OkHttpClient by lazy {
        OkHttpClient.Builder()
            .addInterceptor(AuthInterceptor()) // Add Auth Interceptor
            .addInterceptor(loggingInterceptor) // Add Logging Interceptor
            .build()
    }

    // Retrofit instance
    val ocrApiService: OcrApiService by lazy {
        Retrofit.Builder()
            .baseUrl(BASE_URL)
            .client(client)
            .addConverterFactory(GsonConverterFactory.create())
            .build()
            .create(OcrApiService::class.java)
    }
}
