package one.zoop.sdk.scanner.ui.core.components.crop_handler
//DrawView is just like CustomPainter : used to draw points and rectangle for cropHandler
//used in both auto capture screen when showing points and while preview screen to show crop handier
import android.animation.Animator
import android.animation.AnimatorListenerAdapter
import android.animation.AnimatorSet
import android.animation.ValueAnimator
import android.content.Context
import android.graphics.Canvas
import android.graphics.Paint
import android.graphics.PointF
import android.util.AttributeSet
import android.util.Log
import android.view.MotionEvent
import android.view.View
import android.view.animation.AccelerateDecelerateInterpolator
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.res.colorResource
import androidx.core.content.ContextCompat
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch
import kotlinx.coroutines.withContext
import one.zoop.sdk.scanner.R
import one.zoop.sdk.scanner.model.ScannedPage
import one.zoop.sdk.scanner.utils.ImageProcessor
import one.zoop.sdk.scanner.viewmodel.CameraViewModel
import kotlin.math.sqrt

class DrawView : View {
    companion object {
        const val DEBUG_LOG = "DrawViewLogs"
        const val BLUE_CIRCLE_RADIUS = 20F
        const val ANIMATION_DURATION: Long = 200
    }

    var scannedPage: ScannedPage? = null

    //TL TR BR BL
    private var points: Array<PointF> = arrayOf() // To store scaled points


    // array that holds the balls
    private var indexMatch = -1
    private var isCropPreview = false
    private var isShowCropRect: Boolean = true
    private var isTouchable: Boolean = true
    private lateinit var viewModel: CameraViewModel
    private var currentIndex: Int = 0

    // Helper function to initialize default points based on canvas size ( width and height )
//    Model point system : TL,BL,BR,TL
    fun initializeDefaultPoints(height: Float, width: Float) {

        points = arrayOf(
            PointF(50f, 50f),                                 // Top-left corner
            PointF(50f, height - 50),            // Bottom-left corner
            PointF(width - 50, height - 50), // Bottom-right corner
            PointF(width - 50, 50f),             // Top-right corner
        )
        invalidate()
    }


    private fun scalePoints() {
        points = points.map { point ->
            PointF(point.x * width, point.y * height)
        }.toTypedArray()
        invalidate() // Redraw the view with the new points
    }

    private fun deScalePoints(oldWidth: Int, oldHeight: Int) {
        points = points.map { point ->
            PointF(point.x / oldWidth, point.y / oldHeight)
        }.toTypedArray()
        invalidate() // Redraw the view with the new points
    }

    override fun onSizeChanged(w: Int, h: Int, oldw: Int, oldh: Int) {
        super.onSizeChanged(w, h, oldw, oldh)
        if (oldw != 0 && oldh != 0) {
            deScalePoints(oldWidth = oldw, oldHeight = oldh)
        }
        scalePoints() // Scale points when the size changes
    }

    fun initializeCropScreen() {
        isShowCropRect = true
        isCropPreview = true
    }

    fun initializeViewModel(viewModel: CameraViewModel) {
        this.viewModel = viewModel
    }

    fun changeCurrentScannedPageIndex(currentIndex: Int) {
        this.currentIndex = currentIndex
    }

    fun getModelCornerPoints(): Array<PointF> {
        val currentPoints = Array(points.size) { i ->
            PointF(points[i].x / viewWidth!!, points[i].y / viewHeight!!)
        }
        return currentPoints
    }

    fun getCurrentDrawViewPoints(): Array<PointF> {
        //PASSING DIRECTLY WILL RETURN THE REFERENCED VAR ONLY
        val currentPoints = Array(points.size) { i ->
            PointF(points[i].x, points[i].y)
        }
        return currentPoints
    }


    var imageBitmapSize: PointF? = null

    private val scaleFactor: PointF?
        get() {
            val size = imageBitmapSize ?: return null
            return PointF(size.x / width, size.y / height)
        }

    fun getFlutterDrawViewPoints(): Array<PointF> {
        if (scaleFactor != null)
            return arrayOf()
        val multipliedPoints = Array(points.size) { i ->
            PointF(points[i].x * scaleFactor!!.x, points[i].y * scaleFactor!!.y)
        }

        return multipliedPoints
    }

    private var viewHeight: Int? = null
    private var viewWidth: Int? = null
    override fun onMeasure(widthMeasureSpec: Int, heightMeasureSpec: Int) {
        super.onMeasure(widthMeasureSpec, heightMeasureSpec)
        viewWidth = MeasureSpec.getSize(widthMeasureSpec)
        viewHeight = MeasureSpec.getSize(heightMeasureSpec)
    }

//    override fun onSizeChanged(w: Int, h: Int, oldw: Int, oldh: Int) {
//        super.onSizeChanged(w, h, oldw, oldh)
//        val scale = w.toFloat() / 2250
//        val newHeight = (4000 * scale).toInt()
//        setMeasuredDimension(w, newHeight)
//    }

    //set points and render
    fun setCornerPoints(points: Array<PointF>) {
        this.points = points
        invalidate()
    }

    //animation
    private var cornerAnimation: AnimatorSet? = null
    fun setCornerPointsAnimated(newPoints: Array<PointF>) {
        // Cancel any ongoing animation
        cornerAnimation?.cancel()

        // First time corner show
        if (points.isEmpty()) {
            points = newPoints
            invalidate()
            return
        }

        // Disappear corner when no points
        if (newPoints.isEmpty()) {
            points = newPoints
            invalidate()
            return
        }

        val animators = mutableListOf<Animator>()
        val maxIndex = minOf(newPoints.size, points.size)
        for (i in 0 until maxIndex) {
            val currentPoint = points[i]
            val targetPoint = newPoints[i]

            //0 - start of anim
            //1 - end of anim
            val animator = ValueAnimator.ofFloat(0f, 1f)
            animator.addUpdateListener { animation ->
                val fraction = animation.animatedValue as Float
                currentPoint.x += fraction * (targetPoint.x - currentPoint.x)
                currentPoint.y += fraction * (targetPoint.y - currentPoint.y)
                invalidate()
            }

            animator.duration = ANIMATION_DURATION
            animator.interpolator = AccelerateDecelerateInterpolator()

            animators.add(animator)
        }

        cornerAnimation = AnimatorSet()
        cornerAnimation?.playTogether(animators)
        cornerAnimation?.addListener(object : AnimatorListenerAdapter() {
            override fun onAnimationEnd(animation: Animator) {
                cornerAnimation = null
            }
        })

        cornerAnimation?.start()
    }

    fun resetCornerPoints() {
        setCornerPoints(arrayOf())
        invalidate()
    }

    //disabled rect lines and corner touch-ability
    fun initializeForCameraScreen() {
        isShowCropRect = false
        isTouchable = false
        invalidate()

    }

    //  DrawView class programmatically in code
    constructor(context: Context) : super(context) {
        isFocusable = true // necessary for getting the touch events
//        setMeasuredDimension(2250,4000)
    }

    //constructor for DrawView in an XML layout file
    constructor(context: Context, attrs: AttributeSet?, defStyle: Int) : super(
        context,
        attrs,
        defStyle
    ) {
        isFocusable = true // necessary for getting the touch events
    }

    //constructor for DrawView in an XML layout file but without any style params passed in xml
    constructor(context: Context, attrs: AttributeSet?) : super(context, attrs) {
        isFocusable = true // necessary for getting the touch events
    }

    private var linePaint = Paint().also { paint ->
        paint.isAntiAlias = true
        paint.isDither = true
        paint.strokeJoin = Paint.Join.ROUND
        paint.style = Paint.Style.STROKE
        paint.color = ContextCompat.getColor(context, R.color.colorPrimary)
        paint.strokeWidth = 8f
    }

    // the method that draws on canvas
    override fun onDraw(canvas: Canvas) {
        Log.d(DEBUG_LOG, "$width $height")
        //draw the corners
        for (i in points.indices) {
            if (i + 1 == points.size) {
                if (isShowCropRect) {
                    canvas.drawLine(
                        points[i].x,
                        points[i].y,
                        points[0].x,
                        points[0].y,
                        linePaint
                    )
                }
//                canvas.drawCircle(points[i].x, points[i].y, WHITE_CIRCLE_RADIUS, whitePaint)
                canvas.drawCircle(points[i].x, points[i].y, BLUE_CIRCLE_RADIUS, bluePaint)
//                canvas.drawCircle(points[0].x, points[0].y, WHITE_CIRCLE_RADIUS, whitePaint)
                canvas.drawCircle(points[0].x, points[0].y, BLUE_CIRCLE_RADIUS, bluePaint)
            } else {
                if (isShowCropRect) {
                    canvas.drawLine(
                        points[i].x,
                        points[i].y,
                        points[i + 1].x,
                        points[i + 1].y,
                        linePaint
                    )
                }
                if (i != 0) {
//                    canvas.drawCircle(points[i].x, points[i].y, WHITE_CIRCLE_RADIUS, whitePaint)
                    canvas.drawCircle(points[i].x, points[i].y, BLUE_CIRCLE_RADIUS, bluePaint)
                }
            }
        }
    }

//    //paint used for outer border circle
//    private val whitePaint: Paint = Paint().also { paint ->
//        paint.isDither = true
//        paint.isAntiAlias = true
//        paint.color = ContextCompat.getColor(context, R.color.cSecondaryColor)
//        paint.style = Paint.Style.STROKE
//        paint.strokeWidth = 10f
//    }

    //paint used for inner circle
    private val bluePaint: Paint = Paint().also { paint ->
        paint.isDither = true
        paint.isAntiAlias = true
        paint.color = ContextCompat.getColor(context, R.color.colorPrimary)
        paint.style = Paint.Style.FILL
    }

    // events when touching the screen
    override fun onTouchEvent(event: MotionEvent): Boolean {
        if (!isCropPreview || scaleFactor == null)
            return false
        val localPosition = PointF(event.x, event.y)
        when (event.action) {
            //find the corner that user is trying to drag
            MotionEvent.ACTION_DOWN -> {
                //find point index
                for (i in points.indices) {
                    val radCircle = sqrt(
                        ((points[i].x - localPosition.x) * (points[i].x - localPosition.x) + (points[i].y - localPosition.y)
                                * (points[i].y - localPosition.y)).toDouble()
                    )
                    if (radCircle <= 80F) {
                        indexMatch = i
                        break
                    }
                }
                if (indexMatch != -1) {
                    updateMagnifyingGlassPositionCallback?.invoke(
                        event.x * scaleFactor!!.x,
                        event.y * scaleFactor!!.y,
                        MotionEvent.ACTION_DOWN,
                        getMagnifierLocation(height, width, event.x, event.y)
                    )
                    invalidate()
                }
            }

            MotionEvent.ACTION_MOVE -> {

                if (indexMatch != -1 && PointsManipulation.checkPosition(
                        localPosition,
                        width,
                        height
                    )
                ) {
                    //check for angle between between each side -> if less than threshold -> stop
                    val limits = PointsManipulation.isValidMove(indexMatch, points, localPosition)
//                val limits = PointsManipulation.calculateLimit(
//                    localPosition,
//                    points,
//                    indexMatch,
//                    diagonals.x,
//                    diagonals.y,
//                )
                    if (limits) {
                        updateMagnifyingGlassPositionCallback?.invoke(
                            event.x * scaleFactor!!.x,
                            event.y * scaleFactor!!.y,
                            MotionEvent.ACTION_MOVE,
                            getMagnifierLocation(height, width, event.x, event.y)
                        )
                        points[indexMatch] = localPosition
                        invalidate()
                    }
                }
            }

            MotionEvent.ACTION_UP -> {
                updateMagnifyingGlassPositionCallback?.invoke(
                    event.x * scaleFactor!!.x,
                    event.y * scaleFactor!!.y,
                    MotionEvent.ACTION_UP,
                    getMagnifierLocation(height, width, event.x, event.y)
                )
                CoroutineScope(Dispatchers.Main).launch {

                    val currentScannedPage = viewModel.getCurrentScannedPage(currentIndex)
                    if (currentScannedPage != null) {
                        val originalPoints = ImageProcessor.pointProcessorNormalize(
                            points = points.toList(),
                            viewWidth = width.toDouble(),
                            viewHeight = height.toDouble(),
                            imageWidth = scannedPage!!.imageWidth!!.toDouble(),
                            imageHeight = scannedPage!!.imageHeight!!.toDouble(),
                        )
                        val tfliteModelCorner1 =
                            PointF(originalPoints[0].x, originalPoints[0].y) // Top-left
                        val tfliteModelCorner2 = PointF(originalPoints[3].x, originalPoints[3].y)
                        val tfliteModelCorner3 = PointF(originalPoints[2].x, originalPoints[2].y)
                        val tfliteModelCorner4 = PointF(originalPoints[1].x, originalPoints[1].y)
                        var reversedPoints = arrayOf(
                            tfliteModelCorner1,
                            tfliteModelCorner2,
                            tfliteModelCorner3,
                            tfliteModelCorner4
                        )

                        val targetPath = withContext(Dispatchers.IO) {
                            viewModel.initiateCrop(
                                uriRawImagePath = currentScannedPage.rawImagePath,
                                imageHeight = currentScannedPage.imageHeight!!,
                                imageWidth = currentScannedPage.imageWidth!!,
                                context = context,
                                srcPoints = reversedPoints
                            )
                        }
                        val newScannedPage = ScannedPage(
                            rawImagePath = currentScannedPage.rawImagePath,
                            imageHeight = currentScannedPage.imageHeight,
                            imageWidth = currentScannedPage.imageWidth,
                            relativeImageWidth = currentScannedPage.relativeImageWidth,
                            relativeImageHeight = currentScannedPage.relativeImageHeight,
                            scaleFactorWidth = currentScannedPage.scaleFactorWidth,
                            scaleFactorHeight = currentScannedPage.scaleFactorHeight,
                            processedImagePath = targetPath,
                            detectedCorners = currentScannedPage.detectedCorners,
                            correctedCorners = ImageProcessor.getRelativePoints(
                                originalPoints.toList(),
                                currentScannedPage.imageWidth!!,
                                currentScannedPage.imageHeight!!
                            ),

                            )
                        viewModel.replaceScannedPage(newScannedPage, currentIndex)
                    }
                }
                //
                // Define the destination points for perspective transformation
//                val dstPoints = arrayOf(
//                    PointF(0f, 0f),                // Top-left corner
//                    PointF(width.toFloat(), 0f),   // Top-right corner
//                    PointF(width.toFloat(), height.toFloat()),  // Bottom-right corner
//                    PointF(0f, height.toFloat())   // Bottom-left corner
//                )
//                val transformedPath = getFilePath()
//                perspectiveTransform(rotatedBitmap, points, dstPoints, transformedPath)
                indexMatch = -1
            }
        }
        // redraw the canvas
        invalidate()
        return true
    }

    /* magnifying glass view */
    var updateMagnifyingGlassPositionCallback: ((x: Float, y: Float, MOTION_ACTION: Int, indexCorner: Int) -> Unit)? =
        null

    private fun getMagnifierLocation(
        canvasHeight: Int,
        canvasWidth: Int,
        focusX: Float,
        focusY: Float
    ): Int {
        //if focus in TL quad , move to TR
        if (focusY < canvasHeight / 2 && focusX < canvasWidth / 2) {
            return 1;
        }
        //if focus in TR quad , move to TL
        else if (focusY < canvasHeight / 2 &&
            focusX >= canvasWidth / 2
        ) {
            return 0;
        }
        //if focus in BL quad , move to TR
        else if (focusY > canvasHeight / 2 &&
            focusX < canvasWidth / 2
        ) {
            return 1;
        }
        //if focus in BR quad , move to TL
        else if (focusY > canvasHeight / 2 &&
            focusX >= canvasWidth / 2
        ) {
            return 0;
        } else {
            //default
            return 0;
        }
    }

}