package one.zoop.sdk.scanner.ui.layout.Scanner

import android.util.Log
import android.view.View
import androidx.camera.core.AspectRatio
import androidx.camera.core.CameraSelector
import androidx.camera.core.ImageAnalysis
import androidx.camera.core.ImageCapture
import androidx.camera.core.Preview
import androidx.camera.lifecycle.ProcessCameraProvider
import androidx.camera.view.PreviewView
import androidx.compose.foundation.gestures.detectTransformGestures
import androidx.compose.runtime.Composable
import androidx.compose.runtime.DisposableEffect
import androidx.compose.runtime.LaunchedEffect
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.setValue
import androidx.compose.ui.Modifier
import androidx.compose.ui.input.pointer.pointerInput
import androidx.compose.ui.platform.LocalContext
import androidx.compose.ui.platform.LocalLifecycleOwner
import androidx.compose.ui.viewinterop.AndroidView
import androidx.concurrent.futures.await
import androidx.lifecycle.Observer
import androidx.navigation.NavController
import androidx.window.layout.WindowMetricsCalculator
import io.sentry.Sentry
import one.zoop.sdk.scanner.ui.core.components.crop_handler.DrawView
import one.zoop.sdk.scanner.utils.ScannerConfigManager
import one.zoop.sdk.scanner.viewmodel.CameraViewModel
import java.util.concurrent.Executors
import kotlin.math.abs
import kotlin.math.max
import kotlin.math.min


val RATIO_4_3_VALUE = 4.0 / 3.0
val RATIO_16_9_VALUE = 16.0 / 9.0
private fun aspectRatio(width: Int, height: Int): Int {
    val previewRatio = max(width, height).toDouble() / min(width, height)
    if (abs(previewRatio - RATIO_4_3_VALUE) <= abs(previewRatio - RATIO_16_9_VALUE)) {
        return AspectRatio.RATIO_4_3
    }
    return AspectRatio.RATIO_16_9
}

@Composable
fun CameraPreview(
    viewModel: CameraViewModel,
    modifier: Modifier = Modifier,
        drawView: DrawView?,
    screenHeight: Double, screenWidth: Double,
    navController: NavController,
) {

    val context = LocalContext.current
    val lifecycleOwner = LocalLifecycleOwner.current
    val previewView = remember {
        PreviewView(context).apply {
            setBackgroundColor(android.graphics.Color.BLACK)
            visibility = View.INVISIBLE
        }
    }
    val metrics = WindowMetricsCalculator.getOrCreate()
        .computeCurrentWindowMetrics(context).bounds
    val screenAspectRatio = aspectRatio(metrics.width(), metrics.height())

    var isStreamReady by remember { mutableStateOf(false) }

    DisposableEffect(Unit) {
        val observer = Observer<PreviewView.StreamState> { streamState ->
            isStreamReady = streamState == PreviewView.StreamState.STREAMING
        }
        previewView.previewStreamState.observe(lifecycleOwner, observer)
        onDispose {
            previewView.previewStreamState.removeObserver(observer)
        }
    }

    AndroidView(
        factory = { previewView },
        update = { previewView.visibility = if (isStreamReady) View.VISIBLE else View.INVISIBLE },
        modifier = modifier.pointerInput(Unit) {
            detectTransformGestures(
                onGesture = { _, _, zoom, _ ->
                    if (zoom != 1f) {
                        viewModel.onZoomChange(zoom)
                    }
                }
            )
        }
        //uncomment for focus functionality
        //            .pointerInput(Unit) {
        //                detectTapGestures(
        //                    onTap = { offset ->
        //                        viewModel.onFocus(offset.x, offset.y,previewView)
        //                    }
        //                )
        //            }
    )

    LaunchedEffect(Unit) {
        try {

            val cameraProvider = ProcessCameraProvider.getInstance(context).await()
            val cameraSelector = CameraSelector.DEFAULT_BACK_CAMERA
            val imageAnalyzer = ImageAnalysis.Builder()
                .setTargetAspectRatio(AspectRatio.RATIO_16_9)
                .setBackpressureStrategy(ImageAnalysis.STRATEGY_KEEP_ONLY_LATEST)
                .build()
            val rotation = previewView.display?.rotation ?: 0
            val preview = Preview.Builder()
                .setTargetAspectRatio(AspectRatio.RATIO_16_9)
                .setTargetRotation(rotation)
                .build().also {
                    it.setSurfaceProvider(previewView.surfaceProvider)
                }

            val mImageCapture = ImageCapture.Builder().apply {
                setCaptureMode(ImageCapture.CAPTURE_MODE_MINIMIZE_LATENCY)
                setTargetRotation(rotation).setTargetAspectRatio(AspectRatio.RATIO_16_9)
            }.build()

            viewModel.setImageCapture(mImageCapture)
            viewModel.setImageAnalyzer(imageAnalyzer)
            val scannerConfig=ScannerConfigManager.getConfig()
            if (scannerConfig!= null && scannerConfig.isAutoEnabled) {
                viewModel.changeAutoCapturePromptState(null)
                viewModel.isAutoCaptureTimeout = false
                imageAnalyzer.setAnalyzer(
                    Executors.newSingleThreadExecutor(),
                    viewModel.tfliteHelper.TfLiteAnalyzer(
                        drawView,
                        previewView.width,
                        previewView.height,
                        viewModel = viewModel,
                        screenWidth = screenWidth,
                        screenHeight = screenHeight,
                        context = context,
                        navController = navController
                    )
                )
            }


            cameraProvider.unbindAll()
            val camera = cameraProvider.bindToLifecycle(
                lifecycleOwner,
                cameraSelector,
                preview,
                mImageCapture,
                imageAnalyzer,
            )
            if(viewModel.buttonStates.value[1].isEnabled){
                camera.cameraControl.enableTorch(true)
            }
            viewModel.setCameraController(camera.cameraControl)
            viewModel.setCameraInfo(camera.cameraInfo)

        } catch (exc: Exception) {
            Sentry.captureException(exc);
            Log.e("CameraPreview", "Use case binding failed", exc)
        }

    }
}
