package one.zoop.sdk.scanner.utils

import android.content.Context
import android.util.Log
import com.google.firebase.FirebaseApp
import com.google.firebase.FirebaseOptions
import com.google.firebase.firestore.FirebaseFirestore
import io.sentry.Sentry
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch
import kotlinx.coroutines.withContext
import one.zoop.sdk.scanner.interfaces.ILicenseManager
import one.zoop.sdk.scanner.ndk.NdkConnector

internal object LicenseManager : ILicenseManager {

    @Volatile
    private var isLicenseValid = false
    private lateinit var firestoreSchema: FirestoreService

    // Store the transaction ID for later use
    override fun setLicenseKey(
        context: Context,
        licenseKey: String,
        orgPid: String?,
        onValidationComplete: (Boolean) -> Unit
    ) {
        if (FirebaseApp.getApps(context).isEmpty()) {
            val apiKey = NdkConnector.getApiKey()
            val options = FirebaseOptions.Builder()
                .setApplicationId("1:504354202123:android:ef8d58ce383f872114dd0e") // Required for Analytics.
                .setApiKey(apiKey)
                .setProjectId("zoop-sdk")
                .build()

            FirebaseApp.initializeApp(context, options)
        }

        synchronized(this) {
            firestoreSchema = FirestoreService(FirebaseFirestore.getInstance())
            val (clientId, secretKey) = licenseKey.split(":")

            CoroutineScope(Dispatchers.IO).launch {
                try {
                    val isValid =
                        firestoreSchema.verifyAndUpdateLicense(clientId, secretKey, orgPid)
                    withContext(Dispatchers.Main) {
                        isLicenseValid = true
                        onValidationComplete(isLicenseValid)
                    }

                } catch (e: Exception) {
                    Sentry.captureException(e);
                    Log.e("LicenseManager", e.toString())
                }
            }
        }
    }

    // Call this method when the transaction completes

    override fun isLicenseValid(): Boolean = isLicenseValid
}
