/*
 * Copyright (C) The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package devliving.online.cvscanner.camera;

import android.Manifest;
import android.content.Context;
import android.content.res.Configuration;
import android.support.annotation.RequiresPermission;
import android.util.AttributeSet;
import android.util.Log;
import android.view.SurfaceHolder;
import android.view.SurfaceView;
import android.view.ViewGroup;

import com.google.android.gms.common.images.Size;

import java.io.IOException;

public class CameraSourcePreview extends ViewGroup {
    private static final String TAG = "CameraSourcePreview";

    private Context mContext;
    private SurfaceView mSurfaceView;
    private boolean mStartRequested;
    private boolean mSurfaceAvailable;
    private CameraSource mCameraSource;

    private GraphicOverlay mOverlay;

    public CameraSourcePreview(Context context, AttributeSet attrs) {
        super(context, attrs);
        mContext = context;
        mStartRequested = false;
        mSurfaceAvailable = false;

        mSurfaceView = new SurfaceView(context);
        mSurfaceView.getHolder().addCallback(new SurfaceCallback());

        addView(mSurfaceView);
    }

    @RequiresPermission(Manifest.permission.CAMERA)
    public void start(CameraSource cameraSource) throws IOException, SecurityException {
        if (cameraSource == null) {
            stop();
        }

        mCameraSource = cameraSource;

        if (mCameraSource != null) {
            mStartRequested = true;
            startIfReady();
        }
    }

    @RequiresPermission(Manifest.permission.CAMERA)
    public void start(CameraSource cameraSource, GraphicOverlay overlay) throws IOException, SecurityException {
        mOverlay = overlay;
        start(cameraSource);
    }

    public void stop() {
        if (mCameraSource != null) {
            mCameraSource.stop();
        }
    }

    public void release() {
        if (mCameraSource != null) {
            mCameraSource.release();
            mCameraSource = null;
        }
    }

    @Override
    protected void onConfigurationChanged(Configuration newConfig) {
        super.onConfigurationChanged(newConfig);

        if(mCameraSource != null){
            mCameraSource.updateRotation();
            updateOverlay();
        }
    }

    @RequiresPermission(Manifest.permission.CAMERA)
    private void startIfReady() throws IOException, SecurityException {
        if (mStartRequested && mSurfaceAvailable) {
            mCameraSource.start(mSurfaceView.getHolder());
            updateOverlay();
            mStartRequested = false;
        }
    }

    void updateOverlay(){
        Size size = mCameraSource.getPreviewSize();
        if (mOverlay != null && size != null) {
            int min = Math.min(size.getWidth(), size.getHeight());
            int max = Math.max(size.getWidth(), size.getHeight());
            if (isPortraitMode()) {
                // Swap width and height sizes when in portrait, since it will be rotated by
                // 90 degrees
                mOverlay.setCameraInfo(min, max, mCameraSource.getCameraFacing());
            } else {
                mOverlay.setCameraInfo(max, min, mCameraSource.getCameraFacing());
            }
            mOverlay.clear();
        }
    }

    private class SurfaceCallback implements SurfaceHolder.Callback {
        @Override
        public void surfaceCreated(SurfaceHolder surface) {
            mSurfaceAvailable = true;
            try {
                startIfReady();
            } catch (SecurityException se) {
                Log.e(TAG,"Do not have permission to start the camera", se);
            } catch (IOException e) {
                Log.e(TAG, "Could not start camera source.", e);
            }
        }

        @Override
        public void surfaceDestroyed(SurfaceHolder surface) {
            mSurfaceAvailable = false;
        }

        @Override
        public void surfaceChanged(SurfaceHolder holder, int format, int width, int height) {
            Log.d(TAG, "surface size: w: " + width + ", h:" + height);
        }
    }

    @Override
    protected void onLayout(boolean changed, int left, int top, int right, int bottom) {
        final int layoutWidth = right - left;
        final int layoutHeight = bottom - top;

        /*
        int width = layoutWidth;
        int height = layoutHeight;

        if (mCameraSource != null) {
            Size size = mCameraSource.getPreviewSize();
            if (size != null) {
                width = size.getWidth();
                height = size.getHeight();
            }
        }

        // Swap width and height sizes when in portrait, since it will be rotated 90 degrees
        if (isPortraitMode()) {
            int tmp = width;
            //noinspection SuspiciousNameCombination
            width = height;
            height = tmp;
        }

        final float aspectRatio = (float)width/(float)height;

        Log.d(TAG, "aspect ratio: " + aspectRatio);

        int childWidth = layoutWidth;
        int childHeight = layoutHeight;

        //TODO fix the following
        if(layoutHeight < layoutWidth){
            childWidth = Math.round(childHeight*aspectRatio);

            while (childWidth > layoutWidth) {
                childHeight = childHeight - 5;
                childWidth = Math.round(childHeight*aspectRatio);
            }
        }
        else{
            childHeight = Math.round(childWidth/aspectRatio);

            while (childHeight > layoutHeight) {
                childWidth = childWidth - 5;
                childHeight = Math.round(childWidth/aspectRatio);
            }
        }

        Log.d(TAG, "layout w: " + layoutWidth + ", h: " + layoutHeight);

        if(layoutHeight > layoutWidth){
            //fit height
            childHeight = layoutHeight;
            childWidth = Math.round(childHeight*aspectRatio);
            Log.d(TAG, "fit height -> cw: " + childWidth + ", ch: " + childHeight);

            if(childWidth < layoutWidth){
                int diff = layoutWidth - childWidth;
                childWidth = childWidth + diff;
                childHeight = childHeight + Math.round(diff/aspectRatio);

                Log.d(TAG, "fit height [nested block] -> cw: " + childWidth + ", ch: " + childHeight);
            }
        }
        else{
            //fit width
            childWidth = layoutWidth;
            childHeight = Math.round(childWidth/aspectRatio);
            Log.d(TAG, "fit width -> cw: " + childWidth + ", ch: " + childHeight);

            if(childHeight < layoutHeight){
                int diff = layoutHeight - childHeight;
                childHeight = childHeight + diff;
                childWidth = childWidth + Math.round(diff * aspectRatio);

                Log.d(TAG, "fit width [nested block] -> cw: " + childWidth + ", ch: " + childHeight);
            }
        }

        for (int i = 0; i < getChildCount(); ++i) {
            getChildAt(i).layout(0, 0, childWidth, childHeight);
        }

        int l = Math.abs((layoutWidth - childWidth)/2);
        int t = Math.abs(layoutHeight - childHeight/2);
        int r = l + childWidth;
        int b = t + childHeight;

        for (int i = 0; i < getChildCount(); ++i) {
            getChildAt(i).layout(l, t, r, b);
        }
        */

        for (int i = 0; i < getChildCount(); ++i) {
            getChildAt(i).layout(0, 0, layoutWidth, layoutHeight);
        }

        try {
            startIfReady();
        } catch (SecurityException se) {
            Log.e(TAG,"Do not have permission to start the camera", se);
        } catch (IOException e) {
            Log.e(TAG, "Could not start camera source.", e);
        }
    }

    private boolean isPortraitMode() {
        int orientation = mContext.getResources().getConfiguration().orientation;
        if (orientation == Configuration.ORIENTATION_LANDSCAPE) {
            return false;
        }
        if (orientation == Configuration.ORIENTATION_PORTRAIT) {
            return true;
        }

        Log.d(TAG, "isPortraitMode returning false by default");
        return false;
    }
}