package online.inote.naruto.cache;

import java.util.Set;
import java.util.concurrent.TimeUnit;

/**
 * @description 缓存管理
 * @author XQF.Sui
 * @date 2021/07/29 22:07
 */
public interface CacheManager {

  /** 版本自增步长 */
  long VERSION_INCREMENT_STEP = 1L;

  /**
   * 缓存信息
   *
   * @param key 缓存Key
   * @param value 值
   */
  void cache(final String key, final String value);

  /**
   * 缓存信息
   *
   * @param key 缓存Key
   * @param ttl 过期时间(单位:秒)
   */
  void cache(final String key, final long ttl);

  /**
   * 缓存信息
   *
   * @param key 缓存Key
   * @param value 值
   * @param ttl 过期时间(单位:秒)
   */
  void cache(final String key, final String value, final long ttl);

  /**
   * 自增
   *
   * @param key 缓存Key
   * @return 返回自增后值
   */
  Long increment(String key);

  /**
   * 是否存在
   *
   * @param key 缓存Key
   * @return 缓存中是否存在
   */
  boolean isExist(final String key);

  /**
   * 清除
   *
   * @param key 缓存Key
   */
  void delete(String key);

  /**
   * 获取值
   *
   * @param key 缓存Key
   * @return
   */
  String get(final String key);

  /**
   * 设置过期时间(默认单位:秒)
   *
   * @param key 缓存Key
   * @param ttl 过期时间
   * @return 设置结果
   */
  Boolean expire(String key, long ttl);

  /**
   * 设置过期时间
   *
   * @param key 缓存Key
   * @param ttl 过期时间
   * @param timeUnit 时间单位
   * @return 设置结果
   */
  Boolean expire(String key, long ttl, TimeUnit timeUnit);

  /**
   * 增加集合对象
   *
   * @param key 缓存Key
   * @param values 缓存内容集合
   * @return
   */
  Long addSet(String key, String... values);

  /**
   * 增加集合对象
   *
   * @param key 缓存Key
   * @param set 缓存内容集合
   * @return
   */
  Long addSet(String key, Set<String> set);

  /**
   * 删除集合中的元素对象
   *
   * @param key 缓存Key
   * @param values 元素对象
   * @return
   */
  Long remove(String key, String... values);

  /**
   * 转移元素对象
   *
   * @param key 缓存Key
   * @param value 元素对象
   * @param destKey 目标缓存对象
   * @return
   */
  Boolean move(String key, String value, String destKey);

  /**
   * 获取集合内容
   *
   * @param key 缓存Key
   * @return
   */
  Set<String> members(String key);

  /**
   * 集合内容是否包含元素对象
   *
   * @param key 缓存Key
   * @param value 元素对象
   * @return
   */
  Boolean isMember(String key, String value);

  /**
   * 集合元素个数
   *
   * @param key 缓存Key
   * @return
   */
  Long count(String key);
}
