package online.inote.naruto.cache;

import online.inote.naruto.utils.Assert;
import online.inote.naruto.utils.ClassLoader;
import online.inote.naruto.utils.log.LoggerFactory;

import java.util.Set;
import java.util.concurrent.TimeUnit;

/**
 * @description 缓存支持
 * @author XQF.Sui
 * @date 2021/07/29 22:12
 */
public final class CacheSupport {

  private final LoggerFactory LOGGER = LoggerFactory.getLogger(getClass());
  private static final CacheManager CACHE_MANAGER;

  static {
    CACHE_MANAGER = ClassLoader.load(CacheManager.class, CacheSupport.class.getClassLoader());
    Assert.nonNull(CACHE_MANAGER, " Load CacheManager result is null.");
  }

  /**
   * 缓存信息
   *
   * @param key 缓存Key
   * @param value 值
   */
  public static void cache(String key, String value) {
    CACHE_MANAGER.cache(key, value);
  }

  /**
   * 缓存信息
   *
   * @param key 缓存Key
   * @param ttl 过期时间(单位:秒)
   */
  public static void cache(String key, long ttl) {
    CACHE_MANAGER.cache(key, ttl);
  }

  /**
   * 缓存信息
   *
   * @param key 缓存Key
   * @param ttl 过期时间(单位:秒)
   */
  public static void cache(String key, String value, long ttl) {
    CACHE_MANAGER.cache(key, value, ttl);
  }

  /**
   * 自增
   *
   * @param key 缓存Key
   */
  public static Long increment(String key) {
    return CACHE_MANAGER.increment(key);
  }

  /**
   * 检查是否存在
   *
   * @param key 缓存Key
   * @return 是否存在
   */
  public static boolean isExist(String key) {
    return CACHE_MANAGER.isExist(key);
  }

  /**
   * 检查是否存在
   *
   * @param key 缓存Key
   */
  public static void delete(String key) {
    CACHE_MANAGER.delete(key);
  }

  /**
   * 获取值
   *
   * @param key 缓存Key
   */
  public static String get(String key) {
    return CACHE_MANAGER.get(key);
  }

  /**
   * 设置过期时间(默认单位:秒)
   *
   * @param key 缓存Key
   * @param ttl 过期时间
   * @return 设置结果
   */
  public static Boolean expire(String key, long ttl) {
    return CACHE_MANAGER.expire(key, ttl);
  }

  /**
   * 设置过期时间
   *
   * @param key 缓存Key
   * @param ttl 过期时间
   * @param timeUnit 时间单位
   * @return 设置结果
   */
  public static Boolean expire(String key, long ttl, TimeUnit timeUnit) {
    return CACHE_MANAGER.expire(key, ttl, timeUnit);
  }

  /**
   * Set集合中增加元素
   *
   * @param key 缓存Key
   * @param values 元素对象
   * @return 增加数量
   */
  public static Long addSet(String key, String... values) {
    return CACHE_MANAGER.addSet(key, values);
  }

  /**
   * Set集合中增加元素
   *
   * @param key 缓存Key
   * @param set 元素对象集合
   * @return 增加数量
   */
  public static Long addSet(String key, Set<String> set) {
    return CACHE_MANAGER.addSet(key, set.toArray(new String[0]));
  }

  /**
   * 删除Set集合元素
   *
   * @param key 缓存Key
   * @param values 元素对象
   * @return 增加数量
   */
  public static Long remove(String key, String... values) {
    return CACHE_MANAGER.remove(key, values);
  }

  /**
   * 迁移集合元素
   *
   * @param key 缓存Key
   * @param value 元素对象
   * @param destKey 目标缓存集合Key
   * @return 是否成功
   */
  public static Boolean move(String key, String value, String destKey) {
    return CACHE_MANAGER.move(key, value, destKey);
  }

  /**
   * 获取集合内容
   *
   * @param key 缓存Key
   * @return 集合内容
   */
  public static Set<String> members(String key) {
    return CACHE_MANAGER.members(key);
  }

  /**
   * 判断集合中是否包含该元素
   *
   * @param key 缓存Key
   * @param value 元素
   * @return 是否包含
   */
  public static Boolean isMember(String key, String value) {
    return CACHE_MANAGER.isMember(key, value);
  }

  /**
   * 集合元素个数
   *
   * @param key
   * @return
   */
  public static Long count(String key) {
    return CACHE_MANAGER.count(key);
  }
}
