package online.inote.naruto.utils;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.alibaba.fastjson.TypeReference;
import com.alibaba.fastjson.serializer.SerializerFeature;
import lombok.AllArgsConstructor;
import lombok.Getter;

import java.io.Serializable;

/**
 * @description 返回结果工厂类
 * @author XQF.Sui
 * @date 2021/7/31 12:05
 */
public class ResultFactory<T> implements Serializable {

  private static final long serialVersionUID = -3793616609751038854L;

  public String code;
  public String message;
  public T data;

  private ResultFactory() {}

  private ResultFactory(String code, String message) {
    this.code = code;
    this.message = message;
  }

  private ResultFactory(String code, String message, T data) {
    this.code = code;
    this.message = message;
    this.data = data;
  }

  public JSONObject dataToJsonObject() {
    try {
      if (data == null) {
        return null;
      }
      return JSON.parseObject(JSON.toJSONString(data));
    } catch (Exception e) {
      throw new RuntimeException("data转换JSONObject失败");
    }
  }

  public static ResultFactory<Object> as(String content) {
    return JSON.parseObject(content, new TypeReference<ResultFactory<Object>>() {});
  }

  /**
   * @Description:: 将工厂类转为JSON @Author: XQF.Sui
   *
   * @created 2019年3月28日 上午12:49:00
   * @return JSON字符串
   */
  public String toJson() {
    return JSON.toJSONString(this, SerializerFeature.WriteMapNullValue);
  }

  /**
   * @Description:: 将工厂类转为JSONObject @Author: XQF.Sui
   *
   * @created 2019年3月28日 上午12:49:00
   * @return JSON对象
   */
  public JSONObject toJsonObject() {
    return JSON.parseObject(toJson());
  }

  /**
   * @Description:: 成功 @Author: XQF.Sui
   *
   * @created 2019年3月28日 上午1:19:43
   * @return ResultFactory
   */
  public static <T> ResultFactory<T> success() {
    return new ResultFactory<>(ResultEnum.SUCCESS.key, ResultEnum.SUCCESS.msg);
  }

  /**
   * @Description:: 成功 @Author: XQF.Sui
   *
   * @created 2019年3月28日 上午1:18:19
   * @param data
   * @return cn.creditease.ec.utils.result.ResultFactory<T>
   */
  public static <T> ResultFactory<T> success(final T data) {
    return new ResultFactory<>(ResultEnum.SUCCESS.key, ResultEnum.SUCCESS.msg, data);
  }

  /**
   * @Description: 成功 @Author: XQF.Sui
   *
   * @created 2019年6月17日 上午9:28:55
   * @param data, message
   * @return
   */
  public static <T> ResultFactory<T> success(final T data, final String message) {
    return new ResultFactory<>(ResultEnum.SUCCESS.key, message, data);
  }

  /**
   * @Description:: 失败 @Author: XQF.Sui
   *
   * @created 2019年3月28日 上午1:22:42
   * @return
   */
  public static <T> ResultFactory<T> fail() {
    return new ResultFactory<>(ResultEnum.FAIL.key, ResultEnum.FAIL.msg, null);
  }

  /**
   * @Description:: 失败 @Author: XQF.Sui
   *
   * @created 2019年3月28日 上午1:23:04
   * @param message
   * @return
   */
  public static <T> ResultFactory<T> fail(final String message) {
    return new ResultFactory<>(ResultEnum.FAIL.key, message, null);
  }

  public static <T> ResultFactory<T> fail(final String message, Object... o) {
    return new ResultFactory<>(ResultEnum.FAIL.key, StringUtils.replaceParams(message, o), null);
  }

  /**
   * @Description:: 创建返回结果集 @Author: XQF.Sui
   *
   * @created 2019年3月28日 上午12:49:14
   * @param result
   * @return
   */
  public static <T> ResultFactory<T> create(final ResultEnum result) {
    return new ResultFactory<>(result.key, result.msg, null);
  }

  /**
   * @Description:: 创建返回结果集 @Author: XQF.Sui
   *
   * @created 2019年3月28日 上午12:49:38
   * @param result
   * @param message
   * @return
   */
  public static <T> ResultFactory<T> create(final ResultEnum result, final String message) {
    return new ResultFactory<>(result.key, message, null);
  }

  /**
   * @Description:: 创建返回结果集 @Author: XQF.Sui
   *
   * @created 2019年3月28日 上午12:40:23
   * @param data
   * @param result
   * @return
   */
  public static ResultFactory<String> create(final String data, final ResultEnum result) {
    return new ResultFactory<>(result.key, result.msg, data);
  }

  /**
   * @Description:: 创建返回结果集 @Author: XQF.Sui
   *
   * @created 2019年3月28日 上午12:49:48
   * @param result
   * @param t
   * @return
   */
  public static <T> ResultFactory<T> create(final ResultEnum result, final T t) {
    return new ResultFactory<>(result.key, result.msg, t);
  }

  /**
   * @Description:: 创建返回结果集 @Author: XQF.Sui
   *
   * @created 2019年3月28日 上午1:24:08
   * @param result
   * @param message
   * @param t
   * @return
   */
  public static <T> ResultFactory<T> create(
      final ResultEnum result, final String message, final T t) {
    return new ResultFactory<>(result.key, message, t);
  }

  /**
   * @Description: @Author: XQF.Sui @Date 2020/5/14 11:34 上午 @Param [code, message] @Return
   * cn.creditease.ec.utils.result.ResultFactory<T>
   */
  public static <T> ResultFactory<T> create(final String code, final String message) {
    return new ResultFactory<>(code, message);
  }

  /**
   * @Description:: 创建返回结果集 @Author: XQF.Sui
   *
   * @created 2019年3月28日 上午12:49:57
   * @param result
   * @param message
   * @param o
   * @return
   */
  public static <T> ResultFactory<T> create(
      final ResultEnum result, final String message, final Object... o) {
    return new ResultFactory<>(result.key, StringUtils.replaceParams(message, o), null);
  }

  /**
   * @Description:: 是否成功 @Author: XQF.Sui
   *
   * @created 2019年3月28日 上午12:50:06
   * @param factory
   * @return boolean
   */
  public static boolean isSuccess(final ResultFactory<Object> factory) {
    return factory != null && ResultEnum.SUCCESS.key.equals(factory.code);
  }

  /**
   * @Description: 是否成功 @Author: XQF.Sui
   *
   * @created 2019年4月23日 上午10:51:00
   * @since 1.0
   * @return boolean
   */
  public boolean isSuccess() {
    return ResultEnum.SUCCESS.key.equals(this.code);
  }

  /**
   * @Description:: 返回结果枚举类 @Author: XQF.Sui
   *
   * @created 2019年4月12日 下午11:15:23
   * @version 1.0
   */
  @Getter
  @AllArgsConstructor
  public enum ResultEnum {

    /** A10001 成功 */
    SUCCESS("A10001", "成功"),

    /** A10002 失败 */
    FAIL("A10002", "失败"),

    /** A10003 参数必填 */
    PARAM_REQUIRED("A10003", "参数必填"),

    /** A10004 该信息不存在 */
    NOT_EXISTS("A10004", "该信息不存在"),

    /** A10005 该信息已存在 */
    EXISTS("A10005", "该信息已存在"),

    /** A10006 有权限 */
    OWN_PERMISSION("A10006", "有权限"),

    /** A10007 没有权限 */
    NOT_PERMISSION("A10007", "没有权限"),

    /** A50008 非法令牌 */
    TOKEN_ILLEGAL("A50008", "非法令牌"),

    /** A50014 登录超时 */
    TOKEN_EXPIRED("A50014", "登录超时");

    private String key;
    private String msg;
  }
}
