package online.shuita.gitee.mojo.generator;

import com.google.common.collect.Lists;
import lombok.NoArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import online.shuita.gitee.mojo.model.*;
import org.apache.commons.io.FileUtils;

import java.io.File;
import java.io.IOException;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

@Slf4j
@NoArgsConstructor
public class MarkdownGenerator implements IGenerator {

    static {
        //注册文档生成器
        try {
            GeneratorManager.getManager().registGenerator(MarkdownGenerator.class.newInstance());
            log.info("注册成功");
        } catch (Throwable e) {
            log.error("regist error", e);
        }
    }

    @Override
    public void generateDoc(AnalyResult analyResult, Map<String, String> properties) {
        log.info(properties.get("outPath"));
        String outPath = properties.get("outPath");
        File out = new File(outPath);
        if (!out.isDirectory()) {
            try {
                FileUtils.write(out, "", "utf-8", false);
            } catch (IOException e) {
                log.error("输出文档失败", e);
            }
        }
        List<IntfModel> intfList = analyResult.getIntfModelList();
        Map<String, List<IntfModel>> intfMap = intfList.stream().collect(Collectors.groupingBy(IntfModel::getDomain));
        intfMap.entrySet().forEach(entry -> {
            String domain = entry.getKey();
            File curout = out;
            if (out.isDirectory()) {
                curout = new File(out, ("".equals(domain) ? "接口文档" : domain) + "-" + System.currentTimeMillis() + ".md");
            }
            try {
                FileUtils.write(curout, String.format("# %s接口清单  \n", domain), "utf-8", out.isDirectory() ? false : true);
                FileUtils.write(curout, "| 序号 | 接口 | 中文名 | 备注 |  \n", "utf-8", true);
                FileUtils.write(curout, "| --- | --- | --- | --- |  \n", "utf-8", true);

                List<IntfModel> sublist = entry.getValue();
                //输出接口清单
                for (int idx = 0; idx < sublist.size(); idx++) {
                    IntfModel intf = sublist.get(idx);
                    FileUtils.write(curout, String.format("| %s | [%s](#anchor%s) | %s | %s |  \n", idx + 1, converterHtmlStr(intf.getIntfShortName()), idx + 1, intf.getIntfDesc(), ""), "utf-8", true);
                }
                //输出接口详情
                for (int idx = 0; idx < sublist.size(); idx++) {
                    IntfModel intf = sublist.get(idx);
                    FileUtils.write(curout, String.format("### <span id=\"anchor%s\" name=\"anchor%s\">%s、%s</span>  \n", idx + 1, idx + 1, idx + 1, converterHtmlStr(intf.getIntfShortName())), "utf-8", true);

                    List<MethodModel> methodModelList = intf.getMethodModelList();

                    for (int midx = 0; midx < methodModelList.size(); midx++) {
                        generateMethodDoc(idx, midx, methodModelList.get(midx), curout, analyResult.getPojoModelMap());
                    }
                }
            } catch (IOException e) {
                log.error("输出文档失败", e);
            }
        });
    }

    private void generateMethodDoc(int intfIdx, int methodIdx, MethodModel methodModel, File curout, Map<String, List<PojoModel>> pojoModelMap) {
        try {
            FileUtils.write(curout, String.format("#### %s.%s、%s%s%s  \n", intfIdx + 1, methodIdx + 1, methodModel.getDeprecated() ? "~~" : "", converterHtmlStr(methodModel.getMethodName()), methodModel.getDeprecated() ? "~~" : ""), "utf-8", true);
            FileUtils.write(curout, "**接口名：**  \n", "utf-8", true);
            FileUtils.write(curout, String.format("%s  \n", converterHtmlStr(methodModel.getMethodName())), "utf-8", true);
            FileUtils.write(curout, "**接口说明：**  \n", "utf-8", true);
            FileUtils.write(curout, String.format("%s  \n", methodModel.getMethodDesc()), "utf-8", true);
            FileUtils.write(curout, "**入参：**  \n", "utf-8", true);
            //输出入参表格：
            FileUtils.write(curout, "<table>", "utf-8", true);
            //输出表头
            FileUtils.write(curout, "<tr><th>字段名</th><th>类型</th><th>是否必填</th><th>字段描述</th></tr>", "utf-8", true);
            //输出字段
            List<ParameterModel> paramList = methodModel.getParameterModelList();
            List<PojoModel> pojoModelList = Lists.newArrayList();

            for (int pidx = 0; pidx < paramList.size(); pidx++) {
                ParameterModel parm = paramList.get(pidx);
                FileUtils.write(curout, String.format("<tr><td>%s</td><td>%s</td><td>%s</td><td>%s</td></tr>",
                        converterHtmlStr(parm.getParameterName()), converterHtmlStr(parm.getTypeShotName()), "否", parm.getDesc()), "utf-8", true);
                if (pojoModelMap.containsKey(parm.getGenericType())) {
                    pojoModelList.addAll(pojoModelMap.get(parm.getGenericType()));
                }
            }

            //输出其他pojo
            if (null != pojoModelList) {
                List<String> generatePojo = Lists.newArrayList();
                generatePojoDoc(pojoModelList.stream().distinct().collect(Collectors.toList()),
                        curout, pojoModelMap, generatePojo);
            }

            FileUtils.write(curout, "</table>\n\n", "utf-8", true);

            FileUtils.write(curout, "**出参：**  \n", "utf-8", true);
            //输出入参表格：
            FileUtils.write(curout, "<table>\n", "utf-8", true);
            //输出表头
            FileUtils.write(curout, "<tr><th>字段名</th><th>类型</th><th>是否必填</th><th>字段描述</th></tr>\n", "utf-8", true);
            //输出字段
            List<PojoModel> outPojoModelList = Lists.newArrayList();
            if (pojoModelMap.containsKey(methodModel.getReturnGenericType())) {
                outPojoModelList.addAll(pojoModelMap.get(methodModel.getReturnGenericType()));
            } else {
                FileUtils.write(curout, String.format("<tr><td>%s</td><td>%s</td><td>%s</td><td>%s</td></tr>\n",
                        "returnValue", converterHtmlStr(methodModel.getReturnTypeShotName()), "是", ""), "utf-8", true);
            }

            //输出其他pojo
            if (null != outPojoModelList) {
                List<String> generatePojo = Lists.newArrayList();
                generatePojoDoc(outPojoModelList.stream().distinct().collect(Collectors.toList()),
                        curout, pojoModelMap, generatePojo);
            }

            FileUtils.write(curout, "</table>\n\n", "utf-8", true);
        } catch (IOException e) {
            log.error("输出文档失败", e);
        }
    }

    private void generatePojoDoc(List<PojoModel> pojoModels, File curout, Map<String, List<PojoModel>> pojoModelMap, List<String> generatePojo) {
        pojoModels.forEach(pojo -> {
            if (generatePojo.contains(pojo.getType())) {
                return;
            }
            try {
                FileUtils.write(curout, String.format("<tr><td colspan=\"4\"><b><i>%s</i><b></td></tr>\n", converterHtmlStr(pojo.getShortType())), "utf-8", true);

                List<FieldModel> paramList = pojo.getFieldEntryList();
                List<PojoModel> pojoModelList = Lists.newArrayList();

                for (int pidx = 0; pidx < paramList.size(); pidx++) {
                    FieldModel parm = paramList.get(pidx);
                    FileUtils.write(curout, String.format("<tr><td>%s%s%s</td><td>%s</td><td>%s</td><td>%s</td></tr>\n",
                            parm.getDeprecated() ? "<del>" : "", converterHtmlStr(parm.getName()), parm.getDeprecated() ? "</del>" : "", converterHtmlStr(parm.getShortType()), "否", parm.getDesc()), "utf-8", true);
                    if (pojoModelMap.containsKey(parm.getGenericType())) {
                        pojoModelList.addAll(pojoModelMap.get(parm.getGenericType()));
                    }
                }
                generatePojo.add(pojo.getType());

                //输出其他pojo
                if (null != pojoModelList && pojoModelList.size() > 0) {
                    generatePojoDoc(pojoModelList.stream().distinct().collect(Collectors.toList()),
                            curout, pojoModelMap, generatePojo);
                }
            } catch (IOException e) {
                log.error("输出文档失败", e);
            }
        });
    }

    private String converterHtmlStr(String str) {
        return str.replaceAll("<=", "<= &le;").replaceAll(">=", "&ge;").replaceAll("<", "&lt;").replaceAll(">", "&gt;");
    }
}