package com.iplatform.base.controller;

import com.iplatform.base.SystemController;
import com.iplatform.base.VariableConstants;
import com.iplatform.base.config.CacheProperties;
import com.iplatform.base.util.cache.CacheInfo;
import com.walker.cache.CacheProvider;
import com.walker.cache.SimpleCacheManager;
import com.walker.infrastructure.utils.StringUtils;
import com.walker.web.ResponseValue;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.data.redis.core.RedisCallback;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Properties;

@RestController
@RequestMapping("/monitor/cache")
@ConditionalOnProperty(prefix = "iplatform.cache", name = "redis-enabled", havingValue = "true", matchIfMissing = false)
public class CacheController extends SystemController {

    private RedisTemplate<String, Object> redisTemplate;
    private CacheProperties cacheProperties;

    @Autowired(required = false)
    public CacheController(RedisTemplate<String, Object> redisTemplate, CacheProperties cacheProperties){
        this.redisTemplate = redisTemplate;
        this.cacheProperties = cacheProperties;
    }

    @RequestMapping("/clearCacheName")
    public ResponseValue clearCacheName(String name){
        if(StringUtils.isEmpty(name)){
            return ResponseValue.error("必须指定缓存名称");
        }
        CacheProvider<?> cacheProvider = SimpleCacheManager.getCacheProvider(name);
        if(cacheProvider == null){
            return ResponseValue.error("缓存对象不存在");
        }
        try {
            cacheProvider.reload();
            return ResponseValue.success("缓存重新加载成功");
        } catch (Exception e) {
            logger.error("重新加载缓存失败：" + e.getMessage(), e);
            return ResponseValue.error("重新加载缓存失败:" + name);
        }
    }

    /**
     * 清空 系统缓存，该功能主要对 Redis 缓存方式，通过手动触发重构缓存，在开发阶段使用。
     * @return
     * @date 2023-08-26
     */
    @RequestMapping("/clearCacheAll")
    public ResponseValue reloadAllCacheProvider(){
        if(!this.cacheProperties.isRedisRebuild()){
            return ResponseValue.error("未开启'重构缓存'选项");
        }

        // 2023-08-26 开启了重建redis缓存选项
        Collection<CacheProvider<?>> cacheList = SimpleCacheManager.getCacheProviders();
        if(cacheList != null && cacheList.size() > 0){
            String name = null;
            try{
                for(CacheProvider<?> cacheProvider : cacheList){
                    name = cacheProvider.getProviderName();
                    cacheProvider.reload();
                }
            } catch (Exception ex){
                logger.error("重新加载所有缓存，出现异常:" + ex.getMessage() + ", name=" + name, ex);
                return ResponseValue.error("重新加载缓存，出现错误，name = " + name);
            }
        }
        return ResponseValue.success();
    }

    @RequestMapping("/getInfo")
    public ResponseValue getInfo(){
        if(this.redisTemplate == null){
            return ResponseValue.error("系统未开启Redis方式缓存");
        }

        Properties info = (Properties) redisTemplate.execute((RedisCallback<Object>) connection -> connection.info());
//        Properties commandStats = (Properties) redisTemplate.execute((RedisCallback<Object>) connection -> connection.info("commandstats"));
        Object dbSize = redisTemplate.execute((RedisCallback<Object>) connection -> connection.dbSize());

        Map<String, Object> result = new HashMap<>(3);
        result.put("info", info);
        result.put("dbSize", dbSize);
        result.put("commandStats", new ArrayList<>());

        return ResponseValue.success(result);
    }

    /**
     * 返回缓存对象名称列表，即:系统所有缓存定义名称集合
     * @return
     * @date 2023-01-04
     */
    @GetMapping("/select/getNames")
    public ResponseValue getCacheNames(){
        List<CacheInfo> cacheNames = new ArrayList<>();
        Collection<CacheProvider<?>> cacheList = SimpleCacheManager.getCacheProviders();
        if(cacheList != null && cacheList.size() > 0){
            for(CacheProvider<?> cacheProvider : cacheList){
                cacheNames.add(new CacheInfo(cacheProvider.getProviderName(), cacheProvider.getProviderType().getName()));
            }
        }
        return ResponseValue.success(cacheNames);
    }

    @GetMapping("/select/getKeys/{cacheName}")
    public ResponseValue getCacheKeys(@PathVariable String cacheName){
        if(StringUtils.isEmpty(cacheName)){
            return ResponseValue.error("参数错误");
        }
        List<String> keyList = new ArrayList<>((int) VariableConstants.MAX_CACHE_SHOW);
        CacheProvider<?> cacheProvider = SimpleCacheManager.getCacheProvider(cacheName);
        if(cacheProvider != null){
            if(cacheProvider.getCache().getPersistentSize() <= VariableConstants.MAX_CACHE_SHOW){
//                Cachable cachable = null;
//                for(Iterator<Cachable> it = cacheProvider.getCache().getIterator(); it.hasNext();){
//                    cachable = it.next();
//                    keyList.add(cachable.getKey());
//                }
                return ResponseValue.success(cacheProvider.getCache().getKeys());
            } else {
                return ResponseValue.error("缓存数据量超过 " + VariableConstants.MAX_CACHE_SHOW + ", 暂无法显示");
            }
        }
        return ResponseValue.success(keyList);
    }

    @GetMapping("/select/getValue/{cacheName}/{cacheKey}")
    public ResponseValue getOneCacheValue(@PathVariable String cacheName, @PathVariable String cacheKey){
        CacheProvider<?> cacheProvider = SimpleCacheManager.getCacheProvider(cacheName);
        if(cacheProvider != null){
            Object value = cacheProvider.getCacheData(cacheKey);
            CacheInfo cacheInfo = new CacheInfo(cacheName, cacheKey, value.toString());
            return ResponseValue.success(cacheInfo);
        }
        return ResponseValue.success();
    }
}
