package com.iplatform.base.controller;

import com.iplatform.base.CategoryCacheProvider;
import com.iplatform.base.Constants;
import com.iplatform.base.PlatformAdapterController;
import com.iplatform.base.cache.FormCacheProvider;
import com.iplatform.base.service.CategoryServiceImpl;
import com.iplatform.model.po.S_category;
import com.iplatform.model.vo.CategoryTreeVo;
import com.walker.infrastructure.utils.DateUtils;
import com.walker.infrastructure.utils.StringUtils;
import com.walker.web.ResponseValue;
import com.walker.web.WebRuntimeException;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;

import java.util.List;

/**
 * 平台端基础分类服务
 * @date 2023-05-15
 */
@RestController
@RequestMapping("/system/category")
public class CategoryController extends PlatformAdapterController {

    private CategoryServiceImpl categoryService;
    private CategoryCacheProvider categoryCacheProvider;
    private FormCacheProvider formCacheProvider;

    @Autowired
    public CategoryController(CategoryServiceImpl categoryService
            , CategoryCacheProvider categoryCacheProvider, FormCacheProvider formCacheProvider){
        this.categoryService = categoryService;
        this.categoryCacheProvider = categoryCacheProvider;
        this.formCacheProvider = formCacheProvider;
    }

    @RequestMapping("/list/tree")
    public ResponseValue getListTree(@RequestParam(name = "type") Integer type,
                                     @RequestParam(name = "status") Integer status,
                                     @RequestParam(name = "name", required = false) String name) {
        int owner = (int)this.getOwner();
//        return ResponseValue.success(this.categoryService.getListTree(type, status, name, owner));
        // 2023-05-18 表加上关联表单名称。
        List<CategoryTreeVo> data = this.categoryCacheProvider.getListTree(type, status, name, owner);
        if(!StringUtils.isEmptyList(data)){
            data.stream().forEach(e -> {
                if(e.getType().intValue() == Constants.CATEGORY_TYPE_CONFIG && StringUtils.isNotEmpty(e.getExtra())){
                    // 平台配置分类，extra存储的是formId
                    e.setFormName(this.formCacheProvider.getCacheData(e.getExtra()).getName());
                    if(!StringUtils.isEmptyList(e.getChild())){
                        for(CategoryTreeVo child : e.getChild()){
                            if(StringUtils.isNotEmpty(child.getExtra())){
                                child.setFormName(this.formCacheProvider.getCacheData(child.getExtra()).getName());
                            }
                        }
                    }
                }
            });
        }
        return ResponseValue.success(data);
    }

    @RequestMapping(value = "/info/{id}", method = RequestMethod.GET)
    public ResponseValue getInfo(@PathVariable(value = "id") Integer id){
        S_category category = this.categoryCacheProvider.get(id);
        if(category == null){
            throw new WebRuntimeException("分类不存在");
        }
        return ResponseValue.success(category);
    }

//    @ApiOperation(value = "新增基础分类")
    @RequestMapping(value = "/save", method = RequestMethod.POST)
    public ResponseValue save(@RequestBody S_category category){
        if(category == null){
            return ResponseValue.error("提交内容错误");
        }
        logger.debug(category.toString());

        if(this.categoryService.queryNameUnique(category.getName(), category.getType(), (int)this.getOwner(), 0) > 0){
            return ResponseValue.error("相同分类已经存在");
        }

        long dateTime = DateUtils.getDateTimeNumber();
        int id = this.categoryService.queryNextId();

        category.setId(id);
        category.setPath(this.getPathByPId(category.getPid()));
        category.setOwner((int)this.getOwner());
        category.setExtra(this.clearCdnPrefix(category.getExtra()));    // 这里额外保存的可能是文件路径信息，清除了前缀。
        category.setCreate_time(dateTime);
        category.setUpdate_time(dateTime);
        this.categoryService.insert(category);
        this.categoryCacheProvider.save(category);

        return ResponseValue.success();
    }

//    @ApiOperation(value = "修改基础分类")
    @RequestMapping(value = "/update", method = RequestMethod.POST)
    public ResponseValue update(@RequestBody S_category category){
        if(category == null || category.getId() <= 0){
            throw new WebRuntimeException("分类参数错误");
        }
        S_category exist = this.categoryCacheProvider.get(category.getId());
        if(exist == null){
            throw new WebRuntimeException("分类数据不存在，id=" + category.getId());
        }
        if(exist.getId().intValue() == category.getPid()){
            throw new WebRuntimeException("分类父级不能是自己");
        }
        if(!exist.getName().equals(category.getName())){
            if(this.categoryService.queryNameUnique(category.getName(), category.getType(), (int)this.getOwner(), exist.getId()) > 0){
                return ResponseValue.error("分类名称重复");
            }
        }

        if(StringUtils.isNotEmpty(category.getExtra())){
            category.setExtra(this.clearCdnPrefix(category.getExtra()));
        }
        category.setPath(this.getPathByPId(category.getPid()));
        category.setOwner((int)this.getOwner());
        category.setUpdate_time(DateUtils.getDateTimeNumber());
        this.categoryService.execUpdateCategory(category);
        this.categoryCacheProvider.update(category);
        return ResponseValue.success();
    }

    @RequestMapping("/delete")
    public ResponseValue delete(Integer id){
//        S_category category = this.categoryCacheProvider.get(id);
        if(this.categoryService.queryChildCategorySize(id) > 0){
            return ResponseValue.error("当前分类下有子类，请先删除子类！");
        }
        this.categoryService.delete(new S_category(id));
        this.categoryCacheProvider.remove(id);
        return ResponseValue.success();
    }

    private String getPathByPId(Integer pid) {
        S_category category = this.categoryCacheProvider.get(pid);
        if (category != null) {
            return category.getPath() + pid + StringUtils.FOLDER_SEPARATOR;
        }
        return "/0/";
    }
}
