package com.iplatform.base.controller;

import com.iplatform.base.SystemController;
import com.iplatform.base.cache.FormCacheProvider;
import com.iplatform.base.pojo.ConfigParam;
import com.iplatform.base.pojo.form.FormData;
import com.iplatform.base.pojo.form.FormDataItem;
import com.iplatform.base.service.ConfigArgumentServiceImpl;
import com.iplatform.model.po.S_config;
import com.walker.db.page.GenericPager;
import com.walker.infrastructure.arguments.ArgumentsManager;
import com.walker.infrastructure.arguments.Variable;
import com.walker.infrastructure.utils.DateUtils;
import com.walker.infrastructure.utils.NumberGenerator;
import com.walker.infrastructure.utils.StringUtils;
import com.walker.web.ResponseValue;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;

@RestController
@RequestMapping("/system/config")
public class ConfigController extends SystemController {

    private ConfigArgumentServiceImpl configArgumentService;

    private ArgumentsManager argumentsManager = null;
    private FormCacheProvider formCacheProvider;

    @Autowired
    public ConfigController(ConfigArgumentServiceImpl configArgumentService
            , ArgumentsManager argumentsManager, FormCacheProvider formCacheProvider){
        this.configArgumentService = configArgumentService;
        this.argumentsManager = argumentsManager;
        this.formCacheProvider = formCacheProvider;
    }

    @RequestMapping("/list")
    public ResponseValue list(ConfigParam configParam){
//        this.preparePageSearch();
        GenericPager<S_config> pager = this.configArgumentService.queryPageConfigList(configParam.getConfigName()
                , configParam.getConfigKey(), configParam.getConfigType());
//        return this.acquireTablePage(pager.getDatas(), pager.getTotalRows());
        // 缓存前端封装的table组件，可以直接返回分页对象。2023-04-11
        return ResponseValue.success(pager);
    }

    @RequestMapping("/view/{configId}")
    public ResponseValue view(@PathVariable Long configId){
        if(configId == null || configId.longValue() <= 0){
            return ResponseValue.error("config id required!");
        }
        S_config s_config = this.configArgumentService.get(new S_config(configId));
        return ResponseValue.success(s_config);
    }

    @RequestMapping("/edit")
    public ResponseValue saveEdit(@RequestBody S_config s_config){
        S_config exist = this.configArgumentService.get(new S_config(s_config.getConfig_id()));
        if(exist == null){
            return ResponseValue.error("配置项不存在: " + s_config.getConfig_id());
        }
        this.configArgumentService.save(s_config);
        this.argumentsManager.persist(s_config.getConfig_key(), s_config.getConfig_value());
        return ResponseValue.success();
    }

    /**
     * 添加保存新配置项。
     * @param s_config
     * @return
     */
    @RequestMapping("/add")
    public ResponseValue saveAdd(@RequestBody S_config s_config){
        if(s_config == null || StringUtils.isEmpty(s_config.getConfig_key())){
            return ResponseValue.error("提交配置数据错误");
        }
        S_config exist = this.configArgumentService.queryConfigByKey(s_config.getConfig_key());
        if(exist != null){
            return ResponseValue.error("配置项已经存在: " + s_config.getConfig_key());
        }
        s_config.setConfig_id(NumberGenerator.getSequenceNumber());
        s_config.setCreate_time(DateUtils.getDateTimeNumber(System.currentTimeMillis()));
        this.configArgumentService.insert(s_config);
        this.argumentsManager.persist(s_config.getConfig_key(), s_config.getConfig_value());
        return ResponseValue.success();
    }

    //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    //~ 2023-05-16 新界面使用，电商系统。
    //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

    /**
     * 获得一个唯一配置项的值。
     * @param key
     * @return
     * @date 2023-05-15
     */
    @RequestMapping(value = "/getuniq", method = RequestMethod.GET)
    public ResponseValue getConfigValue(@RequestParam String key){
        String configValue = this.getArgumentVariable(key).getStringValue();
        return ResponseValue.success("success", configValue);
    }

    /**
     * 根据formId（分组）获得包含的所有配置项集合，包括：key，value
     * @param formId
     * @return
     * @date 2023-05-16
     */
    @RequestMapping(value = "/info", method = RequestMethod.GET)
    public ResponseValue info(@RequestParam(value = "formId") Integer formId){
        if(formId == null){
            return ResponseValue.error("参数错误");
        }
        List<Variable> list = this.argumentsManager.getVariableList(String.valueOf(formId));
        if(StringUtils.isEmptyList(list)){
            return ResponseValue.success(new HashMap<>(1));
        }

        HashMap<String, String> map = new HashMap<>(8);
        for (Variable systemConfig : list) {
            map.put(systemConfig.getId(), systemConfig.getStringValue());
        }
        map.put("id", formId.toString());

        if(this.logger.isDebugEnabled()){
            this.logger.debug(map.toString());
        }
        return ResponseValue.success(map);
    }

    @RequestMapping(value = "/save/form", method = RequestMethod.POST)
    public ResponseValue saveFormConfig(@RequestBody FormData formData){
        if(formData == null || formData.getId() == null || StringUtils.isEmptyList(formData.getFields())){
            return ResponseValue.error("参数错误");
        }
        try{
            this.formCacheProvider.validateForm(formData);
        } catch (Exception ex){
            return ResponseValue.error("表单配置数据格式错误：" + ex.getMessage());
        }

        List<Object[]> savedConfigList = new ArrayList<>(8);
        Object[] config = null;
        String value = null;
        for(FormDataItem formDataItem : formData.getFields()){
            value = this.clearCdnPrefix(formDataItem.getValue());
            if(StringUtils.isEmpty(value)){
                // 去掉图片域名之后没有数据则说明当前数据就是图片域名
                logger.debug("配置的值是：图片域名，" + formDataItem.getValue());
                value = formDataItem.getValue();
            }
            config = new Object[2];
            config[0] = value;
            config[1] = formDataItem.getName();
            savedConfigList.add(config);
        }

        // 多个配置项同时更新
        this.configArgumentService.execUpdateFormConfig(savedConfigList);
        for(Object[] one : savedConfigList){
            this.argumentsManager.persist(one[1].toString(), one[0]);
        }
        return ResponseValue.success();
    }
}
