package com.iplatform.base.controller;

import com.iplatform.base.SystemController;
import com.iplatform.base.pojo.DeptParam;
import com.iplatform.base.service.DeptServiceImpl;
import com.iplatform.base.util.DeptUtils;
import com.iplatform.model.po.S_dept;
import com.iplatform.model.po.S_user_core;
import com.walker.infrastructure.tree.TreeNode;
import com.walker.infrastructure.utils.DateUtils;
import com.walker.infrastructure.utils.NumberGenerator;
import com.walker.infrastructure.utils.StringUtils;
import com.walker.web.DataStatus;
import com.walker.web.OrgType;
import com.walker.web.ResponseValue;
import com.walker.web.UserType;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

@RestController
@RequestMapping("/system/dept")
public class DeptController extends SystemController {

    private DeptServiceImpl deptService;

    @Autowired
    public DeptController(DeptServiceImpl deptService){
        this.deptService = deptService;
    }

    /**
     * 前端展示所有顶级单位列表，超级管理员可以选择所有机构，普通用户只能显示自己单位的。
     * @return
     * @date 2022-12-15
     */
    @GetMapping("/select/list_root_org")
    public ResponseValue listOrgRoot(){
        List<S_dept> rootList = this.getOrgListScope();
        return ResponseValue.success(rootList);
    }

    /**
     * 前端展示部门机构树(单个组织机构)，在添加用户选部门使用。
     * @param deptId 提供一个部门ID(界面选择)，该参数在超级管理员时使用。
     * @return
     * @date 2022-12-12
     */
    @GetMapping("/select/tree_dept/{deptId}")
    public ResponseValue listSelectDeptTree(@PathVariable Long deptId){
        long orgId = 0;
        if(this.isSupervisor()){
            if(deptId == null || deptId.longValue() <= 0){
                return ResponseValue.error("查询一个机构树，必须传入机构ID");
            }
            orgId = this.getRootOrgIdByDept(deptId);
        } else {
            orgId = this.getCurrentUser().getOrg_id();
        }
        List<S_dept> orgList = this.deptService.queryOrgListForTree(orgId);
        List<TreeNode> treeNodeList = DeptUtils.getOrgDeptTree(orgList);
        if(StringUtils.isEmptyList(treeNodeList)){
            // 2023-05-31 树组件不能数据为空
            treeNodeList = new ArrayList<>(1);
        }
        return ResponseValue.success(treeNodeList);
    }

    /**
     * 选择整个组织机构树，区分是否管理员。<p></p>
     * <pre>
     *     1)该功能是公共权限，system:dept:select:**
     *     2)展示所有机构树节点，如果是超级管理员则会显示所有根单位(集团)。
     *     3)普通用户只能看到本单位机构树。
     * </pre>
     * @return
     * @date 2022-12-08
     */
    @GetMapping("/select/tree_org")
    public ResponseValue listOrgRootTree(){
        List<S_dept> orgList = null;
        if(this.isSupervisor()){
            orgList = this.deptService.queryOrgListForTree(-1);
        } else {
           long orgId = this.getCurrentUser().getOrg_id();
           orgList = this.deptService.queryOrgListForTree(orgId);
        }
//        DeptTreeGenerator deptTreeGenerator = new DeptTreeGenerator(null);
//        deptTreeGenerator.setEntityList(orgList);
//        List<TreeNode> treeNodeList = deptTreeGenerator.getTreeRootList();
        List<TreeNode> treeNodeList = DeptUtils.getOrgDeptTree(orgList);
        if(StringUtils.isEmptyList(treeNodeList)){
            logger.error("未找到任何顶级机构树列表: treeNodeList = null");
        } else {
            logger.debug(treeNodeList.toString());
        }
        return ResponseValue.success(treeNodeList);
    }

    @GetMapping("/list")
    public ResponseValue listOrgList(DeptParam deptParam){

        String deptDataScope = this.getCurrentDataScope("103");
        logger.info("部门管理(数据权限) = " + deptDataScope);

        Map<String, Object> data = new HashMap<>(4);

        if(deptParam.getLoadSelect() == 1){
            // 来自下拉机构选择框请求
            List<S_dept> selectParentList = this.deptService.queryRootOrgChildrenList(deptParam.getOrgId(), null);
            data.put("deptList", DeptUtils.toSystemDeptList(selectParentList));
            return ResponseValue.success(data);
        }

        List<S_dept> rootList = this.getOrgListScope();
        S_user_core currentUser = this.getCurrentUser();
        List<S_dept> deptSrcList = null;

        long selectedOrgRoot = 0;
        if(currentUser.getUser_type().intValue() == UserType.TYPE_SUPER){
            // 超级管理员，可以看到所有根单位列表
            if(deptParam != null && deptParam.getOrgId() > 0){
                // 管理员选择了一个根机构
                selectedOrgRoot = deptParam.getOrgId();
            } else if(!StringUtils.isEmptyList(rootList)){
                // 没有选择，则默认查询第一个根机构下面的
                selectedOrgRoot = rootList.get(0).getId();
            }
            deptSrcList = this.deptService.queryRootOrgChildrenList(selectedOrgRoot, deptParam.getDeptName());
            data.put("deptList", DeptUtils.toSystemDeptList(deptSrcList));
        } else {
            // 其他用户只能看到自己所在根机构下的所有机构
            deptSrcList = this.deptService.queryRootOrgChildrenList(currentUser.getOrg_id(), deptParam.getDeptName());
            data.put("deptList", DeptUtils.toSystemDeptList(deptSrcList));
        }

        // 记录超级管理员选择的顶级机构，返回前端。2022-12-03
        if(deptParam.getOrgId() > 0){
            data.put("selectedRootOrgId", deptParam.getOrgId());
        } else {
            data.put("selectedRootOrgId", rootList.get(0).getId());
        }

        data.put("rootOrgList", rootList);
        return ResponseValue.success(data);
    }

    @PostMapping("/add")
    public ResponseValue saveAddDept(@RequestBody S_dept s_dept){
        if(s_dept == null || StringUtils.isEmpty(s_dept.getDept_name())){
            return ResponseValue.error("提交保存机构为空");
        }
        logger.info(s_dept.toString());

        s_dept.setId(NumberGenerator.getSequenceNumber());
        s_dept.setCreate_time(DateUtils.getDateTimeNumber(System.currentTimeMillis()));
        s_dept.setCreate_by(this.getCurrentUser().getUser_name());

        Long parentId = s_dept.getParent_id();
        if(parentId == null || parentId.longValue() == 0){
            // 说明是根机构，根机构: id = org_id
            s_dept.setOrg_id(s_dept.getId());
            s_dept.setParent_id(0L);
            s_dept.setAncestors("0");
            if(s_dept.getOrg_type().intValue() != OrgType.TYPE_ORG){
                return ResponseValue.error("顶级机构只能选择第一项");
            }

        } else {
            // 普通机构
            S_dept parentDept = this.deptService.queryOneDept(parentId);
            if(parentDept == null){
                return ResponseValue.error("上级单位不存在");
            }
            if(parentDept.getStatus().intValue() != DataStatus.CONST_NORMAL){
                return ResponseValue.error("该机构已停用，无法创建子机构");
            }
            s_dept.setOrg_id(parentDept.getOrg_id());
            s_dept.setAncestors(parentDept.getAncestors() + "," + s_dept.getParent_id());

            String error = this.checkOrgType(parentDept.getOrg_type(), s_dept.getOrg_type());
            if(error != null){
                return ResponseValue.error(error);
            }
        }
        this.deptService.insert(s_dept);
//        this.deptCacheProvider.putDept(s_dept);
        this.getDeptCacheProvider().putDept(s_dept);
        return ResponseValue.success();
    }

    @RequestMapping("/remove/{deptId}")
    public ResponseValue removeDept(@PathVariable Long deptId){
        if(deptId == null || deptId <= 0){
            return ResponseValue.error("机构参数错误");
        }
        int subDeptSize = this.deptService.querySubDeptSizeInCurrent(deptId);
        if(subDeptSize > 0){
            return ResponseValue.error("该机构下存在子机构，无法删除");
        }
        long userSize = this.deptService.queryUserSizeInCurrent(deptId);
        if(userSize > 0){
            return ResponseValue.error("该机构下存在用户，无法删除");
        }
        this.deptService.delete(new S_dept(deptId));
//        this.deptCacheProvider.removeDept(deptId);
        this.getDeptCacheProvider().removeDept(deptId);
        return ResponseValue.success();
    }

    @PostMapping("/edit")
    public ResponseValue saveEdit(@RequestBody S_dept s_dept){
        if(s_dept == null){
            return ResponseValue.error("编辑的机构不存在");
        }
        Long deptId = s_dept.getId();
        if(deptId == null || deptId.longValue() <= 0){
            return ResponseValue.error("编辑的机构ID不存在");
        }
        Long parentId = s_dept.getParent_id();
        if(parentId == null || parentId.longValue() == 0){
            // 说明编辑的是顶级机构，根机构: id = org_id
            if(s_dept.getOrg_type().intValue() != OrgType.TYPE_ORG){
                return ResponseValue.error("顶级机构只能选择一级机构");
            }
        } else {
            // 普通机构
            S_dept parentDept = this.deptService.queryOneDept(parentId);
            if(parentDept == null){
                return ResponseValue.error("上级单位不存在");
            }
            String error = this.checkOrgType(parentDept.getOrg_type(), s_dept.getOrg_type());
            if(error != null){
                return ResponseValue.error(error);
            }
        }
        s_dept.setCreate_time(DateUtils.getDateTimeNumber(System.currentTimeMillis()));
        this.deptService.save(s_dept);
//        this.deptCacheProvider.updateDept(s_dept);
        this.getDeptCacheProvider().updateDept(s_dept);
        return ResponseValue.success();
    }

    @GetMapping("/view/{deptId}")
    public ResponseValue viewDept(@PathVariable Long deptId){
        if(deptId == null || deptId.longValue() <= 0){
            return ResponseValue.error("参数错误");
        }
        S_dept s_dept = this.deptService.queryOneDept(deptId);
        if(s_dept == null){
            return ResponseValue.error("机构不存在");
        }
        return ResponseValue.success(s_dept);
    }

    private String checkOrgType(int parentOrgTypeValue, int currentOrgTypeValue){
        OrgType parentOrgType = OrgType.getType(parentOrgTypeValue);
        OrgType currentOrtType = OrgType.getType(currentOrgTypeValue);
        if(currentOrtType == OrgType.OrgSub){
            if(parentOrgType != OrgType.Org){
                return "二级单位只能在顶级单位下面";
            }
        } else if (currentOrtType == OrgType.OrgFactory) {
            if(parentOrgType != OrgType.OrgSub){
                return "三级单位只能在二级单位下面";
            }
        } else if (currentOrtType == OrgType.OrgSubFactory) {
            if(parentOrgType != OrgType.OrgFactory){
                return "四级单位只能在三级级单位下面";
            }
        }
        return null;
    }
}
