package com.iplatform.base.controller;

import com.iplatform.base.SystemController;
import com.iplatform.base.cache.MenuCacheProvider;
import com.iplatform.base.service.MenuServiceImpl;
import com.iplatform.base.service.RoleServiceImpl;
import com.iplatform.base.util.MenuUtils;
import com.iplatform.base.util.menu.SystemMenu;
import com.iplatform.model.po.S_menu;
import com.iplatform.model.po.S_user_core;
import com.walker.infrastructure.utils.DateUtils;
import com.walker.infrastructure.utils.StringUtils;
import com.walker.web.DataStatus;
import com.walker.web.ResponseValue;
import com.walker.web.UserPrincipal;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

@RestController
@RequestMapping("/system/menu")
public class MenuController extends SystemController {

    private MenuCacheProvider menuCacheProvider;
    private RoleServiceImpl roleService;
    private MenuServiceImpl menuService;

    @Autowired
    public MenuController(MenuCacheProvider menuCacheProvider
            , RoleServiceImpl roleService, MenuServiceImpl menuService){
        this.menuCacheProvider = menuCacheProvider;
        this.roleService = roleService;
        this.menuService = menuService;
    }

    @RequestMapping("/remove/{menuId}")
    public ResponseValue remove(@PathVariable String menuId){
        boolean existChildren = this.menuCacheProvider.isHasChildren(menuId);
        if(existChildren){
            return ResponseValue.error("该菜单存在子菜单，无法删除");
        }
        int roleMenuSize = this.menuService.queryRoleMenuSize(menuId);
        if(roleMenuSize > 0){
            return ResponseValue.error("菜单已分配，无法删除");
        }
        this.menuService.delete(new S_menu(menuId));
        this.menuCacheProvider.removeCacheData(menuId);
        return ResponseValue.success();
    }

    @RequestMapping("/view/{menuId}")
    public ResponseValue view(@PathVariable String menuId){
        S_menu s_menu = this.menuService.get(new S_menu(menuId));
        if(s_menu == null){
            return ResponseValue.error("菜单不存在:" + menuId);
        }
        return ResponseValue.success(new SystemMenu(s_menu));
    }

    @RequestMapping("/edit")
    public ResponseValue saveEdit(@RequestBody S_menu s_menu){
        if(s_menu == null || StringUtils.isEmpty(s_menu.getParent_id()) || StringUtils.isEmpty(s_menu.getMenu_name())){
            return ResponseValue.error("参数错误");
        }
        S_menu exist = this.menuService.queryExistMenuInParent(s_menu.getParent_id(), s_menu.getMenu_name());
        if(exist != null && !exist.getMenu_id().equals(s_menu.getMenu_id())){
            return ResponseValue.error("父菜单下已有相同名称菜单存在:" + s_menu.getMenu_name());
        }
        if(s_menu.getIs_frame().intValue() == MenuUtils.MENU_FRAME_YES && !StringUtils.isHttpLink(s_menu.getPath())){
            return ResponseValue.error("地址必须以http(s)://开头");
        }
        if(s_menu.getMenu_type().equals(MenuUtils.MENU_TYPE_ITEM) && StringUtils.isEmpty(s_menu.getComponent())){
            return ResponseValue.error("菜单项必须填写：组件路径（component）");
        }
        this.menuService.save(s_menu);
        this.menuCacheProvider.updateCacheData(s_menu.getMenu_id(), s_menu);
        return ResponseValue.success();
    }

    @RequestMapping("/add")
    public ResponseValue saveAddMenu(@RequestBody S_menu s_menu){
        if(s_menu == null || StringUtils.isEmpty(s_menu.getParent_id()) || StringUtils.isEmpty(s_menu.getMenu_name())){
            return ResponseValue.error("参数错误");
        }
        S_menu exist = this.menuService.queryExistMenuInParent(s_menu.getParent_id(), s_menu.getMenu_name());
        if(exist != null){
            return ResponseValue.error("父菜单下已有相同名称菜单存在:" + s_menu.getMenu_name());
        }
        if(s_menu.getIs_frame().intValue() == MenuUtils.MENU_FRAME_YES && !StringUtils.isHttpLink(s_menu.getPath())){
            return ResponseValue.error("地址必须以http(s)://开头");
        }
        if(s_menu.getMenu_type().equals(MenuUtils.MENU_TYPE_ITEM) && StringUtils.isEmpty(s_menu.getComponent())){
            return ResponseValue.error("菜单项必须填写：组件路径（component）");
        }
        s_menu.setMenu_id(DateUtils.getDateTimeSecondForShow());
        this.menuService.insert(s_menu);
        S_menu saved = this.menuService.get(new S_menu(s_menu.getMenu_id()));
        this.menuCacheProvider.putCacheData(s_menu.getMenu_id(), saved);
        return ResponseValue.success();
    }

    /**
     * 角色编辑时，显示已经设置的功能树结构目录。
     * @param roleId
     * @return
     * @date 2022-12-19
     */
    @RequestMapping("/select/roleMenuTree/{roleId}")
    public ResponseValue selectRoleMenuTree(@PathVariable Long roleId){
        if(roleId == null || roleId.longValue() <= 0){
            throw new IllegalArgumentException("角色参数不存在");
        }
        Map<String, Object> data = new HashMap<>(4);
        List<String> roleMenuIds = this.roleService.queryRoleMenuIdList(roleId);
        int menuScope = this.getCurrentOrgMenuScope();
        List<SystemMenu> menuTreeList = this.menuCacheProvider.getMenuTreeAll(null, true, menuScope);
        data.put("checkedKeys", roleMenuIds);
        data.put("menus", menuTreeList);
        return ResponseValue.success(data);
    }

    /**
     * 返回整个系统菜单树结构数据，目前前端角色设置中使用选择关联功能权限。
     * @return
     * @date 2022-12-18
     */
    @RequestMapping("/select/tree")
    public ResponseValue selectMenuTree(){
        int menuScope = this.getCurrentOrgMenuScope();
        List<SystemMenu> menuTreeList = this.menuCacheProvider.getMenuTreeAll(null, true, menuScope);
        return ResponseValue.success(menuTreeList);
    }

    @GetMapping("/list")
    public ResponseValue listMenuWithoutTree(){
        List<SystemMenu> menuList = null;
        UserPrincipal<S_user_core> userPrincipal = this.getCurrentUserPrincipal();
        if(this.isSupervisor()){
            menuList = this.menuCacheProvider.getMenuList(null, MenuUtils.MENU_SCOPE_PLATFORM);
        } else {
            int menuScope = this.getCurrentOrgMenuScope();
            menuList = this.menuCacheProvider.getMenuList(userPrincipal.getRoleIdList(), menuScope);
        }
        return ResponseValue.success(menuList);
    }

}
