package com.iplatform.base.controller;

import com.iplatform.base.ArgumentsConstants;
import com.iplatform.base.SystemController;
import com.iplatform.base.service.CodeServiceImpl;
import com.iplatform.base.support.DictTreeGenerator;
import com.iplatform.model.po.S_dict_data;
import com.walker.file.FileInfo;
import com.walker.infrastructure.tree.TreeNode;
import com.walker.infrastructure.utils.StringUtils;
import com.walker.web.ResponseValue;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;

import java.util.ArrayList;
import java.util.List;

/**
 * 公共权限操作方法。<p></p>
 * 在该对象里面的方法，都无需配置菜单权限，只要登录用户都有权使用。
 * <pre>
 *     1）比如：获取字典树等
 * </pre>
 * @author 时克英
 * @date 2023-03-13
 */
@RestController
@RequestMapping("/permit")
public class PermitController extends SystemController {

//    @RequestMapping("/copyright/get/info")
//    public ResponseValue getCopyRight(){
//        CopyRightVo copyRightVo = new CopyRightVo();
//        copyRightVo.setCompanyName(this.getArgumentVariable(ArgumentsConstants.CONFIG_COPYRIGHT_COMPANY_INFO).getStringValue());
//        copyRightVo.setCompanyImage(this.getArgumentVariable(ArgumentsConstants.CONFIG_COPYRIGHT_COMPANY_IMAGE).getStringValue());
//        return ResponseValue.success(copyRightVo);
//    }

    /**
     * 获取移动端访问站点域名（地址），如：localhost:8080
     * @return
     * @date 2023-05-12
     */
    @RequestMapping("/config/front/domain")
    public ResponseValue getConfigFrontDomain(){
        String data = this.getArgumentVariable(ArgumentsConstants.CONFIG_KEY_SITE_URL).getStringValue();
        return ResponseValue.success("success", data);
    }

    /**
     * 根据文件id下载本地文件
     * @param id 文件id（系统存储的唯一编号）
     * @throws Exception
     * @date 2023-03-13
     */
    @RequestMapping("/file/{id}")
    public void downloadLocalFile(@PathVariable String id) throws Exception{
        FileInfo fileInfo = this.getFileInfo(id);
        if(fileInfo == null){
            throw new IllegalArgumentException("文件未找到，id=" + id);
        }
        logger.info("开始下载本地文件:{}", fileInfo);
        this.downloadSimpleFile(this.getLocalFileData(fileInfo), fileInfo.getFileName());
    }

    /**
     * 根据代码表名字，查询包含的代码项集合。
     * @param dictType
     * @return
     * @date 2023-03-22 该接口在字典管理中存在，这里需要放在公共权限里面一份，方便所有用户查询某个字典类型下的所有项目集合。
     * @date 2023-04-15 使用缓存，这里只能返回一级的情况，如果存在树结构则需要调用方法，参考 API：{@linkplain PermitController#selectDictTreeList(String)}
     */
    @RequestMapping("/dict/data/type/{dictType}")
    public ResponseValue<List<S_dict_data>> dictTypeList(@PathVariable String dictType){
        logger.debug("dictType = " + dictType);
        String id = this.getDictCacheProvider().getDictTypeId(dictType);
        List<S_dict_data> list = this.getDictCacheProvider().getRootChildrenOneLevelList(id);
//        List<S_dict_data> list = this.codeService.queryDictDataByType(dictType);
        if(StringUtils.isEmptyList(list)){
            list = new ArrayList<S_dict_data>(2);
        }
        return ResponseValue.success(list);
    }

    /**
     * 返回代码树结构（列表集合），由前端展示组装。
     * @param dictType 数据字典类型
     * @return
     * @auth 时克英
     * @date 2023-03-13
     */
    @RequestMapping("/dict/list_tree/{dictType}")
    public ResponseValue selectDictTreeList(@PathVariable String dictType){
        if(StringUtils.isEmpty(dictType)){
            return ResponseValue.error("请提供字典类型!");
        }
        List<TreeNode> treeNodeList = new ArrayList<>(8);
//        this.getDictCacheProvider().getCodeChildrenList()
        List<S_dict_data> dictDataList = this.codeService.queryDictTreeList(dictType);
        if(StringUtils.isEmptyList(dictDataList)){
            return ResponseValue.success(treeNodeList);
        }

        // s_dict_data表：第一个根节点的'parent_id'
        // TreeGenerator组件使用的默认父id=0，所以对于字典来说，父id不是0可能是其他数值，所以这里需要先设置。
        long defaultParentId = dictDataList.get(0).getParent_id();
        DictTreeGenerator generator = new DictTreeGenerator(null);
        generator.setDefaultParentId(defaultParentId);
        generator.setEntityList(dictDataList);
        return ResponseValue.success(generator.getTreeRootList());
    }

    @Autowired
    public PermitController(CodeServiceImpl codeService){
        this.codeService = codeService;
    }

    private CodeServiceImpl codeService;
}
