package com.iplatform.base.controller;

import com.iplatform.base.SystemController;
import com.iplatform.base.event.RoleSecurityChangeEvent;
import com.iplatform.base.pojo.role.RoleAuthParam;
import com.iplatform.base.pojo.role.RoleParam;
import com.iplatform.base.pojo.role.RoleUserParam;
import com.iplatform.base.service.RoleServiceImpl;
import com.iplatform.base.util.role.SystemRole;
import com.iplatform.core.BeanContextAware;
import com.iplatform.model.po.S_role;
import com.iplatform.model.po.S_user_core;
import com.walker.db.page.GenericPager;
import com.walker.infrastructure.utils.NumberGenerator;
import com.walker.infrastructure.utils.StringUtils;
import com.walker.web.ResponseValue;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;

@RestController
@RequestMapping("/system/role")
public class RoleController extends SystemController {

    private RoleServiceImpl roleService;

    @Autowired
    public RoleController(RoleServiceImpl roleService){
        this.roleService = roleService;
    }

    /**
     * 批量添加授权用户。
     * @param roleId 角色ID
     * @param userIds 用户ID集合
     * @return
     * @date 2022-12-21
     */
    @RequestMapping("/select/authUser/all")
    public ResponseValue batchAuthUserList(Long roleId, Long[] userIds){
        if(userIds == null || userIds.length == 0 || roleId == null){
            return ResponseValue.error("参数错误");
        }
        logger.debug(roleId + ", " + userIds);
        this.roleService.execInsertRoleUserList(roleId, userIds);
        return ResponseValue.success();
    }

    /**
     * 取消已授权用户，从角色里移除。
     * @param roleUserParam
     * @return
     * @date 2022-12-21
     */
    @RequestMapping("/select/authUser/cancel")
    public ResponseValue cancelAuthUser(@RequestBody RoleUserParam roleUserParam){
        if(roleUserParam == null || roleUserParam.getRoleId() == null || roleUserParam.getUserId() == null){
            return ResponseValue.error("参数错误");
        }
        this.roleService.execDeleteRoleUser(roleUserParam.getRoleId(), roleUserParam.getUserId());
        return ResponseValue.success();
    }

    @RequestMapping("/select/authUser/unallocatedList")
    public ResponseValue unallocatedList(RoleAuthParam roleAuthParam){
        if(roleAuthParam == null || roleAuthParam.getRoleId() == null){
            return ResponseValue.error();
        }
        // 2023-01-28 这个由于url地址设置不合理，没能匹配: /list 所以暂时不能删除
        this.preparePageSearch();

        S_role s_role = this.roleService.get(new S_role(roleAuthParam.getRoleId()));
        GenericPager<S_user_core> pager = this.roleService.queryUnAllocatedUserList(roleAuthParam.getRoleId(), s_role.getOrg_id(), roleAuthParam.getUserName());
//        return this.acquireTablePage(pager.getDatas(), pager.getTotalRows());
        return ResponseValue.success(pager);
    }

    /**
     * 展示给定角色，已关联用户列表。
     * @param roleAuthParam
     * @return
     * @date 2022-12-21
     */
    @RequestMapping("/select/authUser/allocatedList")
    public ResponseValue allocatedList(RoleAuthParam roleAuthParam){
        if(roleAuthParam == null || roleAuthParam.getRoleId() == null){
            return ResponseValue.error();
        }
        // 2023-01-28 这个由于url地址设置不合理，没能匹配: /list 所以暂时不能删除
        this.preparePageSearch();

        GenericPager<S_user_core> pager = this.roleService.queryAllocatedUserList(roleAuthParam.getRoleId(), roleAuthParam.getUserName());
//        return this.acquireTablePage(pager.getDatas(), pager.getTotalRows());
        return ResponseValue.success(pager);
    }

    @RequestMapping("/edit")
    public ResponseValue saveEdit(@RequestBody SystemRole systemRole){
        if(systemRole == null || StringUtils.isEmpty(systemRole.getRole_name()) || systemRole.getOrg_id() == null
            || systemRole.getRole_id() == null){
            return ResponseValue.error("角色参数错误");
        }
        long orgId = systemRole.getOrg_id();
        S_role existRole = this.roleService.queryRoleByName(orgId, systemRole.getRole_name());
        if(existRole != null && existRole.getRole_id().longValue() != systemRole.getRole_id().longValue()){
            return ResponseValue.error("角色名称已经存在");
        }

        this.roleService.execUpdateRole(systemRole.$clone(), systemRole.getMenuIds());

        // 发送事件通知，让security重新加载权限数据
        BeanContextAware.publishEvent(new RoleSecurityChangeEvent(systemRole.getRole_id()));

        return ResponseValue.success();
    }

    @RequestMapping("/view/{roleId}")
    public ResponseValue view(@PathVariable Long roleId){
        S_role s_role = this.roleService.get(new S_role(roleId));
        if(s_role == null){
            return ResponseValue.error("角色不存在");
        }
        return ResponseValue.success(s_role);
    }

    @RequestMapping("/remove/{roleId}")
    public ResponseValue removeRole(@PathVariable Long roleId){
        if(roleId == null || roleId.longValue() <= 0){
            return ResponseValue.error("参数错误");
        }
        int roleUserCount = this.roleService.queryRoleUserSize(roleId);
        if(roleUserCount > 0){
            return ResponseValue.error("角色已分配用户，无法删除");
        }
        this.roleService.execDeleteRole(roleId);
        return ResponseValue.success();
    }

    @RequestMapping("/add")
    public ResponseValue saveAdd(@RequestBody SystemRole systemRole){
        if(systemRole == null || StringUtils.isEmpty(systemRole.getRole_name()) || systemRole.getOrg_id() == null){
            return ResponseValue.error("角色参数错误");
        }
        long orgId = systemRole.getOrg_id();
        if(orgId <= 0){
            return ResponseValue.error("角色所属机构必须提供");
        }
        S_role existRole = this.roleService.queryRoleByName(orgId, systemRole.getRole_name());
        if(existRole != null){
            return ResponseValue.error("角色名称已经存在");
        }
        systemRole.setRole_id(NumberGenerator.getSequenceNumber());
        this.roleService.execInsertRole(systemRole.$clone(), systemRole.getMenuIds());
        return ResponseValue.success();
    }

    @PostMapping("/select/changeStatus")
    public ResponseValue changeStatus(@RequestBody S_role s_role){
        if(s_role == null || s_role.getRole_id() == null){
            return ResponseValue.error("参数错误");
        }
        this.roleService.execUpdateStatus(s_role.getRole_id(), s_role.getStatus());
        return ResponseValue.success();
    }

    @GetMapping("/list")
    public ResponseValue pageList(RoleParam roleParam){
        if(roleParam == null || roleParam.getOrgId() <= 0){
            return ResponseValue.error("无法查询角色:没有条件");
        }
//        this.preparePageSearch();

        long orgId = 0;
        if(!this.isSupervisor()){
            // 普通用户，只能看到自己顶级机构用户
            orgId = this.getCurrentUser().getOrg_id();
        } else {
            orgId = roleParam.getOrgId();
        }
        GenericPager<S_role> pager = this.roleService.queryPageRoleList(orgId, roleParam.getStatus(), roleParam.getRoleName());
//        return this.acquireTablePage(pager.getDatas(), pager.getTotalRows());
        return ResponseValue.success(pager);
    }

}
