package com.iplatform.base.controller;

import com.iplatform.base.*;
import com.iplatform.base.excel.UserDataImportor;
import com.iplatform.base.pojo.UserParam;
import com.iplatform.base.service.LoginServiceImpl;
import com.iplatform.base.service.RoleServiceImpl;
import com.iplatform.base.service.UserServiceImpl;
import com.iplatform.base.util.UserUtils;
import com.iplatform.base.util.role.SystemRole;
import com.iplatform.base.util.user.SystemUser;
import com.iplatform.model.po.S_role;
import com.iplatform.model.po.S_user_core;
import com.walker.db.page.GenericPager;
import com.walker.di.DataImportException;
import com.walker.file.FileInfo;
import com.walker.infrastructure.utils.DateUtils;
import com.walker.infrastructure.utils.NumberGenerator;
import com.walker.infrastructure.utils.StringUtils;
import com.walker.web.ResponseValue;
import com.walker.web.UserOnlineProvider;
import com.walker.web.log.BusinessType;
import com.walker.web.log.Log;
import com.walker.web.log.OperateUser;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.multipart.MultipartFile;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

//@Api(tags = "用户管理模块")
@RestController
@RequestMapping("/system/user")
public class UserController extends SystemController {

//    private DeptCacheProvider deptCacheProvider;
    private UserServiceImpl userService;
    private RoleServiceImpl roleService;

    // 2023-03-22
    private LoginServiceImpl loginService;
    private UserOnlineProvider userOnlineProvider;

    @Autowired
    public UserController(UserServiceImpl userService, RoleServiceImpl roleService
//            , UserCacheProvider userCacheProvider, DeptCacheProvider deptCacheProvider
            , LoginServiceImpl loginService, UserOnlineProvider userOnlineProvider){
//        this.setDeptCacheProvider(deptCacheProvider);
//        this.setUserCacheProvider(userCacheProvider);
        this.userService = userService;
        this.roleService = roleService;
        this.loginService = loginService;
        this.userOnlineProvider = userOnlineProvider;
    }

    @PostMapping("/import")
    public ResponseValue importExcel(MultipartFile file, boolean updateSupport){
        try {
            UserDataImportor dataImportor = new UserDataImportor(file.getInputStream());
            dataImportor.setId("user_import");
            this.getDataImportEngine().executeImport(dataImportor, this.getCurrentUserPrincipal().getUserName());
            logger.info("成功导入记录:{}", dataImportor.getSuccessSize());
            logger.info("错误结果文件:{}", dataImportor.getErrorFile());

//            FileInfo errorFileInfo = this.uploadFileToRemote(dataImportor.getErrorFile());
            FileInfo errorFileInfo = this.uploadFileToRemote(dataImportor.getErrorFile(), null, String.valueOf(this.getOwner()));
            if(errorFileInfo != null){
                logger.debug("用户导入存在'不符合数据': ，", errorFileInfo.toString());
                logger.debug("这里可以返回给前端，让前端点击链接下载!");
            }

            return ResponseValue.success();

        } catch (Exception e) {
            if(e instanceof DataImportException){
                return ResponseValue.error("导入出现异常:" + e.getMessage());
            }
            logger.error("io异常:" + e.getMessage(), e);
            return ResponseValue.error("上传文件异常:" + e.getMessage());
        }
    }

    /**
     * 下载导入模板，该方法可做成统一接口，适合动态生成的模板。<p></p>
     * 对于特定自定义的模板，业务自己提供下载地址。
     * @author 时克英
     * @date 2023-02-08
     */
    @PostMapping("/select/downloadTemplate")
    public void downloadTemplate(){
        // 要保存的模板临时路径，仅仅提供一个本地路径地址，供导入引擎自动生成并保存!
//        String templatePath = "d:/tmp/template_user.xlsx";
//        TemplateInfo templateInfo = new TemplateInfo();
//        templateInfo.setTemplatePath(templatePath);
//        templateInfo.setTableName("s_user_core");
//
//        File templateFile = this.getDataImportEngine().generateTemplate(templateInfo);
//
//        try {
//            this.downloadSimpleFile(FileCopyUtils.copyToByteArray(templateFile), "系统用户导入");
//        } catch (IOException e) {
//            logger.error("下载'导出用户'模板错误:" + e.getMessage(), e);
//            ServletUtils.renderString(getResponse(), "下载'导出用户'模板错误:" + e.getMessage());
//        }
        this.downloadLocalImportTemplate("s_user_core");
    }

    /**
     * 更新用户归属角色，用户管理 --> 设置角色
     * @param userId
     * @param roleIds
     * @return
     * @date 2022-12-15
     */
    @PostMapping("/select/saveAuthRole")
    public ResponseValue updateAuthRole(Long userId, Long[] roleIds){
        if(userId == null || userId.longValue() <= 0){
            return ResponseValue.error("参数错误");
        }

        this.userService.execUpdateAuthRole(userId, roleIds);
        logger.error("更新用户归属角色后，需要更新用户登录缓存中角色列表，记住修改代码!");
        // 处理缓存的登录信息中角色更新，2023-03-22
        String uuid = this.loginService.queryLoginUUID(userId);
        if(StringUtils.isNotEmpty(uuid)){
            List<String> roleIdList = new ArrayList<>(4);
            for(long roleId : roleIds){
                roleIdList.add(String.valueOf(roleId));
            }
            DefaultUserPrincipal userPrincipal = new DefaultUserPrincipal(this.getUser(userId));
            userPrincipal.setRoleIdList(roleIdList);
            // 缓存登录信息
            this.userOnlineProvider.cacheUserPrincipal(uuid, userPrincipal);

        } else {
            logger.debug("用户从未登录过，不更新登录uuid信息，userId={}", userId);
        }

        return ResponseValue.success();
    }

    /**
     * 前端，选定用户后，获取授权角色信息。
     * @param userId
     * @return
     * @date 2022-12-13
     */
    @GetMapping("/select/authRole/{userId}")
    public ResponseValue authRole(@PathVariable Long userId){
//        List<S_role> roleList = this.userService.queryUserRoleList(userId);
        List<SystemRole> userAuthRoleList = this.userService.queryAuthRoleList(this.getUserRootOrgId(userId), userId);
        Map<String, Object> data = new HashMap<>(4);
        // 本单位所有角色
        data.put("roles", userAuthRoleList);
        // 用户信息
        data.put("user", this.getUserCacheProvider().getUser(userId));
        return ResponseValue.success(data);
    }

//    @ApiOperation(value = "重置用户密码")
    @PostMapping("/resetPwd")
    public ResponseValue resetPassword(@RequestBody SystemUser systemUser){
        if(systemUser == null || systemUser.getId() == null){
            return ResponseValue.error("参数错误");
        }
        String initPassword = this.getArgumentVariable(ArgumentsConstants.KEY_SECURITY_PASSWORD_INIT).getStringValue();
        this.userService.execResetPassword(systemUser.getId(), initPassword);
        S_user_core user_core = this.userService.get(new S_user_core(systemUser.getId()));
        this.getUserCacheProvider().updateUser(user_core);
        return ResponseValue.success();
    }

    @GetMapping("/remove/{userId}")
    @Log(title = "用户管理", businessType = BusinessType.Delete, isSaveRequestData = true, isSaveResponseData = true)
    public ResponseValue deleteUser(@PathVariable Long userId){
        if(userId == null || userId.longValue() <= 0){
            return ResponseValue.error("参数错误");
        }
        this.userService.execDeleteUser(userId, this.getPlatformCallback(PlatformUserCallback.class));
        this.getUserCacheProvider().removeUser(userId);
        logger.info("删除一个用户，缓存已更新:" + userId);
        return ResponseValue.success();
    }

    @PostMapping("/edit")
    public ResponseValue editUser(@RequestBody SystemUser user_core){
        if(user_core == null){
            return ResponseValue.error("参数不存在");
        }
        if(StringUtils.isEmpty(user_core.getUser_name()) || StringUtils.isEmpty(user_core.getNick_name())){
            return ResponseValue.error("缺少: 登录ID或用户昵称");
        }
        if(user_core.getUser_name().equals(Constants.SUPERVISOR_NAME_DEFAULT)){
            return ResponseValue.error("无法使用该用户名");
        }
        S_user_core existUser = null;
        String phoneNumber = user_core.getPhonenumber();
        if(StringUtils.isNotEmpty(phoneNumber)){
            existUser = this.userService.queryUserByPhone(phoneNumber);
            if(existUser != null && existUser.getId().longValue() != user_core.getId().longValue()){
                return ResponseValue.error("该手机号已被现有用户使用");
            }
        }
        String email = user_core.getEmail();
        if(StringUtils.isNotEmpty(email)){
            existUser = this.userService.queryUserByEmail(email);
            if(existUser != null && existUser.getId().longValue() != user_core.getId().longValue()){
                return ResponseValue.error("该邮箱地址已被现有用户使用");
            }
        }
        this.userService.execUpdateUser(user_core.$clone(), user_core.getRoleIds(), this.getPlatformCallback(PlatformUserCallback.class));
        this.getUserCacheProvider().updateUser(user_core);
        logger.info("编辑用户成功，并更新缓存: " + user_core.getUser_name());
        return ResponseValue.success();
    }

    @PostMapping("/select/changeStatus")
    public ResponseValue changeStatus(@RequestBody SystemUser systemUser){
        if(systemUser == null || systemUser.getId() == null || systemUser.getStatus() == null){
            return ResponseValue.error("缺少参数");
        }
        this.userService.execUpdateUserStatus(systemUser.getId(), systemUser.getStatus());
        S_user_core user_core = this.userService.get(new S_user_core(systemUser.getId()));
        this.getUserCacheProvider().updateUser(user_core);
        return ResponseValue.success();
    }

    @PostMapping("/add")
    @Log(title = "用户管理", businessType = BusinessType.Insert, operatorType = OperateUser.Manage)
    public ResponseValue saveUser(@RequestBody SystemUser user_core){
        if(user_core == null){
            return ResponseValue.error("参数不存在");
        }
        if(StringUtils.isEmpty(user_core.getUser_name()) || StringUtils.isEmpty(user_core.getNick_name())){
            return ResponseValue.error("缺少: 登录ID或用户昵称");
        }
        if(user_core.getUser_name().equals(Constants.SUPERVISOR_NAME_DEFAULT)){
            return ResponseValue.error("无法使用该用户名");
        }
        S_user_core existUser = this.userService.queryUserByLoginId(user_core.getUser_name());
        if(existUser != null){
            return ResponseValue.error("登录名已存在");
        }

        // 检查并设置密码
        String encryptInitPass = null;
        if(StringUtils.isEmpty(user_core.getPassword())){
            // 用户没有录入特定密码，用平台配置的默认密码
            encryptInitPass = this.getArgumentVariable(ArgumentsConstants.KEY_SECURITY_PASSWORD_INIT).getStringValue();
        } else {
            // 用户输入了明文密码
            encryptInitPass = this.encryptPassword(user_core.getPassword());
        }
        user_core.setPassword(encryptInitPass);

        String phoneNumber = user_core.getPhonenumber();
        if(StringUtils.isNotEmpty(phoneNumber)){
            if(this.userService.queryUserByPhone(phoneNumber) != null){
                return ResponseValue.error("该手机号已被现有用户使用");
            }
        }
        String email = user_core.getEmail();
        if(StringUtils.isNotEmpty(email)){
            if(this.userService.queryUserByEmail(email) != null){
                return ResponseValue.error("该邮箱地址已被现有用户使用");
            }
        }
        user_core.setCreate_time(DateUtils.getDateTimeNumber(System.currentTimeMillis()));
        user_core.setCreate_by(this.getCurrentUser().getUser_name());
        user_core.setOrg_id(this.getRootOrgIdByDept(user_core.getDept_id()));
        user_core.setId(NumberGenerator.getLongSequenceNumber());
        // 注意：这里保存必须用生成的 S_user_core 对象，所以通过 SystemUser克隆一个即可。
        // 否则报错, 2022-12-13
        this.userService.execSaveUser(user_core.$clone(), user_core.getRoleIds(), this.getPlatformCallback(PlatformUserCallback.class));
        // 更新缓存
        this.getUserCacheProvider().putUser(user_core);
        logger.info("添加一个用户成功，缓存已加入: " + user_core.getUser_name());
        return ResponseValue.success();
    }

    /**
     * 创建新用户时，选择角色列表。
     * @param deptId
     * @return
     * @date 2022-12-12
     */
    @GetMapping("/select/role/{deptId}")
    public ResponseValue getNewUserRoles(@PathVariable Long deptId){
        long orgId = 0;
        if(deptId == null || deptId.longValue() == 0){
            //普通用户
            orgId = this.getCurrentUser().getOrg_id();
        } else {
            // 超级管理员选了特定机构
            orgId = this.getRootOrgIdByDept(deptId);
        }
        List<S_role> roleList = this.roleService.queryRoleList(orgId);
        Map<String, Object> data = new HashMap<>(4);
        data.put("roles", roleList);
//        data.put("init_password", this.getArgumentVariable(ArgumentsConstants.KEY_SECURITY_PASSWORD_INIT).getStringValue());
        return ResponseValue.success(data);
    }

    /**
     * 编辑用户时，获得用户信息。
     * @param userId
     * @return
     * @date 2022-12-13
     */
//    @GetMapping(value = { "/view", "/view/{userId}" })
    @GetMapping("/view/{userId}")
    public ResponseValue getUserInfo(@PathVariable Long userId){
        if(userId == null || userId <= 0){
            return ResponseValue.error("参数错误");
        }
        long orgId = this.getUserCacheProvider().getUser(userId).getOrg_id();
        List<S_role> roleList = this.roleService.queryRoleList(orgId);
        List<String> roleIds = this.userService.queryUserRoleIdList(userId);
        Map<String, Object> data = new HashMap<>(4);
        // 本单位所有角色
        data.put("roles", roleList);
        // 用户所属角色
        data.put("roleIds", UserUtils.toRoleIdLongList(roleIds));
        // 用户信息
        data.put("data", this.getUserCacheProvider().getUser(userId));
        return ResponseValue.success(data);
    }

    @GetMapping("/list")
    public ResponseValue list(UserParam userParam){
        if(userParam == null){
            return ResponseValue.error("无法查询用户:没有条件");
        }
//        this.preparePageSearch();

        long orgId = 0;
        long deptId = userParam.getDeptId();

        if(!this.isSupervisor()){
            // 普通用户，只能看到自己顶级机构用户
            orgId = this.getCurrentUser().getOrg_id();
        } else {
            // 超级管理员
            if(deptId <= 0){
                // 没有选择机构树，无法查询用户
                return ResponseValue.error("请选择一个机构查询");
            }
            orgId = this.getRootOrgIdByDept(deptId);
        }
        if(orgId == deptId){
            // 点击的顶级机构
            deptId = 0;
        }

        GenericPager<S_user_core> pager = this.userService.queryPageUserList(orgId
                , deptId, userParam.getUserName(), userParam.getPhonenumber(), userParam.getStatus());
        // 补充所在'部门名字'属性，2023-03-23
        List<S_user_core> data = pager.getDatas();
        if(!StringUtils.isEmptyList(data)){
            data.stream().forEach(p -> {
                p.setParameterString("dept_name", this.getDeptName(p.getDept_id()));
            });
        }

//        return this.acquireTablePage(pager.getDatas(), pager.getTotalRows());
        return ResponseValue.success(pager);
    }

//    private long getRootOrgIdByDept(long deptId){
//        S_dept dept = this.deptCacheProvider.getDept(deptId);
//        if(dept == null){
//            throw new IllegalStateException("缓存中未找到机构: " + dept);
//        }
//        return dept.getOrg_id();
//    }
}
