package com.iplatform.base.controller;

import com.iplatform.base.Constants;
import com.iplatform.base.SystemController;
import com.iplatform.base.callback.UserProfileCallback;
import com.iplatform.base.config.SecurityUserProperties;
import com.iplatform.base.pojo.UserInfoRequest;
import com.iplatform.base.service.UserServiceImpl;
import com.iplatform.base.util.PlatformRSAUtils;
import com.iplatform.core.BeanContextAware;
import com.iplatform.core.TokenAwareContext;
import com.iplatform.core.TokenEntity;
import com.iplatform.model.po.S_user_core;
import com.iplatform.model.po.S_user_login;
import com.walker.file.FileInfo;
import com.walker.infrastructure.utils.StringUtils;
import com.walker.web.ResponseValue;
import com.walker.web.UserOnlineProvider;
import com.walker.web.UserPrincipal;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.multipart.MultipartFile;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

//@Api(tags = "用户档案")
@RestController
@RequestMapping("/system/user/profile")
public class UserProfileController extends SystemController {

    private UserServiceImpl userService;
    private SecurityUserProperties securityUserProperties;

    @Autowired
    public UserProfileController(UserServiceImpl userService, SecurityUserProperties securityUserProperties){
        this.userService = userService;
        this.securityUserProperties = securityUserProperties;
    }

    /**
     * 返回登录用户基本信息。
     * @return
     * @date 2023-02-15
     */
    @GetMapping
    public ResponseValue index(){
        List<String> roleIdList = this.getCurrentUserPrincipal().getRoleIdList();
        Map<String, Object> data = new HashMap<>(4);
        data.put("user", this.getCurrentUser());
        if(!StringUtils.isEmptyList(roleIdList)){
            data.put("roleGroup", StringUtils.collectionToCommaDelimitedString(roleIdList));
        } else {
            data.put("roleGroup", StringUtils.EMPTY_STRING);
        }
        return ResponseValue.success(data);
    }

//    @ApiOperation(value = "上传头像")
    @PostMapping("/avatar")
    public ResponseValue avatar(MultipartFile file){
        if(file == null){
            return ResponseValue.error("上传头像为空");
        }
        long userId = this.getCurrentUserId();
        try {
            FileInfo fileInfo = this.uploadFileToRemote(file.getInputStream(), "avatar.jpg", String.valueOf(userId), file.getSize(), null, String.valueOf(Constants.OWNER_PLATFORM));
            S_user_core s_user_core = new S_user_core(userId);
            s_user_core.setAvatar(fileInfo.getId());
            // 更新数据库用户记录
            this.userService.save(s_user_core);
            // 更新缓存
            S_user_core cacheUser = this.getUserCacheProvider().getUser(userId);
            cacheUser.setAvatar(fileInfo.getId());
            this.getUserCacheProvider().updateUser(cacheUser);
            return ResponseValue.success("success","imageUrl");

        } catch (Exception e) {
            logger.error("上传头像错误:" + e.getMessage(), e);
            return ResponseValue.error("头像处理异常!");
        }
    }

    /**
     * 后台用户修改资料：姓名和密码，后续扩展。
     * @param request
     * @return
     * @date 2023-08-02
     */
    @RequestMapping(value = "/updateInfo", method = RequestMethod.POST)
    public ResponseValue updateNameAndPassword(@RequestBody UserInfoRequest request){
        if(request == null){
            return ResponseValue.error("请填写更新资料");
        }
        if(StringUtils.isEmpty(request.getRealName()) && StringUtils.isEmpty(request.getPassword())){
            return ResponseValue.error("未填写任何资料");
        }

        S_user_core currentUser = this.getCurrentUser();

        if(StringUtils.isNotEmpty(request.getPassword())){
            String originPassword = PlatformRSAUtils.getRsaDecryptValue(request.getPassword(), PlatformRSAUtils.PRIK);
//            String error = PasswordUtils.filterText(originPassword);
//            if(error != null){
//                return ResponseValue.error(error);
//            }
//            int passLevelConfig = this.securityUserProperties.getPassLevel();
//            if(!PasswordUtils.validateComplex(originPassword, passLevelConfig)){
//                return ResponseValue.error("密码级别过低，请输入：大小写字母、数字以及至少一种特殊符号");
//            }
            String error = this.validatePasswordRule(originPassword);
            if(error != null){
                return ResponseValue.error(error);
            }

            // 设置加密后的密码，直接保存更新
            request.setPassword(this.encryptPassword(originPassword));
            currentUser.setPassword(request.getPassword());
            currentUser.setModify_pwd(1);
        }

        if(StringUtils.isNotEmpty(request.getRealName())){
            currentUser.setNick_name(request.getRealName());
        }

        UserProfileCallback callback = this.getPlatformCallback(UserProfileCallback.class);
        this.getUserService().execUpdateUserInfo(request, currentUser.getId(), callback);
        this.getUserCacheProvider().updateUser(currentUser);
        logger.debug(TokenAwareContext.getCurrentToken().toString());
        return ResponseValue.success();
    }

//    /**
//     * 验证密码是否符合平台政策。
////     * @param encryptPassword 前端修改的密码（密文），RSA加密，后台要解密的
//     * @param originPassword 原始明文密码
//     * @return 返回错误提示，返回空表示成功
//     * @date 2023-08-05
//     */
//    protected String validatePasswordRule(String originPassword){
////        String originPassword = PlatformRSAUtils.getRsaDecryptValue(encryptPassword, PlatformRSAUtils.PRIK);
//        String error = PasswordUtils.filterText(originPassword);
//        if(error != null){
//            return error;
//        }
//        int passLevelConfig = this.securityUserProperties.getPassLevel();
//        if(!PasswordUtils.validateComplex(originPassword, passLevelConfig)){
//            return "密码级别过低，请输入：大小写字母、数字以及至少一种特殊符号";
//        }
//        return null;
//    }

    /**
     * 强制当前用户修改密码
     * @param encryptPassword
     * @return
     * @date 2023-08-05
     */
    @RequestMapping(value = "/force_change_pass", method = RequestMethod.POST)
    public ResponseValue forceChangePassword(String encryptPassword){
        if(StringUtils.isEmpty(encryptPassword)){
            return ResponseValue.error("必须填写修改密码内容");
        }
        String originPassword = PlatformRSAUtils.getRsaDecryptValue(encryptPassword, PlatformRSAUtils.PRIK);
        String error = this.validatePasswordRule(originPassword);
        if(error != null){
            return ResponseValue.error(error);
        }

        long userId = this.getCurrentUserId();
        this.getUserService().execForceChangePassword(userId, this.encryptPassword(originPassword));
        // 更新缓存
        S_user_core userCore = this.getUserService().get(new S_user_core(userId));
        this.getUserCacheProvider().updateUser(userCore);

        // 更新登录用户缓存
        String uuid = null;
        UserPrincipal<S_user_core> userPrincipal = null;
        UserOnlineProvider userOnlineProvider = BeanContextAware.getBeanByType(UserOnlineProvider.class);
        TokenEntity tokenEntity = TokenAwareContext.getCurrentToken();
        if(tokenEntity != null){
            uuid = tokenEntity.getUuid();
            userPrincipal = (UserPrincipal<S_user_core>)userOnlineProvider.getUserPrincipal(uuid);
        } else {
            logger.debug("TokenAwareContext 未获取到token信息，需要从登录缓存中查询，userId={}", userId);
            S_user_login user_login = this.getLoginStrategyManager().getUserLogin(userCore.getUser_name());
            if(user_login == null){
                logger.warn("用户已（强制）修改密码，但未找到登录缓存(user_login)，需要重新登录，user = {}", userCore.getUser_name());
                throw new IllegalStateException("");
            }
            uuid = user_login.getUuid();
            userPrincipal = (UserPrincipal<S_user_core>)userOnlineProvider.getUserPrincipal(uuid);
        }
        userPrincipal.getUserInfo().setModify_pwd(1);
        userOnlineProvider.cacheUserPrincipal(uuid, userPrincipal);
        logger.debug("密码已修改，userId={}", userId);

        return ResponseValue.success();
    }
}
