package com.iplatform.base.excel;

import com.iplatform.base.di.JdbcExcelDataImportor;
import com.iplatform.base.util.DataImportUtils;
import com.walker.di.UpdateResult;
import com.walker.di.UpdateType;
import com.walker.infrastructure.utils.PhoneNumberUtils;
import com.walker.infrastructure.utils.StringUtils;
import org.springframework.jdbc.core.namedparam.MapSqlParameterSource;

import java.io.InputStream;
import java.util.Arrays;
import java.util.List;
import java.util.Map;

/**
 * 系统用户导入器实现。
 * @author 时克英
 * @date 2023-02-07
 */
public class UserDataImportor extends JdbcExcelDataImportor {

//    public UserDataImportor(File file){
//        this.setBatchEnabled();
//        this.setBatchSleepMills(200);
//        this.setUpdateType(UpdateType.Override);    // 设置更新策略为:覆盖记录
//        try {
//            this.setSource(new BufferedInputStream(new FileInputStream(file)));
//        } catch (FileNotFoundException e) {
//            throw new RuntimeException(e);
//        }
//    }
    public UserDataImportor(InputStream inputStream){
        this.setBatchEnabled();
        this.setBatchSleepMills(200);
        this.setUpdateType(UpdateType.Override);    // 设置更新策略为:覆盖记录
        this.setSource(inputStream);
    }

    @Override
    protected String acquireTableName() {
        return "s_user_core";
    }

    @Override
    protected boolean isCheckDataExist() {
        return true;
    }

    @Override
    protected UpdateResult checkDataExist(String tableName, List<Map<String, Object>> mapList) {
        // 1 - 根据导入原始数据查询是否在表中存在这些记录
        MapSqlParameterSource sqlParameterSource = new MapSqlParameterSource();
        List<String> paramUserName = DataImportUtils.acquireWhereInStringValues("user_name", mapList);
        sqlParameterSource.addValue("userName", paramUserName);
        List<Map<String, Object>> existList = this.getDataImportService().queryListObjectWhereIn(SQL_QUERY, sqlParameterSource);

        if(!StringUtils.isEmptyList(existList)){
            if(this.logger.isDebugEnabled()){
                for(Map<String, Object> existOne : existList){
                    logger.debug("存在一条数据 = " + existOne);
                }
            }
        }

        // 创建更新结果对象
        UpdateResult updateResult = new UpdateResult();

        // 2 - 设置准备更新的'条件字段集合'，可支持多个
        List<String> whereColumnNames = Arrays.asList("user_name");

        // 3 - 找到新写入和需要更新的数据集合
        Object[] result = DataImportUtils.calculateInsertAndUpdateList(mapList, existList, whereColumnNames);
        List<Map<String, Object>> insertList = (List<Map<String, Object>>)result[0];
        List<Map<String, Object>> updateList = (List<Map<String, Object>>)result[1];

        // 4 - 设置要写入集合
        updateResult.setInsertList(insertList);

        // 5 - 如果更新策略为覆盖，则设置要更新集合
        if(this.getUpdateType() == UpdateType.Override){
            // 如果更新数据策略为: 覆盖模式，需要批量更新数据，这里获得更新集合给平台。
            updateResult.setUpdateColumnNames(Arrays.asList("dept_id", "org_id", "nick_name", "user_type", "email", "phonenumber", "sex"));
            updateResult.setWhereColumnNames(whereColumnNames);
            updateResult.setUpdateList(updateList);
        }

        return updateResult;
    }

    @Override
    protected String validateData(Map<String, String> data) {
        if(data.get("user_name") == null){
            return "登录名不存在";
        }
        if(data.get("nick_name") == null){
            return "昵称不存在";
        }
        if(data.get("user_type") == null){
            return "用户类型不存在";
        }
        String phoneNumber = data.get("phonenumber");
        if(StringUtils.isNotEmpty(phoneNumber) && !PhoneNumberUtils.isCellPhoneNumber(phoneNumber)){
            return "手机号格式错误";
        }
        return null;
    }

    private static final String SQL_QUERY = "select * from s_user_core where user_name in (:userName)";
}
