package com.iplatform.tcp.config;

import com.iplatform.base.config.TcpProperties;
import com.iplatform.core.config.LoadBalanceProperties;
import com.iplatform.tcp.EngineType;
import com.iplatform.tcp.support.TestConnectionCallback;
import com.iplatform.tcp.support.WebSocketPush;
import com.iplatform.tcp.util.ws.LoginAction;
import com.walker.infrastructure.ApplicationRuntimeException;
import com.walker.push.PushManager;
import com.walker.push.Pushable;
import com.walker.queue.QueueManager;
import com.walker.tcp.ConnectionCallback;
import com.walker.tcp.ConnectionManager;
import com.walker.tcp.ProtocolResolver;
import com.walker.tcp.netty.WebSocketEngine;
import com.walker.tcp.netty.WebSocketNettyHandler;
import com.walker.tcp.netty.WebSocketServerInitializer;
import com.walker.tcp.websocket.WebsocketHandler;
import com.walker.tcp.websocket.WebsocketProtocolResolver;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.lang.Nullable;

import java.util.ArrayList;
import java.util.List;

@Configuration
@ConditionalOnProperty(prefix = "iplatform.tcp", name = "enabled", havingValue = "true", matchIfMissing = false)
public class WebsocketConfig {

    @Bean
    public WebsocketProtocolResolver websocketProtocolResolver(){
        WebsocketProtocolResolver websocketProtocolResolver =  new WebsocketProtocolResolver();
        websocketProtocolResolver.setPermitNotRegisterConnect(false);
        return websocketProtocolResolver;
    }

    private List<ProtocolResolver<?>> acquireProtocolResolverList(){
        List<ProtocolResolver<?>> protocolResolverList = new ArrayList<>(2);
        protocolResolverList.add(websocketProtocolResolver());
        return protocolResolverList;
    }

    /**
     * 配置业务定义的处理方法实现。
     * @param connectionManager
     * @param queueManager
     * @return
     * @date 2023-04-17
     */
    @Bean
    public WebsocketHandler wsServerHandler(ConnectionManager connectionManager
            , QueueManager queueManager, TcpProperties tcpProperties){
        WebsocketHandler websocketHandler = new WebsocketHandler();
        websocketHandler.setEmptyMsgDisconnect(false);
        websocketHandler.setEngineId(EngineType.INDEX_TCP_WEBSOCKET);
        websocketHandler.setConnectionManager(connectionManager);
        websocketHandler.setQueueManager(queueManager);
        websocketHandler.setProtocolResolverList(this.acquireProtocolResolverList());
//        websocketHandler.setScanPackages("com.iplatform.tcp.util.ws");
        websocketHandler.setScanPackages(tcpProperties.getScanPackagesWs());
        // 2023-09-27
        websocketHandler.setConnectionHost(tcpProperties.getConnectionHost());
        return websocketHandler;
    }

    @Bean
    public WebSocketNettyHandler wsNettyLongHandler(ConnectionManager connectionManager
            , WebsocketHandler wsServerHandler, TcpProperties tcpProperties){
        WebSocketNettyHandler webSocketNettyHandler = new WebSocketNettyHandler();
        webSocketNettyHandler.setConnectionManager(connectionManager);
        webSocketNettyHandler.setTcpServerHandler(wsServerHandler);
        webSocketNettyHandler.setUri(tcpProperties.getWebsocketUri());
        return webSocketNettyHandler;
    }

    @Bean
    public WebSocketServerInitializer wsServerInitializer(WebSocketNettyHandler wsNettyLongHandler, TcpProperties tcpProperties){
        WebSocketServerInitializer webSocketServerInitializer = new WebSocketServerInitializer();
        webSocketServerInitializer.setHandler(wsNettyLongHandler);
        webSocketServerInitializer.setProtocolResolverList(this.acquireProtocolResolverList());
        webSocketServerInitializer.setShowLog(tcpProperties.isShowLog());
        return webSocketServerInitializer;
    }

    @Bean
    public WebSocketEngine webSocketEngine(ConnectionManager connectionManager
            , WebSocketServerInitializer webSocketServerInitializer
            , TcpProperties tcpProperties, @Nullable ConnectionCallback connectionCallback
            , LoadBalanceProperties loadBalanceProperties){
        WebSocketEngine webSocketEngine = new WebSocketEngine();
        webSocketEngine.setId(EngineType.INDEX_TCP_WEBSOCKET);
        webSocketEngine.setName("websocket测试引擎");
        webSocketEngine.setConnectionManager(connectionManager);
        webSocketEngine.setServerInitializer(webSocketServerInitializer);
        webSocketEngine.setOpenHeartBeat(tcpProperties.isOpenHeartBeat());
        webSocketEngine.setBossThreadNum(tcpProperties.getBossThreadNum());
        webSocketEngine.setWorkerThreadNum(tcpProperties.getWorkerThreadNum());
//        webSocketEngine.setHeartBeatTimeInterval(180000);
        webSocketEngine.setHeartBeatTimeInterval(tcpProperties.getHeartBeatSeconds() * 1000);
        webSocketEngine.setPort(tcpProperties.getPortWs());
        // 2023-08-25
        if(connectionCallback != null){
            webSocketEngine.setConnectionCallback(connectionCallback);
        } else {
            webSocketEngine.setConnectionCallback(new TestConnectionCallback());
        }
        // 延时启动，测试用。正式注释掉。2023-09-27
        webSocketEngine.setEngineStartDelaySeconds(30);
        try {
            webSocketEngine.start();
        } catch (Exception e) {
            throw new ApplicationRuntimeException("启动'WebSocketEngine'失败：" + e.getMessage(), e);
        }
        return webSocketEngine;
    }

    /**
     * Websocket登录认证动作配置
     * @return
     * @date 2023-04-17
     */
    @Bean
    public LoginAction wsLoginAction(){
        LoginAction loginAction = new LoginAction();
        return loginAction;
    }

    /**
     * 配置'WebSocketPush'推送实现，同时注册到管理器中。
     * @param webSocketEngine
     * @param pushManager
     * @return
     * @date 2023-04-21
     */
    @Bean
    public Pushable websocketPush(WebSocketEngine webSocketEngine, PushManager pushManager){
        WebSocketPush webSocketPush = new WebSocketPush();
        webSocketPush.setWebSocketEngine(webSocketEngine);
        webSocketPush.startup();
        pushManager.register(webSocketPush);
        return webSocketPush;
    }

}
