package com.iplatform.base;

import com.iplatform.base.callback.PlatformCallbackPostProcessor;
import com.iplatform.core.BeanContextAware;
import com.walker.db.page.ListPageContext;
import com.walker.db.page.PageSearch;
import com.walker.infrastructure.ApplicationRuntimeException;
import com.walker.infrastructure.arguments.ArgumentsManager;
import com.walker.infrastructure.arguments.Variable;
import com.walker.infrastructure.utils.DateUtils;
import com.walker.infrastructure.utils.FileUtils;
import com.walker.infrastructure.utils.StringUtils;
import com.walker.web.Constants;
import com.walker.web.ResponseValue;
import com.walker.web.util.ServletUtils;
import jakarta.servlet.http.HttpServletRequest;
import jakarta.servlet.http.HttpServletResponse;
import org.apache.commons.io.IOUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.InitializingBean;

import java.io.*;
import java.net.URLDecoder;
import java.util.List;

public abstract class AbstractController implements InitializingBean {

    protected final transient Logger logger = LoggerFactory.getLogger(this.getClass());

    private static String contextPath;
    private static final Object lock = new Object();

    /**
     * 返回第三方对接管理器对象。
     * @return
     * @date 2023-07-03
     */
    protected ThirdPartyManager getThirdPartyManager(){
        return BeanContextAware.getBeanByType(ThirdPartyManager.class);
    }

    /**
     * 返回服务端调用地址前缀，如:<br>
     * http://localhost:8080/demo
     * @return
     * @date 2023-01-06
     */
    protected String getServerDomain(){
        HttpServletRequest request = this.getRequest();
        return ServletUtils.getServerDomain(request);
    }

    /**
     * 返回给定的回调实现对象。
     * @param clazz 定义的回调接口，如: PlatformUserCallback.class
     * @return
     * @param <T>
     * @date 2023-01-05
     * @date 2023-01-28 该方法只能获取单个类型Callback，对于存在多种实现的请使用方法:
     * {@linkplain PlatformCallbackPostProcessor#getCallbackMultipleBean(Class)}
     */
    protected <T> T getPlatformCallback(Class<T> clazz){
        return PlatformCallbackPostProcessor.getCallbackObject(clazz);
    }

    /**
     * 返回系统配置可变参数对象。
     * @param key 参数 key
     * @return
     * @date 2022-11-29
     */
    protected Variable getArgumentVariable(String key){
        Variable v =  this.getArgumentManager().getVariable(key);
        if(v == null){
            throw new IllegalArgumentException("可变配置参数不存在: " + key);
        }
        return v;
    }

    protected ArgumentsManager getArgumentManager(){
        return BeanContextAware.getBeanByType(ArgumentsManager.class);
    }

    //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    //~ 新添加的方法，2022-11-16
    //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

    /**
     * 下载简单的文件，通常文件不大。
     * @param data 文件字节码内容
     * @param fileName 文件名称，如: demo.zip
     * @throws IOException
     * @date 2022-11-28
     */
    protected void downloadSimpleFile(byte[] data, String fileName) throws IOException{
        HttpServletResponse response = this.getResponse();
        response.reset();
        response.addHeader("Access-Control-Allow-Origin", "*");
        response.addHeader("Access-Control-Expose-Headers", "Content-Disposition");
        response.setHeader("Content-Disposition", "attachment; filename=\"" + fileName + "\"");
        response.addHeader("Content-Length", "" + data.length);
        response.setContentType("application/octet-stream; charset=UTF-8");
        IOUtils.write(data, response.getOutputStream());
    }

    /**
     * 从前端获得分页查询条件，写入线程参数中。
     * @author 时克英
     * @date 2022-11-16
     * @date 2023-01-28 该方法废弃，由拦截器根据特征'/list'自动执行分页准备调用。
     */
    @Deprecated
    protected PageSearch preparePageSearch(){
        PageSearch pageSearch = new PageSearch();
        Integer pn = this.getIntParameter(PageSearch.PAGE_NUM);
        if(pn != null){
            pageSearch.setPageIndex(pn.intValue());
        }
        Integer pageSize = this.getIntParameter(PageSearch.PAGE_SIZE);
        if(pageSize != null){
            pageSearch.setPageSize(pageSize.intValue());
        }
        pageSearch.setOrderByColumn(this.getParameter(PageSearch.ORDER_BY_COLUMN));
        pageSearch.setOrderAsc(this.getParameter(PageSearch.IS_ASC));
//        pageSearch.setReasonable(this.getParameter(PageSearch));
        ListPageContext.setPageSearch(pageSearch);
        return pageSearch;
    }

    /**
     * 把一个日期或时间字符串，转换成数值。
     * <pre>
     *     1.2022-11-16 --> 20221116000000
     *     2.2022-11-16 12:01:30 --> 20221116120130
     * </pre>
     * @param dateTime 给定时间或日期字符串
     * @param isTime 是否时间，否表示只有日期
     * @return
     */
    protected long getParamsDateTime(String dateTime, boolean isTime){
        if(StringUtils.isEmpty(dateTime)){
            return -1;
        }
        if(isTime){
            return DateUtils.getDateTimeNumber(dateTime);
        } else {
            return DateUtils.toLongDateTime(dateTime);
        }
    }

    /**
     * 包装返回带分页的表格集合，加上total属性(适配前端)
     * @param data
     * @param total
     * @return
     * @param <T>
     * @date 2022-11-19
     * @date 2023-05-16 从2.3.0之后废弃，因为使用了新的完整前端界面，返回的业务数据是一个整体，不再随意增加：ResponseValue属性。
     */
    @Deprecated
    protected <T> ResponseValue<List<?>> acquireTablePage(List<T> data, long total){
        ResponseValue<List<?>> rv = ResponseValue.success(data);
        rv.setTotal(total);
        return rv;
    }

    //~~~~~~~~~~~~~~~~~~~~~~~~~~ end ~~~~~~~~~~~~~~~~~~~~~~

    protected HttpServletRequest getRequest(){
//        ServletRequestAttributes servletRequestAttributes = (ServletRequestAttributes) RequestContextHolder.getRequestAttributes();
//        HttpServletRequest request = servletRequestAttributes.getRequest();
//        return request;
        return ServletUtils.getRequest();
    }

    protected HttpServletResponse getResponse(){
//        return ((ServletRequestAttributes)RequestContextHolder.getRequestAttributes()).getResponse();
        return ServletUtils.getResponse();
    }

    /**
     * 得到WEB应用上下文路径，如：/web
     * @return
     */
    protected String getContextPath(){
        if(contextPath == null){
            synchronized (lock) {
                HttpServletRequest request = getRequest();
                if(request == null)
                    throw new Error("request not found: getRequest().");
                contextPath = request.getContextPath();
            }
        }
        return contextPath;
    }

    protected String getParameter(String name){
        return getRequest().getParameter(name);
    }

    protected String getParameterUTF8(String name) throws UnsupportedEncodingException {
        String value = this.getParameter(name);
        return value == null ? null : URLDecoder.decode(value, "UTF-8");
    }

    protected String[] getParameterValues(String name) {
        return this.getRequest().getParameterValues(name);
    }

    protected Integer getIntParameter(String name) {
        String value = this.getParameter(name);
        return value == null ? null : Integer.valueOf(value);
    }

    protected Long getLongParameter(String name) {
        String value = this.getParameter(name);
        return value == null ? null : Long.valueOf(value);
    }

    protected Float getFloatParameter(String name) {
        String value = this.getParameter(name);
        return value == null ? null : Float.valueOf(value);
    }

    protected Double getDoubleParameter(String name) {
        String value = this.getParameter(name);
        return value == null ? null : Double.valueOf(value);
    }

    /**
     * @param name
     * @return Attribute value
     */
    protected Object getAttribute(String name) {
        return this.getRequest().getAttribute(name);
    }

    /**
     * @param name
     * @param value
     */
    protected void setAttribute(String name, Object value) {
        this.getRequest().setAttribute(name, value);
    }

    protected void setDefaultContentType() {
        this.getResponse().setContentType("text/html; charset=utf-8");
    }

    private void ajaxOutPut(ResponseFormat format, Object outputString) throws IOException {
        HttpServletResponse response = this.getResponse();
        response.setCharacterEncoding(StringUtils.DEFAULT_CHARSET_UTF8);
        response.setContentType(format.getValue());
        this.print(response, outputString == null ? StringUtils.EMPTY_STRING : outputString.toString());
    }

    protected void print(HttpServletResponse response, String str) throws IOException {
//		ServletOutputStream outputStream = this.getResponse().getOutputStream();
        PrintWriter pw = response.getWriter();
        pw.print(str);
    }

    protected void ajaxOutPutText(Object outputString) throws IOException {
        this.ajaxOutPut(ResponseFormat.TextPlain, outputString);
    }

    protected void ajaxOutPutJson(Object outputString) throws IOException {
        this.ajaxOutPut(ResponseFormat.ApplicationJson, outputString);
    }

    protected void ajaxOutPutXml(Object outputString) throws IOException {
        this.ajaxOutPut(ResponseFormat.TextXml, outputString);
    }

    protected void ajaxOutPutHtml(Object outputString) throws IOException {
        setDefaultContentType();
        this.print(this.getResponse(), outputString == null ? StringUtils.EMPTY_STRING : outputString.toString());
    }

    protected void ajaxOutputFileStream(String contentType, String filePath){
        ajaxOutputFileStream(contentType, new File(filePath));
    }

    protected void ajaxOutputFileStream(String contentType, File file){
        HttpServletResponse response = this.getResponse();
        response.setCharacterEncoding(StringUtils.DEFAULT_CHARSET_UTF8);
        response.setContentType(contentType);
        byte[] content = FileUtils.getFileBytes(file);
        if(content == null) return;
        OutputStream out = null;
        try {
            out = response.getOutputStream();
            out.write(content);
            out.flush();
        } catch (IOException e) {
            throw new ApplicationRuntimeException("输出服务器文件流出现异常!", e);
        } finally {
            if(out != null){
                try {
                    out.close();
                } catch (IOException e) {}
            }
        }
    }

    public enum ResponseFormat{
        ApplicationJson{
            public String getValue(){
                return Constants.APPLICATION_JSON;
            }
        },
        ApplicationXml{
            public String getValue(){
                return Constants.APPLICATION_XML;
            }
        },
        TextXml{
            public String getValue(){
                return Constants.TEXT_XML;
            }
        },
        TextPlain{
            public String getValue(){
                return Constants.TEXT_PLAIN;
            }
        };
//        ImagePng{
//            public String getValue(){
//                return Constants.IMAGE_PNG;
//            }
//        },
//        ImageJpeg{
//            public String getValue(){
//                return Constants.IMAGE_JPEG;
//            }
//        },
//        ImageGif{
//            public String getValue(){
//                return Constants.IMAGE_GIF;
//            }
//        };
        public String getValue(){
            throw new AbstractMethodError();
        }
    }

    //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    //~ 为老的freemarker界面预留的方法，2022-11-08
    //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

    /**
     * 前端页面调用的JS方面名称
     */
    protected static final String DEFAULT_JS_NAME = "reload";

    /**
     * 前端使用的分页对象引用名称，默认值
     */
    protected static final String DEFAULT_PAGER_VIEW_NAME = "pagerView";

}
