package com.iplatform.base;

import com.iplatform.base.push.DefaultPushManager;
import com.iplatform.base.util.NotificationUtils;
import com.walker.infrastructure.utils.DateUtils;
import com.walker.infrastructure.utils.NumberGenerator;
import com.walker.infrastructure.utils.StringUtils;
import com.walker.push.Notification;
import com.walker.push.NotificationChannel;
import com.walker.push.util.PushUtils;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

public abstract class PushController extends SystemController{

    /**
     * 发送短信验证码，登录与否都可调用。
     * @param code 发送的验证码
     * @param userIdOrMobile 用户ID 或者 手机号
     * @date 2023-04-26
     */
    protected void pushSmsValidateCode(String code, String userIdOrMobile){
        String creator = this.getCurrentUserPrincipal() == null? "creator" : this.getCurrentUserPrincipal().getId();
        ((DefaultPushManager)this.getPushManager()).pushSmsValidateCode(code, userIdOrMobile, creator);
    }

    /**
     * 推送短信内容，登录与否都可调用。
     * @param templateCode 短信模板代码
     * @param param 参数
     * @param userIdOrMobile 用户ID 或者 手机号
     * @date 2023-04-26
     */
    protected void pushSmsNotification(String templateCode, Map<String, String> param, String userIdOrMobile){
        String creator = this.getCurrentUserPrincipal() == null? "creator" : this.getCurrentUserPrincipal().getId();
        Notification notification = NotificationUtils
                .acquireSmsNotification(templateCode, param, null, userIdOrMobile, creator);
        this.getPushManager().pushSms(notification);
    }

    /**
     * 推送邮件通知，该方法无论登录都可调用。
     * <p>不登录的话，userIdOrMail 必须是邮件地址</p>
     * @param title 标题
     * @param content 内容
     * @param userIdOrMail 用户ID（或邮件地址）
     * @date 2023-04-26
     */
    protected void pushMailNotification(String title, String content, String userIdOrMail){
        if(StringUtils.isEmpty(title)
                || StringUtils.isEmpty(content)
                || StringUtils.isEmpty(userIdOrMail)){
            throw new IllegalArgumentException("推送邮件失败：参数为空!");
        }
        String creator = this.getCurrentUserPrincipal() == null? "creator" : this.getCurrentUserPrincipal().getId();
        Notification notification = PushUtils.acquireEmailNotificationOne(title, content
                , ((DefaultPushManager)this.getPushManager()).getMailFrom(), userIdOrMail, creator);
        this.getPushManager().push(notification, null);
    }

    /**
     * 推送平台普通消息，不包括：邮件和短信。
     * <p>这些通知会保存到数据库中，一般为操作业务的各种通知，如：待办事项等。</p>
     * @param pushData 业务传递的数据
     * @date 2023-04-26
     */
    protected void pushMessageNotification(PushData pushData){
        if(pushData == null){
            throw new IllegalArgumentException("pushData is null!");
        }
        if(StringUtils.isEmpty(pushData.getBusinessType())
                || StringUtils.isEmpty(pushData.getBusinessId())
                || StringUtils.isEmpty(pushData.getUserId())){
            throw new IllegalArgumentException("推送业务消息缺少条件：businessType, businessId, userId");
        }

        String creator = this.getCurrentUserPrincipal() == null? "creator" : this.getCurrentUserPrincipal().getId();

        Notification notification = new Notification();
        notification.setCreator(creator);
        notification.setCreateTime(DateUtils.getDateTimeNumber());
        notification.setFrom(Constants.PUSH_FROM_DEFAULT);
        notification.setTitle(pushData.getTitle());
        notification.setOptionId(pushData.getBusinessId());
        notification.setOptionType(pushData.getBusinessType());
        notification.setContent(pushData.toJson());
        notification.setId(NumberGenerator.getLongSequenceId());

        List<String> receiverList = new ArrayList<>(2);
        receiverList.add(pushData.getUserId());
        notification.setReceiverList(receiverList);

        // 根据平台配置实用通道
        List<NotificationChannel> channelList = new ArrayList<>(4);
        for(String index : this.getPushManager().getMessageChannelNames()){
            channelList.add(NotificationChannel.getType(index));
        }
        notification.setChannelList(channelList);
        // 是否并行
        notification.setParallel(this.getPushManager().isMessageParallel());

        this.getPushManager().push(notification, null);
    }
}
