package com.iplatform.base;

import com.walker.infrastructure.utils.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

/**
 * 第三方管理器默认实现。
 * @author 时克英
 * @date 2023-07-03
 */
public class ThirdPartyManager {

    protected final transient Logger logger = LoggerFactory.getLogger(this.getClass());

    private final Map<String, ThirdPartyAuthentication> authenticationMap = new ConcurrentHashMap<>(8);

    /**
     * 注册一个第三方认证对象到平台。
     * <pre>
     *     1)其中，name属性通常是第三方请求登录路径中的后缀部分，如：/third_party/login/aaa
     *     2)平台支持多种第三方认证同时使用，只要在业务中注册即可。
     * </pre>
     * @param thirdPartyAuthentication
     * @date 2023-07-03
     */
    public void register(ThirdPartyAuthentication thirdPartyAuthentication){
        if(thirdPartyAuthentication == null){
            throw new IllegalArgumentException("注册第三方认证对象为空");
        }
        String key = thirdPartyAuthentication.getName();
        if(StringUtils.isEmpty(key)){
            throw new IllegalArgumentException("自定义第三方认证对象名称不存在：getName()");
        }
        if(this.authenticationMap.get(key) != null){
            throw new IllegalArgumentException("已经存在第三方认证定义:" + key);
        }
        this.authenticationMap.put(key, thirdPartyAuthentication);
        logger.info("注册'第三方认证对象' = " + key + ", " + thirdPartyAuthentication.getDescription());
    }

    /**
     * 返回一个第三方认证对象
     * @param key 认证对象的名称，参考：{@linkplain ThirdPartyAuthentication#getName()}
     * @return
     * @date 2023-07-03
     */
    public ThirdPartyAuthentication getThirdPartyAuthentication(String key){
        if(StringUtils.isEmpty(key)){
            throw new IllegalArgumentException("请提供第三方认证标识");
        }
        return this.authenticationMap.get(key);
    }
}
