package com.iplatform.base.cache;

import com.iplatform.base.Constants;
import com.iplatform.base.service.CodeServiceImpl;
import com.iplatform.model.po.S_dict_data;
import com.iplatform.model.po.S_dict_type;
import com.walker.cache.tree.AbstractCacheTreeProvider;
import com.walker.cache.tree.CacheTreeNode;
import com.walker.cache.tree.DefaultCacheTreeNode;
import com.walker.infrastructure.utils.StringUtils;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.TreeMap;

/**
 * 数据字典缓存定义，该缓存因为树结构复杂，并没有提供'Redis'方式实现。<br>
 * 不过一般字典数据变动很小，因此在集群环境中也是相对安全。<p></p>
 * 1) 注意：如果生产环境字典变动较大，而且是集群环境则必须实现'Redis'集中存储缓存对象。
 * @author 时克英
 * @date 2023-03-10
 */
public class DictCacheProvider extends AbstractCacheTreeProvider<S_dict_data> {

    private final SortComparator sort = new SortComparator();
    private final NodeComparator nodeSort = new NodeComparator();

    // 2023-04-15 添加字典类型与id映射记录，因为
    private final Map<String, String> dictTypeIdMap = new HashMap<>();

    /**
     * 根据字典类型，找出对应id，因为前端若依框架中，需要传类型，但系统使用id查询。
     * @param dictType
     * @return
     * @date 2023-04-15
     */
    public String getDictTypeId(String dictType){
        String id = this.dictTypeIdMap.get(dictType);
        if(id == null){
            throw new IllegalArgumentException("根据字典类型，未找到id：" + dictType);
        }
        return id;
    }

    /**
     * 返回代码集合对象，通常在数据库中可能会存储多个代码ID，可以通过此 方法获取代码集合。
     * @param codeIds 输入多个代码ID数组
     * @return
     */
    public List<CacheTreeNode> getCodeList(String[] codeIds){
        List<CacheTreeNode> result = new ArrayList<CacheTreeNode>(2*2);
        if(codeIds != null && codeIds.length > 0){
            for(String cid : codeIds){
                result.add(this.get(cid));
            }
        }
        return result;
    }

    /**
     * 返回给定代码表的子代码项，树结构
     * @param codeTableName 代码表ID
     * @return
     */
    public List<CacheTreeNode> getCodeChildrenList(String codeTableName){
        CacheTreeNode node = this.getOneRootNode(codeTableName);
        if(node == null){
            throw new IllegalArgumentException("not found node: " + codeTableName);
        }
        Collection<CacheTreeNode> list = node.getChildren();
        if(list != null){
            List<CacheTreeNode> result = new ArrayList<CacheTreeNode>();
            for(CacheTreeNode n : list){
                result.add(n);
            }
            Collections.sort(result, nodeSort);
            return result;
        }
        return null;
    }

    /**
     * 返回代码表的下一级子代码集合，返回的是代码对象
     * @param codeTableName 代码表ID
     * @return
     */
    public List<S_dict_data> getRootChildrenOneLevelList(String codeTableName){
        CacheTreeNode node = this.getOneRootNode(codeTableName);
//        CacheTreeNode node2 =this.getOneRootNode("101");
//        Object rootCacheNode = null;
//        for(Iterator<Cachable> it = this.getCache().getIterator(); it.hasNext();){
//            rootCacheNode = it.next().getValue();
//            logger.debug("........ {}" + rootCacheNode);
//        }
        if(node == null){
            throw new IllegalArgumentException("not found node: " + codeTableName);
        }
        Collection<CacheTreeNode> list = node.getChildren();
        if(list != null){
            List<S_dict_data> result = new ArrayList<>();
            S_dict_data temp = null;
            for(CacheTreeNode n : list){
                temp = (S_dict_data)n.getSource();
//                temp.setChildSum(n.getChildrenSize());
                result.add(temp);
            }
            Collections.sort(result, sort);
            return result;
        }
        return null;
    }

    /**
     * 根据某个代码ID，返回其下一级所有子代码集合
     * @param id
     * @return
     */
    public List<S_dict_data> getCodeChildrenOneLevelList(String id){
        CacheTreeNode parent = this.get(id);
        if(parent == null){
            return null;
        }
        Collection<CacheTreeNode> list = parent.getChildren();
        if(list != null){
            List<S_dict_data> result = new ArrayList<>();
            for(CacheTreeNode n : list){
                result.add((S_dict_data)n.getSource());
            }
            return result;
        }
        return null;
    }

    /**
     * 返回某个代码下面的所有代码树。
     * @param id 给定的代码id
     * @return
     */
    public List<CacheTreeNode> getCodeChildrenTreeList(String id){
        CacheTreeNode parent = this.get(id);
        if(parent == null){
            return null;
        }
        Collection<CacheTreeNode> list = parent.getChildren();
        if(list != null){
            List<CacheTreeNode> result = new ArrayList<CacheTreeNode>();
            for(CacheTreeNode n : list){
                result.add(n);
            }
            return result;
        }
        return null;
    }

//    /**
//     * 搜索代码树，支持多关键字搜索，返回集合数据展示给ztree，注意：返回结果只有一级不再分层
//     * @param codeTableId 代码表ID
//     * @param keyLike 查询关键字，如：'锅炉' 或者 '锅炉 电 9号'，多关键词用空格分隔
//     * @return
//     */
//    public String searchLikeTreeNodeList(String codeTableId, String keyLike){
////		Collection<CacheTreeNode> deptList = this.getRootList();
//        Collection<CacheTreeNode> deptList = this.getCodeChildrenList(codeTableId);
//        JSONArray array = new JSONArray();
//
//        if(deptList != null){
//            // 过滤某个单位下的根数据集合，后续补充
//            // ...
//            String[] keys = StringUtils.stringToArray(keyLike, " ");
//            boolean multiKeys = false;
//            if(keys != null && keys.length > 1){
//                multiKeys = true;
//            }
//
//            List<CacheTreeNode> tempList = null;
//            for(CacheTreeNode node : deptList){
//                // 不是给定单位的数据不要
////				if(((TypeTree)node.getSource()).getDept() != dept){
////					continue;
////				}
//                if(multiKeys){
//                    logger.debug("------------------");
//                    tempList = node.searchLike(keys);
//                } else {
//                    tempList = node.searchLike(keyLike);
//                }
//                if(!StringUtils.isEmptyList(tempList)){
//                    for(CacheTreeNode c : tempList){
//                        array.add(CodeUtils.createJsonTree(c.getKey(), c.getText()
//                                , Constants.DEFAULT_TREE_ROOT_ID, false, true, CodeUtils.DEFAULT_SEARCH_ICON, false));
//                    }
//                }
//            }
//        }
//        return array.toString();
//    }

    @Override
    protected Map<String, CacheTreeNode> loadRootList() {
        List<S_dict_type> rootList = this.codeService.queryRootCodeList();
        if(StringUtils.isEmptyList(rootList)){
            return null;
        }
        // 需要转成'S_dict_data'统一对象，老系统中代码表是一个表，这里是两个表。2023-03-10
        List<S_dict_data> dictList = new ArrayList<>();
        S_dict_data data = null;
        for(S_dict_type type : rootList){
            data = new S_dict_data();
            data.setDict_code(type.getDict_id());
            data.setDict_label(type.getDict_name());
            data.setParent_id(0L);  // 根节点不存在父id，默认；0
            dictList.add(data);
            //
            this.dictTypeIdMap.put(type.getDict_type(), String.valueOf(type.getDict_id()));
        }
        return obtainMap(dictList);
    }

    @Override
    protected Map<String, CacheTreeNode> loadChildList() {
        List<S_dict_data> list = this.codeService.queryAllCodeItemList();
        return obtainMap(list);
    }

    @Override
    protected CacheTreeNode toCacheTreeNode(S_dict_data entity) {
        return new DefaultCacheTreeNode(entity.getDict_code().toString()
                , entity.getDict_label(), entity, entity.getParent_id().toString());
    }

    @Override
    public String getProviderName() {
        return Constants.CACHE_NAME_DICT;
    }

    @Override
    public Class<?> getProviderType() {
        return S_dict_data.class;
    }

    private Map<String, CacheTreeNode> obtainMap(List<S_dict_data> list){
        if(list != null && list.size() > 0){
            Map<String, CacheTreeNode> map = new TreeMap<String, CacheTreeNode>();
            for(S_dict_data code : list){
                map.put(code.getDict_code().toString(), toCacheTreeNode(code));
            }
            return map;
        } else
            return null;
    }

    private class SortComparator implements Comparator<S_dict_data> {
        @Override
        public int compare(S_dict_data o1, S_dict_data o2) {
            return (int)(o1.getDict_sort() - o2.getDict_sort());
        }
    }
    private class NodeComparator implements Comparator<CacheTreeNode>{
        @Override
        public int compare(CacheTreeNode o1, CacheTreeNode o2) {
            S_dict_data n1 = (S_dict_data)o1.getSource();
            S_dict_data n2 = (S_dict_data)o2.getSource();
            return (int)(n1.getDict_sort() - n2.getDict_sort());
        }
    }

    public void setCodeService(CodeServiceImpl codeService) {
        this.codeService = codeService;
    }

    private CodeServiceImpl codeService;
}
