package com.iplatform.base.cache;

import com.iplatform.base.Constants;
import com.iplatform.base.DeptCacheProvider;
import com.iplatform.base.service.DeptServiceImpl;
import com.iplatform.base.util.DeptUtils;
import com.iplatform.model.po.S_dept;
import com.walker.cache.AbstractCacheProvider;
import com.walker.cache.Cache;
import com.walker.infrastructure.utils.StringUtils;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

/**
 * 基于本地内存实现的机构缓存提供者。
 * @author 时克英
 * @date 2022-12-05
 */
public class LocalDeptCacheProvider extends AbstractCacheProvider<S_dept> implements DeptCacheProvider {

    private DeptServiceImpl deptService;

    private Map<String, List<String>> deptChildrenIdList = new ConcurrentHashMap<>();

    public void setDeptService(DeptServiceImpl deptService) {
        this.deptService = deptService;
    }

    @Override
    public void setAllowCacheChildren(boolean allow) {

    }

    @Override
    public S_dept getDept(long deptId) {
        if(deptId == 0){
            return DeptUtils.SUPER_VISOR_DEPT;
        }
        return this.getCacheData(String.valueOf(deptId));
    }

    @Override
    public void updateDept(S_dept s_dept) {
        this.updateCacheData(String.valueOf(s_dept.getId()), s_dept);
    }

    @Override
    public void removeDept(long deptId) {
        String key = String.valueOf(deptId);
        String keyChildren = key + DeptUtils.KEY_CHILDREN_PREFIX;
        S_dept current = this.getDept(deptId);
        if(current == null){
            logger.warn("机构缓存已不存在，无需删除");
            return;
        }

        // 删除该机构子机构缓存列表
        this.removeCacheData(keyChildren);
        this.removeCacheData(key);

        // 更新父机构对应的childrenList
//        String parentId = String.valueOf(current.getParent_id());
        List<String> parentChildrenList = this.getChildrenDeptIdOneLevel(current.getParent_id());
        if(parentChildrenList != null){
            parentChildrenList.remove(key);
            // 由于在内存中，所以一旦删除，引用也就变了。2022-12-03
//            this.deptChildrenIdList.put(parentId + DeptUtils.KEY_CHILDREN_PREFIX, parentChildrenList);
        }
    }

    @Override
    public void putDept(S_dept s_dept) {
        String id = String.valueOf(s_dept.getId());
        try{
            this.putCacheData(id, s_dept);
            this.deptChildrenIdList.put(id + DeptUtils.KEY_CHILDREN_PREFIX, new ArrayList<>());

            if(s_dept.getParent_id().longValue() == 0L){
                // 顶级单位，没有父机构了
                logger.debug("顶级机构，不用更新父机构列表, " + id);
                return;
            }
            // 更新父机构列表
            List<String> parentChildrenList = this.getChildrenDeptIdOneLevel(s_dept.getParent_id());
            if(StringUtils.isEmptyList(parentChildrenList)){
                parentChildrenList = new ArrayList<>();
            }
            parentChildrenList.add(id);
            this.deptChildrenIdList.put(String.valueOf(s_dept.getParent_id()) + DeptUtils.KEY_CHILDREN_PREFIX, parentChildrenList);

        } catch (Exception e){
            throw new RuntimeException("添加 [机构] 缓存错误:" + e.getMessage(), e);
        }
    }

    @Override
    public List<String> getChildrenDeptIdOneLevel(long deptId) {
        String key = deptId + DeptUtils.KEY_CHILDREN_PREFIX;
        List<String> childrenIdList = this.deptChildrenIdList.get(key);
        if(childrenIdList == null){
            logger.error("未找到缓存中子机构列表: " + key);
            return null;
        }
        return childrenIdList;
    }

    @Override
    public List<S_dept> getChildrenDeptOneLevel(long deptId) {
        List<String> childrenIdList = this.getChildrenDeptIdOneLevel(deptId);
        if(StringUtils.isEmptyList(childrenIdList)){
            return null;
        }
        List<S_dept> resultList = new ArrayList<>();
        S_dept s_dept = null;
        for(String id : childrenIdList){
            s_dept = this.getDept(Long.parseLong(id));
            resultList.add(s_dept);
        }
        return resultList;
    }

    @Override
    protected int loadDataToCache(Cache cache) {
        List<S_dept> list = this.deptService.queryAllDeptListForCache();
        if(!StringUtils.isEmptyList(list)){
            for(S_dept h : list){
                this.putDept(h);
            }
            return list.size();
        }
        return 0;
    }

    @Override
    public String getProviderName() {
        return Constants.CACHE_NAME_DEPT;
    }

    @Override
    public Class<?> getProviderType() {
        return S_dept.class;
    }
}
