package com.iplatform.base.cache;

import com.iplatform.base.CategoryCacheProvider;
import com.iplatform.base.Constants;
import com.iplatform.base.service.CategoryServiceImpl;
import com.iplatform.base.util.CategoryUtils;
import com.iplatform.base.util.cache.CategorySortComparator;
import com.iplatform.model.po.S_category;
import com.iplatform.model.vo.CategoryTreeVo;
import com.walker.cache.Cache;
import com.walker.infrastructure.ApplicationRuntimeException;
import com.walker.infrastructure.utils.JsonUtils;
import com.walker.infrastructure.utils.StringUtils;
import com.walker.support.redis.cache.RedisCache;
import com.walker.support.redis.cache.RedisCacheProvider;

import java.io.UnsupportedEncodingException;
import java.net.URLDecoder;
import java.util.ArrayList;
import java.util.List;

/**
 * 平台分类数据，Redis缓存实现。
 * @author 时克英
 * @date 2023-05-17
 */
public class RedisCategoryCacheProvider extends RedisCacheProvider<S_category> implements CategoryCacheProvider {

    private final CategorySortComparator sortComparator = new CategorySortComparator();

    public RedisCategoryCacheProvider(){
        this.setUseRedis(true);
        this.setLoadPage(false);
    }

    @Override
    protected int loadDataToCache(Cache cache) {
        List<S_category> hosts = this.categoryService.selectAll(new S_category());
        if(!StringUtils.isEmptyList(hosts)){
            // ------------------------- 切换成普通缓存步骤：3
            if(this.isUseRedis()){
                // 如果redis中缓存数量和数据库中不一致（少），则清空redis缓存，重新加载数据库数据到缓存中。
                long totalCache = cache.getPersistentSize();
                if(totalCache != hosts.size()){
                    logger.info("redis缓存中Category数量小于实际用户，需要清空缓存重新加载! cache = " + totalCache + ", db = " + hosts.size());
                    cache.clear();

                    for(S_category h : hosts){
                        cache.put(String.valueOf(h.getId()), h);
                    }
                }
            }//------------------------------------------
            return hosts.size();
        }
        return 0;
    }

    @Override
    public List<CategoryTreeVo> getTree(Integer type, Integer status, String name, List<Integer> categoryIdList, int owner) {
        List<S_category> allTree = new ArrayList<>();
        S_category tempCategory = null;

        for(String value : ((RedisCache)this.getCache()).getIterator(null)){
            try{
                tempCategory = JsonUtils.jsonStringToObject(value, S_category.class);
            } catch (Exception ex){
                throw new ApplicationRuntimeException("redis存储'S_category'解析错误：" + value, ex);
            }
            if(CategoryUtils.isCondition(tempCategory, type, status, name, categoryIdList, owner)){
                allTree.add(tempCategory);
            }
        }
        return CategoryUtils.acquireListTree(allTree, sortComparator);
    }

    @Override
    public List<CategoryTreeVo> getListTree(Integer type, Integer status, String name, int owner) {
        if(StringUtils.isNotEmpty(name)){
            try {
                name = URLDecoder.decode(name, StringUtils.DEFAULT_CHARSET_UTF8);
            } catch (UnsupportedEncodingException e) {
                throw new RuntimeException(e);
            }
        }
        return getTree(type, status, name, null, owner);
    }

    @Override
    public S_category get(int id) {
        return this.getCacheData(String.valueOf(id));
    }

    @Override
    public void save(S_category category) {
        this.putCacheData(String.valueOf(category.getId()), category);
    }

    @Override
    public void update(S_category category) {
        this.updateCacheData(String.valueOf(category.getId()), category);
    }

    @Override
    public void remove(int id) {
        this.removeCacheData(String.valueOf(id));
    }

    @Override
    public String getProviderName() {
        return Constants.CACHE_NAME_CATEGORY;
    }

    @Override
    public Class<?> getProviderType() {
        return S_category.class;
    }

    public void setCategoryService(CategoryServiceImpl categoryService) {
        this.categoryService = categoryService;
    }

    private CategoryServiceImpl categoryService;
}
