package com.iplatform.base.callback;

import com.iplatform.base.PlatformLoginCallback;
import com.iplatform.base.PlatformUserCallback;
import com.iplatform.base.SystemController;
import com.walker.infrastructure.ApplicationCallback;
import com.walker.infrastructure.ApplicationRuntimeException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.config.BeanPostProcessor;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * 应用程序后处理器 - 回调接口的处理，启动扫描所有配置的实现类。</p>
 * 同时可以使用该对象的静态方法获取某个bean。
 * @author shikeying
 * @date 2014-11-25
 *
 */
public class PlatformCallbackPostProcessor implements BeanPostProcessor {

    protected final transient Logger logger = LoggerFactory.getLogger(getClass());

    /** 只有单个实现的才放这里面 */
    private static final Map<Class<?>, Object> callbackReference = new java.util.HashMap<Class<?>, Object>(4);

    // 2023-01-26 所有回调bean对象缓存，无论单个或多个实现类型。
    private static final Map<String, Object> multipleCallbackBeans = new HashMap<>(8);

    // 这里是系统支持的回调接口定义，后续开发会在这里添加新的。
    private List<Class<?>> callbackList = Arrays.asList(PlatformUserCallback.class
            , PlatformLoginCallback.class, UserProfileCallback.class, AfterLoginCallback.class
            , SecurityCallback.class);

    // 2023-03-10 集中初始化 <code>SystemController</code> 中的一些通用bean
    private List<SystemController> systemControllerList = new ArrayList<>(64);

    @Override
    public Object postProcessAfterInitialization(Object bean, String beanName) throws BeansException {
        if(ApplicationCallback.class.isAssignableFrom(bean.getClass())){
            String beanClazzName = bean.getClass().getName();
            Object existBean = null;
            for(Class<?> callbackClazz : callbackList){
                if(callbackClazz.isAssignableFrom(bean.getClass())){
                    if(((ApplicationCallback) bean).supportMultiple()){
                        multipleCallbackBeans.put(beanClazzName, bean);
                    } else {
                        existBean = callbackReference.get(callbackClazz);
                        if(existBean != null){
                            throw new ApplicationRuntimeException("...... 回调实现类已经加载，不能重复配置: " + existBean.getClass().getName());
                        }
                        callbackReference.put(callbackClazz, bean);
                    }
                    logger.info("加载了'CallbackBean'=" + beanClazzName);
                }
            }
        }
        else if(SystemController.class.isAssignableFrom(bean.getClass())){
            // 2023
//            logger.debug("+++++++++++++ 找到一个 SystemController = {}", bean.getClass().getName());
            this.systemControllerList.add((SystemController)bean);
        }
        return bean;
    }

    @Override
    public Object postProcessBeforeInitialization(Object bean, String beanName) throws BeansException {
        return bean;
    }

    /**
     * 返回给定的回调实现bean，如果不存在返回<code>null</code>
     * @param callbackClass 如: PlatformUserCallback.class
     * @return
     */
    public static final <T> T getCallbackObject(Class<T> callbackClass){
        if(callbackClass == null){
            return null;
        }
        Object bean = callbackReference.get(callbackClass);
        if(bean != null){
            return (T)bean;
        }
        return null;
    }

    /**
     * 根据给定bean实现类型，返回具体的回调对象。<p></p>
     * 例如: PlatformLoginCallback会存在多个实现。
     * @param callbackBean 给定的类型，如: WebLoginCallback.class
     * @return
     * @param <T>
     * @date 2023-01-26
     */
    public static final <T> T getCallbackMultipleBean(Class<T> callbackBean){
        if(callbackBean == null){
            return null;
        }
        Object bean = multipleCallbackBeans.get(callbackBean.getName());
        if(bean != null){
            return (T)bean;
        }
        return null;
    }

    /**
     * 返回系统收集到的 <code>SystemController</code> 对象集合。
     * @return
     * @date 2023-03-10
     */
    public List<SystemController> getSystemControllerList(){
        return this.systemControllerList;
    }
}
