package com.iplatform.base.config;

import com.iplatform.base.CategoryCacheProvider;
import com.iplatform.base.NotificationTemplateCache;
import com.iplatform.base.PushCacheProvider;
import com.iplatform.base.SystemGroupCache;
import com.iplatform.base.UserCacheProvider;
import com.iplatform.base.UserLoginCache;
import com.iplatform.base.WechatCacheProvider;
import com.iplatform.base.cache.RedisCaptchaCacheProvider;
import com.iplatform.base.cache.RedisCategoryCacheProvider;
import com.iplatform.base.cache.RedisDeptCacheProvider;
import com.iplatform.base.cache.RedisHostCacheProvider;
import com.iplatform.base.cache.RedisNotificationTemplateCache;
import com.iplatform.base.cache.RedisPushCacheProvider;
import com.iplatform.base.cache.RedisSystemGroupCache;
import com.iplatform.base.cache.RedisUserCacheProvider;
import com.iplatform.base.cache.RedisUserLoginCache;
import com.iplatform.base.cache.RedisUserOnlineProvider;
import com.iplatform.base.cache.RedisWechatCache;
import com.iplatform.base.service.CategoryServiceImpl;
import com.iplatform.base.service.ConfigArgumentServiceImpl;
import com.iplatform.base.service.DeptServiceImpl;
import com.iplatform.base.service.GroupServiceImpl;
import com.iplatform.base.service.NotificationServiceImpl;
import com.iplatform.base.service.UserServiceImpl;
import com.iplatform.base.support.RedisArgumentsManager;
import com.iplatform.model.po.S_host;
import com.walker.cache.CacheProvider;
import com.walker.jdbc.service.PubService;
import com.walker.support.redis.RedisHelper;
import com.walker.support.redis.cache.RedisCacheProvider;
import org.springframework.boot.autoconfigure.AutoConfigureAfter;
import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.boot.autoconfigure.data.redis.RedisAutoConfiguration;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.data.redis.connection.RedisConnectionFactory;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.data.redis.serializer.StringRedisSerializer;

@Configuration
@ConditionalOnProperty(prefix = "iplatform.cache", name = "redis-enabled", havingValue = "true", matchIfMissing = false)
@ConditionalOnClass({RedisCacheProvider.class})
@AutoConfigureAfter(RedisAutoConfiguration.class)
public class RedisCacheConfig extends CacheConfiguration {

    @Bean
    public RedisTemplate<String, Object> redisTemplate(RedisConnectionFactory redisConnectionFactory){
        RedisTemplate<String, Object> template = new RedisTemplate<String, Object>();
        template.setConnectionFactory(redisConnectionFactory);
        StringRedisSerializer stringRedisSerializer = new StringRedisSerializer();
        // key采用String的序列化方式
        template.setKeySerializer(stringRedisSerializer);
        // hash的key也采用String的序列化方式
        template.setHashKeySerializer(stringRedisSerializer);
        // value序列化方式采用jackson
//        template.setValueSerializer(jackson2JsonRedisSerializer);
        template.setValueSerializer(stringRedisSerializer);
        // hash的value序列化方式采用jackson
//        template.setHashValueSerializer(jackson2JsonRedisSerializer);
        template.setHashValueSerializer(stringRedisSerializer);
        template.afterPropertiesSet();
        return template;
    }

    @Bean
    public RedisHelper redisHelper(RedisTemplate<String, Object> redisTemplate){
        RedisHelper redisHelper = new RedisHelper(redisTemplate);
        return redisHelper;
    }

    /**
     * 如果存在配置项(打开redis缓存)，同时存在包(walker-support-redis)则使用redis缓存对象。
     * @param pubService
     * @return
     * @throws Exception
     * @date 2022-09-20
     */
    @Bean
    public CacheProvider<S_host> cacheProviderHostRedis(PubService pubService, RedisHelper redisHelper) throws Exception{
        RedisHostCacheProvider redisHostCacheProvider = new RedisHostCacheProvider();
        redisHostCacheProvider.setRedisHelper(redisHelper);
        redisHostCacheProvider.setPubService(pubService);
//        hostRedisCacheProvider.afterPropertiesSet();
        logger.info("初始化缓存: HostRedisCacheProvider");
        return redisHostCacheProvider;
    }

    /**
     * 加载Redis用户缓存提供者。
     * @param userService
     * @param redisHelper
     * @return
     * @date 2022-11-06
     * @date 2023-07-17 添加参数：cacheProperties
     */
    @Bean
    public UserCacheProvider userCacheProviderForRedis(UserServiceImpl userService
            , RedisHelper redisHelper, CacheProperties cacheProperties){
        RedisUserCacheProvider redisUserCacheProvider = new RedisUserCacheProvider();
        redisUserCacheProvider.setUserService(userService);
        redisUserCacheProvider.setRedisHelper(redisHelper);
        // 用户量太大，不用初始化加载缓存。2023-07-17
        redisUserCacheProvider.setAllowInitCache(!cacheProperties.isOrgUserBig());
        return redisUserCacheProvider;
    }

    /**
     * Redis实现的验证码缓存对象。<p></p>
     * 1.因为<code>CacheProvider</code>按照类型从<code>SimpleCacheManager</code>中获取，类型是不能重复的，<br>
     * 因此会导致多个String类型的缓存定义无法区别，实际上系统只能记录第一个。
     * 2.系统在定义时有两种办法:
     * <pre>
     *     1) Bean加上唯一名字，业务在引用时使用名称
     *     2）把缓存定义再封装一层接口，就像用户、机构缓存那样定义。
     * </pre>
     * @param redisHelper
     * @return
     * @date 2022-11-07
     */
    @Bean("captchaCacheProvider")
    public CacheProvider<String> cacheProviderCaptchaRedis(RedisHelper redisHelper){
        RedisCaptchaCacheProvider redisCaptchaCacheProvider = new RedisCaptchaCacheProvider();
        redisCaptchaCacheProvider.setRedisHelper(redisHelper);
        return redisCaptchaCacheProvider;
    }

    @Bean
    public RedisUserOnlineProvider redisUserOnlineProvider(RedisHelper redisHelper){
        RedisUserOnlineProvider userOnlineProvider = new RedisUserOnlineProvider();
        userOnlineProvider.setRedisHelper(redisHelper);
        return userOnlineProvider;
    }

    /**
     * 基于 Redis 的运行参数管理器实现。
     * @param configArgumentService
     * @param redisHelper
     * @return
     */
    @Bean
    public RedisArgumentsManager redisArgumentsManager(ConfigArgumentServiceImpl configArgumentService, RedisHelper redisHelper){
        RedisArgumentsManager redisArgumentsManager = new RedisArgumentsManager();
        redisArgumentsManager.setConfigArgumentService(configArgumentService);
        redisArgumentsManager.setRedisHelper(redisHelper);
        return redisArgumentsManager;
    }

    /**
     * 基于 Redis 的机构缓存实现。
     * @param deptService
     * @param redisHelper
     * @return
     * @date 2022-12-03
     */
    @Bean
    public RedisDeptCacheProvider redisDeptCacheProvider(DeptServiceImpl deptService
            , RedisHelper redisHelper, CacheProperties cacheProperties){
        RedisDeptCacheProvider redisDeptCacheProvider = new RedisDeptCacheProvider();
        redisDeptCacheProvider.setDeptService(deptService);
        redisDeptCacheProvider.setRedisHelper(redisHelper);
        // 用户量太大，不用缓存机构树。2023-07-17
        redisDeptCacheProvider.setAllowCacheChildren(!cacheProperties.isOrgUserBig());
        return redisDeptCacheProvider;
    }

    /**
     * 推送提醒临时数据缓存定义。
     * @return
     * @date 2023-04-27
     */
    @Bean
    public PushCacheProvider pushCacheProvider(RedisHelper redisHelper){
        RedisPushCacheProvider pushCacheProvider = new RedisPushCacheProvider();
        pushCacheProvider.setRedisHelper(redisHelper);
        return pushCacheProvider;
    }

    /**
     * 平台分类数据，Redis缓存实现。
     * @param categoryService
     * @param redisHelper
     * @return
     * @date 2023-05-17
     */
    @Bean
    public CategoryCacheProvider categoryCacheProvider(RedisHelper redisHelper, CategoryServiceImpl categoryService){
        RedisCategoryCacheProvider categoryCacheProvider = new RedisCategoryCacheProvider();
        categoryCacheProvider.setRedisHelper(redisHelper);
        categoryCacheProvider.setCategoryService(categoryService);
        return categoryCacheProvider;
    }

    @Bean
    public SystemGroupCache systemGroupCache(RedisHelper redisHelper, GroupServiceImpl groupService){
        RedisSystemGroupCache cache = new RedisSystemGroupCache();
        cache.setRedisHelper(redisHelper);
        cache.setGroupService(groupService);
        return cache;
    }

    /**
     * 用户登录记录缓存，用于判断登录策略。
     * @return
     * @date 2023-07-11
     */
    @Bean
    public UserLoginCache userLoginCache(RedisHelper redisHelper){
        RedisUserLoginCache cache = new RedisUserLoginCache();
        cache.setRedisHelper(redisHelper);
        return cache;
    }

    /**
     * 微信相关缓存配置。
     * @param redisHelper
     * @return
     * @date 2023-07-16
     * @date 2023-08-23 迁移到 base 模块
     */
    @Bean
    public WechatCacheProvider wechatCacheProvider(RedisHelper redisHelper){
        RedisWechatCache cache = new RedisWechatCache();
        cache.setRedisHelper(redisHelper);
        return cache;
    }

    /**
     * 通知提醒模板配置缓存。
     * @param notificationService
     * @return
     * @date 2023-08-25
     */
    @Bean
    public NotificationTemplateCache notificationTemplateCache(RedisHelper redisHelper, NotificationServiceImpl notificationService){
        RedisNotificationTemplateCache cache = new RedisNotificationTemplateCache();
        cache.setRedisHelper(redisHelper);
        cache.setNotificationService(notificationService);
        return cache;
    }
}
